'****************************************************************
'*  Name    : DT_INTS-18xv.bas                                  *
'*  Author  : based on work by Darrel Taylor and others         *
'*  Version : 4.30                                              *
'*  Date    : NOV 20,2022                                       *
'****************************************************************
'*  DT_INTS support for 18xv devices with VIC peripheral        *
'*    (including K42, K83, Q43, Q83/Q84, Q40/Q41)               *
'*                                                              *
'*  Versions:                                                   *
'*  4.00 : JUN 24, 2022                                         *
'*    - initial release with support for Q43                    *
'*  4.10 : JUN 26, 2022                                         *
'*    - add K42 (taken from DT_INTS-K42c)                       *
'*    - fix various K42 intr definitions                        *
'*    - add K83                                                 *
'*  4.20 : JUL 14, 2022                                         *
'*    - incorporate Q43 asm .inc file bit definitions           *
'*    - correct misc definitions for K42/K83                    *
'*    - add aliases for C1/CM1 and C2/CM2 so all devices match  *
'*  4.30 : NOV 20, 2022                                         *
'*    - fix INT0 PIE definitions for K42/K83                    *
'*                                                              *
'*  for use with ReEnterPBP-18xv.bas (high priority) and        *
'*    ReEnterPBP-18xvLP.bas (low priority) support files        *
'****************************************************************
  DISABLE DEBUG

' this module adds DT_INTS support for PIC18xv devices with a VIC peripheral
'
' xv18 notes
' ----------
' unlike previous PIC18F devices, the xv18 core devices (those with a VIC peripheral)
' contain hardware shadow registers that can be used to save and restore most uC
' registers, including:
'     WREG, STATUS, BSR, FSR0/1/2, and PRODH/L
' there are individual shadow register sets for both high and low priority interrupt contexts.
' the registers are automatically saved upon isr entry, and restored when using a 'RETFIE 1'
' the only important registers the hardware does not save are the TBLPTRU/H/L registers, 
' which are now saved and restored automatically as part of this module (this was previously 
' done as part of 'reenterPBP')
'
' PBP and Bank Select Register operations
' ---------------------------------------
' the PBP CHK?RP and RST?RP bank select .lib macros are used to set and track the setting
' of the Bank Select Register (BSR)
' the macros depend on the current setting of the asm compile-time PREV_BANK variable,
' and will not change the BSR if *it thinks* it doesn't have to, so it can be fooled
'
' whenever you manually change the bank select register you should inform PBP of the change
' by setting PREV_BANK to the same value, ie 'PREV_BANK = high (reg)' or 'PREV_BANK = x'
' that way, the check done in CHK?RP will see the change
' when writing asm code you will have to handle setting the BSR manually
' (see macros SET_BANK, RESET_BANK, and SET_BSR below)
'
' PBP GOSUB, CALL, and RETURN statements will add a RST?RP to the entry and exit of subs,
' so the code generated depends on the code seen immediately preceeding the routines...
' any statements that change PREV_BANK will influence the code generated by these calls
' the manual states GOSUB is used for PBP routines and CALL is used for ASM routines, 
' but the only real difference seems to be where the subs are located relative to the 
' call/gosub statement.
' note that asm call and return instructions do not add RST?RP
'
' SFR access bank
' ---------------
' this module assumes that INTCON0 and INTCON1 registers are located in the SFR access bank
' the PIEx and PIRx registers may or may not be in the SFR access bank (K42/K83), but
' it is assumed that they are both located in the same bank
' the IPRx registers are typically NOT in the SFR access bank
' SET_BANK must be used for registers outside the SFR access bank
' mpasmx can automatically determine the approriate setting for the ", a" portion of an
' asm instruction so this can be left off, but it assumes that the user will set the BSR if required
'
' CONFIG settings (**IMPORTANT**)
' -------------------------------
' ensure that the VIC module is set for legacy interrupt mode (MVECEN = OFF)
' NOTE: the default config setting for most devices is MVECEN = ON, which enables multi-vector mode!!
'   #config
'     CONFIG MVECEN = OFF   ; Interrupt contoller does not use vector table (legacy mode)
'   #endconfig
'
' adding support for new devices
' ------------------------------
' to add support for other devices there are two sections that require changes...
' these are marked with a '** device specific section **' header
' - add a '#define DT_INTS_18XV_FAMILY' type for the processor in the following section
' - extend the 'interrupt source definitions' section to include a list of specific 
'   interrupts for the added device
' no other changes should be required
' device-specific bit definitions can be found in the mpasmx device 'p18fxxxx.inc' file
'
;----------------------------------------------------------------------------
;---- ** device specific section **------------------------------------------
;---- define the device family based on the selected processor --------------
;----------------------------------------------------------------------------
  ' 18xv family types: K42, K83, Q43, Q83, Q84, Q40, Q41
  ' note: the Q83, Q84, Q40, and Q41 are not currently supported by PBP/mpasmx 5.87
  #if (__PROCESSOR__ = "18F25Q43") or (__PROCESSOR__ = "18F26Q43")  or (__PROCESSOR__ = "18F27Q43") 
    #define DT_INTS_18XV_FAMILY "Q43"
  #endif
  #if (__PROCESSOR__ = "18F45Q43") or (__PROCESSOR__ = "18F46Q43")  or (__PROCESSOR__ = "18F47Q43")
    #define DT_INTS_18XV_FAMILY "Q43"
  #endif
  #if (__PROCESSOR__ = "18F55Q43") or (__PROCESSOR__ = "18F56Q43")  or (__PROCESSOR__ = "18F57Q43")
    #define DT_INTS_18XV_FAMILY "Q43"
  #endif
  ' K42 family added v4.10
  #if (__PROCESSOR__ = "18F24K42") or (__PROCESSOR__ = "18F25K42") or (__PROCESSOR__ = "18F26K42")  or (__PROCESSOR__ = "18F27K42") 
    #define DT_INTS_18XV_FAMILY "K42"
  #endif
  #if (__PROCESSOR__ = "18F45K42") or (__PROCESSOR__ = "18F46K42")  or (__PROCESSOR__ = "18F47K42")
    #define DT_INTS_18XV_FAMILY "K42"
  #endif
  #if (__PROCESSOR__ = "18F55K42") or (__PROCESSOR__ = "18F56K42")  or (__PROCESSOR__ = "18F57K42")
    #define DT_INTS_18XV_FAMILY "K42"
  #endif
  ' K83 family added v4.10
  #if (__PROCESSOR__ = "18F25K83") or (__PROCESSOR__ = "18F26K83")
    #define DT_INTS_18XV_FAMILY "K83"
  #endif
  ' add other processor family types here... for example:
  '#if (__PROCESSOR__ = "18F16Q40")
  '  #define DT_INTS_18XV_FAMILY "Q40"
  '#endif

  ' check to see if we support the device
  #ifndef DT_INTS_18XV_FAMILY
    #error "unsupported device type"
  #endif

  ' define SFR access bank locations
  #if (DT_INTS_18XV_FAMILY = "K42") or (DT_INTS_18XV_FAMILY = "K83")
    define SFR_BANKA_START 0x3F60
    define SFR_BANKA_END   0x3FFF
  #else     ' all others (Q43, Q83, Q84, Q40, Q41)
    define SFR_BANKA_START 0x0460
    define SFR_BANKA_END   0x04FF
  #endif

;----------------------------------------------------------------------------
;---- end of device family section ------------------------------------------
;----------------------------------------------------------------------------

  #define DT_INTS_18XV 1

  DEFINE  INTHAND       INT_ENTRY_H  ' interrupt entry point for High Priority ints
  DEFINE  INTLHAND      INT_ENTRY_L  ' interrupt entry point for Low Priority ints
  
  ' these variables must be located in the access bank
  INT_Flags   VAR BYTE BANKA  SYSTEM
    Serviced_H      VAR INT_Flags.0
    Serviced_L      VAR INT_Flags.1
    InHPint         VAR INT_Flags.2
    NoPriority      VAR INT_Flags.3
    PBPVarsSaved_H  VAR INT_Flags.4     ' added v4, used by ReEnterPBP-18xv
    PBPVarsSaved_L  VAR INT_Flags.5     ' added v4, used by ReEnterPBP-18xvLP
  ' high priority TBLPTR context
  TBLPTRU_H   VAR BYTE BANKA  SYSTEM
  TBLPTRH_H   VAR BYTE BANKA  SYSTEM
  TBLPTRL_H   VAR BYTE BANKA  SYSTEM
  ' low priority TBLPTR context
  TBLPTRU_L   VAR BYTE BANKA  SYSTEM
  TBLPTRH_L   VAR BYTE BANKA  SYSTEM
  TBLPTRL_L   VAR BYTE BANKA  SYSTEM

  ' clear interrupt tracking variable
  INT_Flags = 0 
  goto  OVER_DT_INTS_18
  
; disable asm 'found label after column 1' warning
  ASM
  ERRORLEVEL -207
  ENDASM
 
;----[check for PBPL (PBP v2.50 and above only)]---
; make PBPL '__LONG__' visible to asm
  #if (__LONG__)  'PBPL
    asm
      __LONG__ set 1
    endasm
  #else           'PBPW
    asm
      __LONG__ set 0
   endasm
  #endif

;---[asm definitions and macros]---------------------------------
  ASM
  asm = 0    ; Assembly language Interrupts
  ASM = 0
  Asm = 0
  pbp = 1    ; Basic language interrupts
  PBP = 1
  Pbp = 1
  YES = 1
  yes = 1
  Yes = 1
  NO = 0
  no = 0
  No = 0
  H equ 'H'  ; High Priority
  h equ 'H'
  L equ 'L'  ; Low Priority
  l equ 'L'
  
  list
  ENDASM

  ASM
;--[bank set/reset macros]--
; note: setting the BSR and PREV_BANK should track each other

; set BSR to 'num' value and update PREV_BANK
  SET_BSR macro num 
    movlb  num
    PREV_BANK = num
  endm

; set the BSR using a register name
; if 'reg' is located in the SFR access bank then setting the BSR is skipped
; similar to PBP CHK?RP macro, but does not rely on the PREV_BANK setting
  SET_BANK macro reg
    if ((reg) < SFR_BANKA_START) || ((reg) > SFR_BANKA_END)
      SET_BSR high(reg)
    endif
  endm

; reset BSR to 0
; similar to PBP RST?RP macro, but does not rely on the PREV_BANK setting
  RESET_BANK macro
    SET_BSR 0
  endm

; --[movff/movffl asm instruction macro]--
; given two registers, determine if MOVFF or MOVFFL instruction required
  MOVE_FF macro reg1, reg2
    if (reg1 > 0x0FFF) || (reg2 > 0x0FFF)
      movffl reg1, reg2
    else
      movff reg1, reg2
    endif
  endm
  
;--[save/restore TBLPTR context macros]--
  SAVE_TABLEPTR macro priority
    if (priority == L) || (priority == 0)
      MOVE_FF TBLPTRU, TBLPTRU_L
      MOVE_FF TBLPTRH, TBLPTRH_L
      MOVE_FF TBLPTRL, TBLPTRL_L
    else
      MOVE_FF TBLPTRU, TBLPTRU_H
      MOVE_FF TBLPTRH, TBLPTRH_H
      MOVE_FF TBLPTRL, TBLPTRL_H
    endif
  endm

  RESTORE_TABLEPTR macro priority
    if (priority == L) || (priority == 0)
      MOVE_FF TBLPTRU_L, TBLPTRU
      MOVE_FF TBLPTRH_L, TBLPTRH
      MOVE_FF TBLPTRL_L, TBLPTRL
    else
      MOVE_FF TBLPTRU_H, TBLPTRU
      MOVE_FF TBLPTRH_H, TBLPTRH
      MOVE_FF TBLPTRL_H, TBLPTRL
    endif
  endm

;--[global interrupt enable/disable macros]--
  INTERRUPT_ENABLE macro priority
    if (priority == L) || (priority == 0)
        bsf  INTCON0, IPEN_INTCON0, 0   ; enable interrupt priorities
        bsf  INTCON0, GIEL, 0           ; enable Low Priority interrupts
    else
        bsf  INTCON0, GIE, 0            ; enable interrupts
    endif
  endm

  INTERRUPT_DISABLE macro priority
    if (priority == L) || (priority == 0)
        bcf  INTCON0, GIEL, 0           ; disable Low Priority interrupts
    else
        bcf  INTCON0, GIE, 0            ; disable interrupts (disables all)
    endif
  endm
  ENDASM

;----------------------------------------------------------------------------
  ASM
  INT_Source  macro  IFR, IFB, IER, IEB, IPR, IPB
    if (IflagReg == IFR) && (IflagBit == IFB)
      list
      INT_Flag_Reg = IFR
      INT_Flag_Bit = IFB
      INT_Enable_Reg = IER
      INT_Enable_Bit = IEB
      INT_Priority_Reg = IPR
      INT_Priority_Bit = IPB
      IntFound = 1
    endif
    nolist
  endm 
  ENDASM
  
;----------------------------------------------------------------------------
;---- ** device specific section **------------------------------------------
;---- interrupt source definitions ------------------------------------------
;----------------------------------------------------------------------------
;-- define the interrupt sources

;-------------------------
;---- 18FxxQ43 family ----
;-------------------------
#if (DT_INTS_18XV_FAMILY = "Q43")
  ASM
    nolist
    ; Q43 datasheet shows comparator flags as 'CMxxx' but .inc file uses 'Cxxx'
    ; create alias symbols to match datasheet (and other devices)
    #define CM1IF C1IF
    #define CM1IE C1IE
    #define CM1IP C1IP
    #define CM2IF C2IF
    #define CM2IE C2IE
    #define CM2IP C2IP
    #define C1_INT CM1_INT
    #define C2_INT CM2_INT
  
; -=-   PIR0 Interrupts: -=-
    #define IOC_INT   PIR0, IOCIF         ;-- IOC INT Port Change 
    ; --bit 6 unused--
    #define CLC1_INT  PIR0, CLC1IF        ;-- Configurable Logic Cell 1
    ; --bit 4 unused--
    #define CSW_INT   PIR0, CSWIF         ;-- Clock Switch Interrupt
    #define OSF_INT   PIR0, OSFIF         ;-- Oscillator Fail
    #define HLVD_INT  PIR0, HLVDIF        ;-- High/Low Voltage Detect 
    #define SW_INT    PIR0, SWIF          ;-- Software Interrupt
; -=-   PIR1 Interrupts: -=-
    #define SMT1PWA_INT PIR1, SMT1PWAIF   ;-- SMT1 Pulse-Width Acquisition
    #define SMT1PRA_INT PIR1, SMT1PRAIF   ;-- SMT1 Period Acquisition
    #define SMT1_INT  PIR1, SMT1IF        ;-- Signal Measurement Timer 1 Interrupt
    #define CM1_INT   PIR1, CM1IF          ;-- Comparator1
    #define ACT_INT   PIR1, ACTIF         ;-- Active Clock Tuning Interrupt
    #define AD_INT    PIR1, ADIF          ;-- A/D Converter 
    #define ZCD_INT   PIR1, ZCDIF         ;-- Zero Cross Detect
    #define INT0_INT  PIR1, INT0IF        ;-- INT0 External
; -=-   PIR2 Interrupts: -=-
    #define DMA1A_INT PIR2, DMA1AIF       ;-- DMA1 Abort
    #define DMA1OR_INT PIR2, DMA1ORIF     ;-- DMA1 Overrun
    #define DMA1DCNT_INT PIR2, DMA1DCNTIF  ;-- DMA1 Destination Count
    #define DMA1SCNT_INT PIR2, DMA1SCNTIF  ;-- DMA1 Source Count
    ; --bit 3 unused--
    ; --bit 2 unused--
    ; --bit 1 unused--
    #define ADT_INT   PIR2, ADTIF         ;-- ADC Threshold
; -=-   PIR3 Interrupts: -=-
    #define TMR0_INT  PIR3, TMR0IF        ;-- TMR0 Overflow
    #define CCP1_INT  PIR3, CCP1IF        ;-- CCP1
    #define TMR1G_INT PIR3, TMR1GIF       ;-- TMR1 Gate
    #define TMR1_INT  PIR3, TMR1IF        ;-- TMR1 Overflow  
    #define TMR2_INT  PIR3, TMR2IF        ;-- TMR2 to T2PR Match
    #define SPI1_INT  PIR3, SPI1IF        ;-- Serial Peripherial Interface 1
    #define SPI1TX_INT PIR3, SPI1TXIF     ;-- Serial Peripherial Interface 1 TX
    #define SPI1RX_INT PIR3, SPI1RXIF     ;-- Serial Peripherial Interface 1 RX
; -=-   PIR4 Interrupts: -=-
    #define PWM1_INT  PIR4, PWM1IF        ;-- PWM1 Parameter Interrupt
    #define PWM1P_INT PIR4, PWM1PIF       ;-- PWM1 Period Interrupt
    ; --bit 5 unused--
    ; --bit 4 unused--
    #define U1_INT    PIR4, U1IF          ;-- USART1
    #define U1E_INT   PIR4, U1EIF         ;-- USART1 Framing Error
    #define U1TX_INT  PIR4, U1TXIF        ;-- USART1 Transmit 
    #define U1RX_INT  PIR4, U1RXIF        ;-- USART1 Receive
; -=-   PIR5 Interrupts: -=-
    #define PWM2_INT  PIR5, PWM2IF        ;-- PWM2 Parameter Interrupt
    #define PWM2P_INT PIR5, PWM2PIF       ;-- PWM2 Period Interrupt
    #define TMR3G_INT PIR5, TMR3GIF       ;-- TMR3 Gate
    #define TMR3_INT  PIR5, TMR3IF        ;-- TMR3 Overflow 
    ; --bit 3 unused--
    #define SPI2_INT  PIR5, SPI2IF        ;-- Serial Peripherial Interface 2
    #define SPI2TX_INT PIR5, SPI2TXIF     ;-- Serial Peripherial Interface 2 TX
    #define SPI2RX_INT PIR5, SPI2RXIF     ;-- Serial Peripherial Interface 2 RX
; -=-   PIR6 Interrupts: -=-
    #define DMA2A_INT  PIR6, DMA2AIF      ;-- DMA2 Abort
    #define DMA2OR_INT PIR6, DMA2ORIF     ;-- DMA2 Overrun
    #define DMA2DCNT_INT PIR6, DMA2DCNTIF  ;-- DMA2 Destination Count
    #define DMA2SCNT_INT PIR6, DMA2SCNTIF  ;-- DMA2 Source Count
    #define NCO1_INT  PIR6, NCO1IF        ;-- NCO1 Interrupt
    #define CWG1_INT  PIR6, CWG1IF        ;-- Configurable Wave Generator 1
    #define CLC2_INT  PIR6, CLC2IF        ;-- Configurable Logic Cell 2
    #define INT1_INT  PIR6, INT1IF        ;-- INT1 External
; -=-   PIR7 Interrupts: -=-
    #define PWM3_INT  PIR7, PWM3IF        ;-- PWM3 Parameter Interrupt
    #define PWM3P_INT PIR7, PWM3PIF       ;-- PWM3 Period Interrupt
    #define CLC3_INT  PIR7, CLC3IF        ;-- Configurable Logic Cell 3
    ; --bit 4 unused--
    #define I2C1E_INT PIR7, I2C1EIF       ;-- I2C1 Error
    #define I2C1_INT  PIR7, I2C1IF        ;-- I2C1 Interrupt
    #define I2C1TX_INT PIR7, I2C1TXIF     ;-- I2C1 Transmit
    #define I2C1RX_INT PIR7, I2C1RXIF     ;-- I2C1 Receive
; -=-   PIR8 Interrupts: -=-
    #define SCAN_INT  PIR8, SCANIF        ;-- Memory Scanner Interrupt
    #define CCP2_INT  PIR8, CCP2IF        ;-- CCP2
    #define TMR5G_INT PIR8, TMR5GIF       ;-- TMR5 Gate
    #define TMR5_INT  PIR8, TMR5IF        ;-- TMR5 Overflow
    #define U2_INT    PIR8, U2IF          ;-- USART2
    #define U2E_INT   PIR8, U2EIF         ;-- USART2 Framing Error
    #define U2TX_INT  PIR8, U2TXIF        ;-- USART2 Transmit
    #define U2RX_INT  PIR8, U2RXIF        ;-- USART2 Receive
; -=-   PIR9 Interrupts: -=-
    ; --bit 7 unused--
    ; --bit 6 unused--
    #define CLC4_INT  PIR9, CLC4IF        ;-- Configurable Logic Cell 4
    ; --bit 4 unused--
    #define U3_INT    PIR9, U3IF          ;-- USART3
    #define U3E_INT   PIR9, U3EIF         ;-- USART3 Framing Error
    #define U3TX_INT  PIR9, U3TXIF        ;-- USART3 Transmit
    #define U3RX_INT  PIR9, U3RXIF        ;-- USART3 Receive
; -=-   PIR10 Interrupts: -=-
    #define DMA3A_INT  PIR10, DMA3AIF     ;-- DMA3 Abort
    #define DMA3OR_INT PIR10, DMA3ORIF    ;-- DMA3 Overrun
    #define DMA3DCNT_INT PIR10, DMA3DCNTIF  ;-- DMA3 Destination Count
    #define DMA3SCNT_INT PIR10, DMA3SCNTIF  ;-- DMA3 Source Count
    #define NCO2_INT  PIR10, NCO2IF       ;-- NCO2 Interrupt
    #define CWG2_INT  PIR10, CWG2IF       ;-- Configurable Wave Generator 2
    #define CLC5_INT  PIR10, CLC5IF       ;-- Configurable Logic Cell 5
    #define INT2_INT  PIR10, INT2IF       ;-- INT2 External 
; -=-   PIR11 Interrupts: -=-
    #define DMA4A_INT  PIR11, DMA4AIF     ;-- DMA4 Abort
    #define DMA4OR_INT PIR11, DMA4ORIF    ;-- DMA4 Overrun
    #define DMA4DCNT_INT PIR11, DMA4DCNTIF  ;-- DMA4 Destination Count
    #define DMA4SCNT_INT PIR11, DMA4SCNTIF  ;-- DMA4 Source Count
    #define TMR4_INT  PIR11, TMR4IF       ;-- TMR4 TMR4-T4PR Match
    #define CWG3_INT  PIR11, CWG3IF       ;-- Configurable Wave Generator 3
    #define CLC6_INT  PIR11, CLC6IF       ;-- Configurable Logic Cell 6
    #define CCP3_INT  PIR11, CCP3IF       ;-- CCP3 
; -=-   PIR12 Interrupts: -=-
    #define DMA5A_INT  PIR12, DMA5AIF     ;-- DMA5 Abort
    #define DMA5OR_INT PIR12, DMA5ORIF    ;-- DMA5 Overrun
    #define DMA5DCNT_INT PIR12, DMA5DCNTIF  ;-- DMA5 Destination Count
    #define DMA5SCNT_INT PIR12, DMA5SCNTIF  ;-- DMA5 Source Count
    #define U4_INT    PIR12, U4IF         ;-- USART4
    #define U4E_INT   PIR12, U4EIF        ;-- USART4 Framing Error
    #define U4TX_INT  PIR12, U4TXIF       ;-- USART4 Transmit
    #define U4RX_INT  PIR12, U4RXIF       ;-- USART4 Receive
; -=-   PIR13 Interrupts: -=-
    #define DMA6A_INT  PIR13, DMA6AIF     ;-- DMA6 Abort
    #define DMA6OR_INT PIR13, DMA6ORIF    ;-- DMA6 Overrun
    #define DMA6DCNT_INT PIR13, DMA6DCNTIF  ;-- DMA6 Destination Count
    #define DMA6SCNT_INT PIR13, DMA6SCNTIF  ;-- DMA6 Source Count
    #define U5_INT    PIR13, U5IF         ;-- USART5
    #define U5E_INT   PIR13, U5EIF        ;-- USART5 Framing Error
    #define U5TX_INT  PIR13, U5TXIF       ;-- USART5 Transmit
    #define U5RX_INT  PIR13, U5RXIF       ;-- USART5 Receive
; -=-   PIR14 Interrupts: -=-
    ; --bit 7 unused--
    ; --bit 6 unused--
    ; --bit 5 unused--
    ; --bit 4 unused--
    #define NCO3_INT  PIR14, NCO3IF       ;-- NCO3 Interrupt
    #define CM2_INT   PIR14, CM2IF         ;-- Comparator2 
    #define CLC7_INT  PIR14, CLC7IF       ;-- Configurable Logic Cell 7
    ; --bit 0 unused--
; -=-   PIR15 Interrupts: -=-
    ; --bit 7 unused--
    ; --bit 6 unused--
    ; --bit 5 unused--
    ; --bit 4 unused--
    #define TMR6_INT  PIR15, TMR6IF       ;-- TMR6 TMR6-T6PR Match
    #define CRC_INT   PIR15, CRCIF        ;-- Cyclic Redundancy Check
    #define CLC8_INT  PIR15, CLC8IF       ;-- Configurable Logic Cell 8
    #define NVM_INT   PIR15, NVMIF        ;-- EEPROM/FLASH Write Operation
    list
  ENDASM
#endif

;-- GetIntInfo (**device specific**)
#if (DT_INTS_18XV_FAMILY = "Q43")
  ASM
  GetIntInfo  macro  IflagReg, IflagBit
    nolist
    INT_Flag_Reg = -1
    INT_Flag_Bit = -1
    INT_Enable_Reg = -1
    INT_Enable_Bit = -1
    ifdef  IPR1
      INT_Priority_Reg = -1     ; cannot change priority (ie INT0_INT on some devices)
      INT_Priority_Bit = -1
    endif
    IntFound = 0
  
; -=- 18FxxQ43 family -=-
; -=-   PIR0 Interrupts: -=-
    ifdef IOCIF  ;----{ IOC External Interrupt }----------[PIR0, IOCIF]---
      INT_Source  PIR0,IOCIF, PIE0,IOCIE, IPR0,IOCIP
    endif
    ifdef CLC1IF  ;----{ CLC1 Configurable Logic Cell 1 }----------[PIR0, CLC1IF]---
      INT_Source  PIR0,CLC1IF, PIE0,CLC1IE, IPR0,CLC1IP
    endif
    ifdef CSWIF  ;----{ CSW Clock Switch }----------[PIR0, CSWIF]---
      INT_Source  PIR0,CSWIF, PIE0,CSWIE, IPR0,CSWIP
    endif
    ifdef OSFIF  ;----{ OSF Oscillator Fail }----------[PIR0, OSFIF]---
      INT_Source  PIR0,OSFIF, PIE0,OSFIE, IPR0,OSFIP
    endif
    ifdef HLVDIF  ;----{ HLVD High/Low Voltage Detect  }----------[PIR0, HLVDIF]---
      INT_Source  PIR0,HLVDIF, PIE0,HLVDIE, IPR0,HLVDIP
    endif
    ifdef SWIF  ;----{ SW Software Interrupt }----------[PIR0, SWIF]---
      INT_Source  PIR0,SWIF, PIE0,SWIE, IPR0,SWIP
    endif
; -=-   PIR1 Interrupts: -=-
    ifdef SMT1PWAIF  ;----{ SMT1PWA SMT1 Pulse-Width Acquisition }----------[PIR1, SMT1PWAIF]---
      INT_Source  PIR1,SMT1PWAIF, PIE1,SMT1PWAIE, IPR1,SMT1PWAIP
    endif
    ifdef SMT1PRAIF  ;----{ SMT1PRA SMT1 Period Acquisition }----------[PIR1, SMT1PRAIF]---
      INT_Source  PIR1,SMT1PRAIF, PIE1,SMT1PRAIE, IPR1,SMT1PRAIP
    endif
    ifdef SMT1IF  ;----{ SMT1 Signal Measurement Timer 1 }----------[PIR1, SMT1IF]---
      INT_Source  PIR1,SMT1IF, PIE1,SMT1IE, IPR1,SMT1IP
    endif
    ifdef CM1IF  ;----{ CM1 Comparator 1}----------[PIR1, CM1IF]---
      INT_Source  PIR1,CM1IF, PIE1,CM1IE, IPR1,CM1IF
    endif
    ifdef ACTIF  ;----{ ACT Active Clock Tuning}----------[PIR1, ACTIF]---
      INT_Source  PIR1,ACTIF, PIE1,ACTIE, IPR1,ACTIP
    endif
    ifdef ADIF  ;----{ AD A/D Converter }----------[PIR1, ADIF]---
      INT_Source  PIR1,ADIF, PIE1,ADIE, IPR1,ADIP
    endif
    ifdef ZCDIF  ;----{ ZCD Zero Cross Detect }----------[PIR1, ZCDIF]---
      INT_Source  PIR1,ZCDIF, PIE1,ZCDIE, IPR1,ZCDIP
    endif
    ifdef INT0IF  ;----{ INT0 INT0 External }----------[PIR1, INT0IF]---
      INT_Source  PIR1,INT0IF, PIE1,INT0IE, IPR1,INT0IP
    endif
; -=-   PIR2 Interrupts: -=-
    ifdef DMA1AIF  ;----{ DMA1A DMA1 Abort }----------[PIR2, DMA1AIF]---
      INT_Source  PIR2,DMA1AIF, PIE2,DMA1AIE, IPR2,DMA1AIP
    endif
    ifdef DMA1ORIF  ;----{ DMA1OR DMA1 Overrun }----------[PIR2, DMA1ORIF]---
      INT_Source  PIR2,DMA1ORIF, PIE2,DMA1ORIE, IPR2,DMA1ORIP
    endif
    ifdef DMA1DCNTIF  ;----{ DMA1DCNT DMA1 Destination Count }----------[PIR2, DMA1DCNTIF]---
      INT_Source  PIR2,DMA1DCNTIF, PIE2,DMA1DCNTIE, IPR2,DMA1DCNTIP
    endif
    ifdef DMA1SCNTIF  ;----{ DMA1SCNT DMA1 Source Count }----------[PIR2, DMA1SCNTIF]---
      INT_Source  PIR2,DMA1SCNTIF, PIE2,DMA1SCNTIE, IPR2,DMA1SCNTIP
    endif
    ifdef ADTIF  ;----{ ADT ADC Threshold }----------[PIR2, ADTIF]---
      INT_Source  PIR2,ADTIF, PIE2,ADTIE, IPR2,ADTIP
    endif
; -=-   PIR3 Interrupts: -=-
    ifdef TMR0IF  ;----{ TMR0 TMR0 Overflow }----------[PIR3, TMR0IF]---
      INT_Source  PIR3,TMR0IF, PIE3,TMR0IE, IPR3,TMR0IP
    endif
    ifdef CCP1IF  ;----{ CCP1 CCP1 }----------[PIR3, CCP1IF]---
      INT_Source  PIR3,CCP1IF, PIE3,CCP1IE, IPR3,CCP1IP
    endif
    ifdef TMR1GIF  ;----{ TMR1G TMR1 Gate }----------[PIR3, TMR1GIF]---
      INT_Source  PIR3,TMR1GIF, PIE3,TMR1GIE, IPR3,TMR1GIP
    endif
    ifdef TMR1IF  ;----{ TMR1 TMR1 Overflow }----------[PIR3, TMR1IF]---
      INT_Source  PIR3,TMR1IF, PIE3,TMR1IE, IPR3,TMR1IP
    endif
    ifdef TMR2IF  ;----{ TMR2 TMR2 to T2PR Match }----------[PIR3, TMR2IF]---
      INT_Source  PIR3,TMR2IF, PIE3,TMR2IE, IPR3,TMR2IP
    endif
    ifdef SPI1IF  ;----{ SPI1 Serial Peripherial Interface 1 }----------[PIR3, SPI1IF]---
      INT_Source  PIR3,SPI1IF, PIE3,SPI1IE, IPR3,SPI1IP
    endif
    ifdef SPI1TXIF  ;----{ SPI1TX Serial Peripherial Interface 1 TX }----------[PIR3, SPI1TXIF]---
      INT_Source  PIR3,SPI1TXIF, PIE3,SPI1TXIE, IPR3,SPI1TXIP
    endif
    ifdef SPI1RXIF  ;----{ SPI1RX Serial Peripherial Interface 1 RX }----------[PIR3, SPI1RXIF]---
      INT_Source  PIR3,SPI1RXIF, PIE3,SPI1RXIE, IPR3,SPI1RXIP
    endif
; -=-   PIR4 Interrupts: -=-
    ifdef PWM1IF  ;----{ PWM1 PWM1 Parameter Interrupt }----------[PIR4, PWM1IF]---
      INT_Source  PIR4,PWM1IF, PIE4,PWM1IE, IPR4,PWM1IP
    endif
    ifdef PWM1PIF  ;----{ PWM1P PWM1 Period Interrupt }----------[PIR4, PWM1PIF]---
      INT_Source  PIR4,PWM1PIF, PIE4,PWM1PIE, IPR4,PWM1PIP
    endif
    ifdef U1IF  ;----{ U1 USART1 }----------[PIR4, U1IF]---
      INT_Source  PIR4,U1IF, PIE4,U1IE, IPR4,U1IP
    endif
    ifdef U1EIF  ;----{ U1 USART1 Framing Error }----------[PIR4, U1EIF]---
      INT_Source  PIR4,U1EIF, PIE4,U1EIE, IPR4,U1EIP
    endif
    ifdef U1TXIF  ;----{ U1TX USART1 Transmit  }----------[PIR4, U1TXIF]---
      INT_Source  PIR4,U1TXIF, PIE4,U1TXIE, IPR4,U1TXIP
    endif
    ifdef U1RXIF  ;----{ U1RX USART1 Receive }----------[PIR4, U1RXIF]---
      INT_Source  PIR4,U1RXIF, PIE4,U1RXIE, IPR4,U1RXIP
    endif
; -=-   PIR5 Interrupts: -=-
    ifdef PWM2IF  ;----{ PWM2 PWM2 Parameter Interrupt }----------[PIR5, PWM2IF]---
      INT_Source  PIR5,PWM2IF, PIE5,PWM2IE, IPR5,PWM2IP
    endif
    ifdef PWM2PIF  ;----{ PWM2P PWM2 Period Interrupt }----------[PIR5, PWM2PIF]---
      INT_Source  PIR5,PWM2PIF, PIE5,PWM2PIE, IPR5,PWM2PIP
    endif
    ifdef TMR3GIF  ;----{ TMR3G TMR3 Gate }----------[PIR5, TMR3GIF]---
      INT_Source  PIR5,TMR3GIF, PIE5,TMR3GIE, IPR5,TMR3GIP
    endif
    ifdef TMR3IF  ;----{ TMR3 TMR3 Overflow  }----------[PIR5, TMR3IF]---
      INT_Source  PIR5,TMR3IF, PIE5,TMR3IE, IPR5,TMR3IP
    endif
    ifdef SPI2IF  ;----{ SPI2 Serial Peripherial Interface 2 }----------[PIR5, SPI2IF]---
      INT_Source  PIR5,SPI2IF, PIE5,SPI2IE, IPR5,SPI2IP
    endif
    ifdef SPI2TXIF  ;----{ SPI2TX Serial Peripherial Interface 2 TX }----------[PIR5, SPI2TXIF]---
      INT_Source  PIR5,SPI2TXIF, PIE5,SPI2TXIE, IPR5,SPI2TXIP
    endif
    ifdef SPI2RXIF  ;----{ SPI2RX Serial Peripherial Interface 2 RX }----------[PIR5, SPI2RXIF]---
      INT_Source  PIR5,SPI2RXIF, PIE5,SPI2RXIE, IPR5,SPI2RXIP
    endif
; -=-   PIR6 Interrupts: -=-
    ifdef DMA2AIF  ;----{ DMA2A DMA2 Abort }----------[PIR6, DMA2AIF]---
      INT_Source  PIR6,DMA2AIF, PIE6,DMA2AIE, IPR6,DMA2AIP
    endif
    ifdef DMA2ORIF  ;----{ DMA2OR DMA2 Overrun }----------[PIR6, DMA2ORIF]---
      INT_Source  PIR6,DMA2ORIF, PIE6,DMA2ORIE, IPR6,DMA2ORIP
    endif
    ifdef DMA2DCNTIF  ;----{ DMA2DCNT DMA2 Destination Count }----------[PIR6, DMA2DCNTIF]---
      INT_Source  PIR6,DMA2DCNTIF, PIE6,DMA2DCNTIE, IPR6,DMA2DCNTIP
    endif
    ifdef DMA2SCNTIF  ;----{ DMA2SCNT Source Count }----------[PIR6, DMA2SCNTIF]---
      INT_Source  PIR6,DMA2SCNTIF, PIE6,DMA2SCNTIE, IPR6,DMA2SCNTIP
    endif
    ifdef NCO1IF  ;----{ NCO1 NCO1 Interrupt }----------[PIR6, NCO1IF]---
      INT_Source  PIR6,NCO1IF, PIE6,NCO1IE, IPR6,NCO1IP
    endif
    ifdef CWG1IF  ;----{ CWG1 Configurable Wave Generator 1 }----------[PIR6, CWG1IF]---
      INT_Source  PIR6,CWG1IF, PIE6,CWG1IE, IPR6,CWG1IP
    endif
    ifdef CLC2IF  ;----{ CLC2 Configurable Logic Cell 2 }----------[PIR6, CLC2IF]---
      INT_Source  PIR6,CLC2IF, PIE6,CLC2IE, IPR6,CLC2IP
    endif
    ifdef INT1IF  ;----{ INT1 INT1 External }----------[PIR6, INT1IF]---
      INT_Source  PIR6,INT1IF, PIE6,INT1IE, IPR6,INT1IP
    endif
; -=-   PIR7 Interrupts: -=-
    ifdef PWM3IF  ;----{ PWM3 PWM3 Parameter Interrupt }----------[PIR7, PWM3IF]---
      INT_Source  PIR7,PWM3IF, PIE7,PWM3IE, IPR7,PWM3IP
    endif
    ifdef PWM3PIF  ;----{ PWM3P PWM3 Period Interrupt }----------[PIR7, PWM3PIF]---
      INT_Source  PIR7,PWM3PIF, PIE7,PWM3PIE, IPR7,PWM3PIP
    endif
    ifdef CLC3IF  ;----{ CLC3 Configurable Logic Cell 3 }----------[PIR7, CLC3IF]---
      INT_Source  PIR7,CLC3IF, PIE7,CLC3IE, IPR7,CLC3IP
    endif
    ifdef I2C1EIF  ;----{ I2C1E I2C1 Error }----------[PIR7, I2C1EIF]---
      INT_Source  PIR7,I2C1EIF, PIE7,I2C1EIE, IPR7,I2C1EIP
    endif
    ifdef I2C1IF  ;----{ I2C1 I2C1 Interrupt }----------[PIR7, I2C1IF]---
      INT_Source  PIR7,I2C1IF, PIE7,I2C1IE, IPR7,I2C1IP
    endif
    ifdef I2C1TXIF  ;----{ I2C1TX I2C1 Transmit }----------[PIR7, I2C1TXIF]---
      INT_Source  PIR7,I2C1TXIF, PIE7,I2C1TXIE, IPR7,I2C1TXIP
    endif
    ifdef I2C1RXIF  ;----{ I2C1RX I2C1 Receive }----------[PIR7, I2C1RXIF]---
      INT_Source  PIR7,I2C1RXIF, PIE7,I2C1RXIE, IPR7,I2C1RXIP
    endif
; -=-   PIR8 Interrupts: -=-
    ifdef SCANIF  ;----{ SCAN Memory Scanner Interrupt }----------[PIR8, SCANIF]---
      INT_Source  PIR8,SCANIF, PIE8,SCANIE, IPR8,SCANIP
    endif
    ifdef CCP2IF  ;----{ CCP2 CCP2 Interrupt }----------[PIR8, CCP2IF]---
      INT_Source  PIR8,CCP2IF, PIE8,CCP2IE, IPR8,CCP2IP
    endif
    ifdef TMR5GIF  ;----{ TMR5G TMR5 Gate }----------[PIR8, TMR5GIF]---
      INT_Source  PIR8,TMR5GIF, PIE8,TMR5GIE, IPR8,TMR5GIP
    endif
    ifdef TMR5IF  ;----{ TMR5 TMR5 Overflow }----------[PIR8, TMR5IF]---
      INT_Source  PIR8,TMR5IF, PIE8,TMR5IE, IPR8,TMR5IP
    endif
    ifdef U2IF  ;----{ U2 USART2 Interrupt }----------[PIR8, U2IF]---
      INT_Source  PIR8,U2IF, PIE8,U2IE, IPR8,U2IP
    endif
    ifdef U2EIF  ;----{ U2E USART2 Framing Error }----------[PIR8, U2EIF]---
      INT_Source  PIR8,U2EIF, PIE8,U2EIE, IPR8,U2EIP
    endif
    ifdef U2TXIF  ;----{ U2TX USART2 Transmit }----------[PIR8, U2TXIF]---
      INT_Source  PIR8,U2TXIF, PIE8,U2TXIE, IPR8,U2TXIP
    endif
    ifdef U2RXIF  ;----{ U2RX USART2 Receive }----------[PIR8, U2RXIF]---
      INT_Source  PIR8,U2RXIF, PIE8,U2RXIE, IPR8,U2RXIP
    endif
; -=-   PIR9 Interrupts: -=-
    ifdef CLC4IF  ;----{ CLC4 Configurable Logic Cell 4 }----------[PIR9, CLC4IF]---
      INT_Source  PIR9,CLC4IF, PIE9,CLC4IE, IPR9,CLC4IP
    endif
    ifdef U3IF  ;----{ U3 USART3 Interrupt }----------[PIR9, U3IF]---
      INT_Source  PIR9,U3IF, PIE9,U3IE, IPR9,U3IP
    endif
    ifdef U3EIF  ;----{ U3E USART3 Framing Error }----------[PIR9, U3EIF]---
      INT_Source  PIR9,U3EIF, PIE9,U3EIE, IPR9,U3EIP
    endif
    ifdef U3TXIF  ;----{ U3TX USART3 Transmit }----------[PIR9, U3TXIF]---
      INT_Source  PIR9,U3TXIF, PIE9,U3TXIE, IPR9,U3TXIP
    endif
    ifdef U3RXIF  ;----{ U3RX USART3 Receive }----------[PIR9, U3RXIF]---
      INT_Source  PIR9,U3RXIF, PIE9,U3RXIE, IPR9,U3RXIP
    endif
; -=-   PIR10 Interrupts: -=-
    ifdef DMA3AIF  ;----{ DMA3A DMA3 Abort }----------[PIR10, DMA3AIF]---
      INT_Source  PIR10,DMA3AIF, PIE10,DMA3AIE, IPR10,DMA3AIP
    endif
    ifdef DMA3ORIF  ;----{ DMA3OR DMA3 Overrun }----------[PIR10, DMA3ORIF]---
      INT_Source  PIR10,DMA3ORIF, PIE10,DMA3ORIE, IPR10,DMA3ORIP
    endif
    ifdef DMA3DCNTIF  ;----{ DMA3DCNT DMA3 Destination Count }----------[PIR10, DMA3DCNTIF]---
      INT_Source  PIR10,DMA3DCNTIF, PIE10,DMA3DCNTIE, IPR10,DMA3DCNTIP
    endif
    ifdef DMA3SCNTIF  ;----{ DMA3SCNT DMA3 Source Count }----------[PIR10, DMA3SCNTIF]---
      INT_Source  PIR10,DMA3SCNTIF, PIE10,DMA3SCNTIE, IPR10,DMA3SCNTIP
    endif
    ifdef NCO2IF  ;----{ NCO2 NCO2 Interrupt }----------[PIR10, NCO2IF]---
      INT_Source  PIR10,NCO2IF, PIE10,NCO2IE, IPR10,NCO2IP
    endif
    ifdef CWG2IF  ;----{ CWG2 Configurable Wave Generator 2 }----------[PIR10, CWG2IF]---
      INT_Source  PIR10,CWG2IF, PIE10,CWG2IE, IPR10,CWG2IP
    endif
    ifdef CLC5IF  ;----{ CLC5 Configurable Logic Cell 5 }----------[PIR10, CLC5IF]---
      INT_Source  PIR10,CLC5IF, PIE10,CLC5IE, IPR10,CLC5IP
    endif
    ifdef INT2IF  ;----{ INT2 INT2 External  }----------[PIR10, INT2IF]---
      INT_Source  PIR10,INT2IF, PIE10,INT2IE, IPR10,INT2IP
    endif
; -=-   PIR11 Interrupts: -=-
    ifdef DMA4AIF  ;----{ DMA4A DMA4 Abort }----------[PIR11, DMA4AIF]---
      INT_Source  PIR11,DMA4AIF, PIE11,DMA4AIE, IPR11,DMA4AIP
    endif
    ifdef DMA4ORIF  ;----{ DMA4OR DMA4 Overrun }----------[PIR11, DMA4ORIF]---
      INT_Source  PIR11,DMA4ORIF, PIE11,DMA4ORIE, IPR11,DMA4ORIP
    endif
    ifdef DMA4DCNTIF  ;----{ DMA4DCNT DMA4 Destination Count }----------[PIR11, DMA4DCNTIF]---
      INT_Source  PIR11,DMA4DCNTIF, PIE11,DMA4DCNTIE, IPR11,DMA4DCNTIP
    endif
    ifdef DMA4SCNTIF  ;----{ DMA4SCNT DMA4 Source Count }----------[PIR11, DMA4SCNTIF]---
      INT_Source  PIR11,DMA4SCNTIF, PIE11,DMA4SCNTIE, IPR11,DMA4SCNTIP
    endif
    ifdef TMR4IF  ;----{ TMR4 TMR4-T4PR Match }----------[PIR11, TMR4IF]---
      INT_Source  PIR11,TMR4IF, PIE11,TMR4IE, IPR11,TMR4IP
    endif
    ifdef CWG3IF  ;----{ CWG3 Configurable Wave Generator 3 }----------[PIR11, CWG3IF]---
      INT_Source  PIR11,CWG3IF, PIE11,CWG3IE, IPR11,CWG3IP
    endif
    ifdef CLC6IF  ;----{ CLC6 Configurable Logic Cell 6 }----------[PIR11, CLC6IF]---
      INT_Source  PIR11,CLC6IF, PIE11,CLC6IE, IPR11,CLC6IP
    endif
    ifdef CCP3IF  ;----{ CCP3  CCP3 }----------[PIR11, CCP3IF]---
      INT_Source  PIR11,CCP3IF, PIE11,CCP3IE, IPR11,CCP3IP
    endif
; -=-   PIR12 Interrupts: -=-
    ifdef DMA5AIF  ;----{ DMA5A DMA5 Abort }----------[PIR12, DMA5AIF]---
      INT_Source  PIR12,DMA5AIF, PIE12,DMA5AIE, IPR12,DMA5AIP
    endif
    ifdef DMA5ORIF  ;----{ DMA5OR DMA5 Overrun }----------[PIR12, DMA5ORIF]---
      INT_Source  PIR12,DMA5ORIF, PIE12,DMA5ORIE, IPR12,DMA5ORIP
    endif
    ifdef DMA5DCNTIF  ;----{ DMA5DCNT DMA5 Destination Count }----------[PIR12, DMA5DCNTIF]---
      INT_Source  PIR12,DMA5DCNTIF, PIE12,DMA5DCNTIE, IPR12,DMA5DCNTIP
    endif
    ifdef DMA5SCNTIF  ;----{ DMA5SCNT DMA5 Source Count }----------[PIR12, DMA5SCNTIF]---
      INT_Source  PIR12,DMA5SCNTIF, PIE12,DMA5SCNTIE, IPR12,DMA5SCNTIP
    endif
    ifdef U4IF  ;----{ U4 USART4 Interrupt }----------[PIR12, U4IF]---
      INT_Source  PIR12,U4IF, PIE12,U4IE, IPR12,U4IP
    endif
    ifdef U4EIF  ;----{ U4E USART4 Framing Error }----------[PIR12, U4EIF]---
      INT_Source  PIR12,U4EIF, PIE12,U4EIE, IPR12,U4EIP
    endif
    ifdef U4TXIF  ;----{ U4TX USART4 Transmit }----------[PIR12, U4TXIF]---
      INT_Source  PIR12,U4TXIF, PIE12,U4TXIE, IPR12,U4TXIP
    endif
    ifdef U4RXIF  ;----{ U4RX USART4 Receive }----------[PIR12, U4RXIF]---
      INT_Source  PIR12,U4RXIF, PIE12,U4RXIE, IPR12,U4RXIP
    endif
; -=-   PIR13 Interrupts: -=-
    ifdef DMA6AIF  ;----{ DMA6A DMA6 Abort }----------[PIR13, DMA6AIF]---
      INT_Source  PIR13,DMA6AIF, PIE13,DMA6AIE, IPR13,DMA6AIP
    endif
    ifdef DMA6ORIF  ;----{ DMA6OR DMA6 Overrun }----------[PIR13, DMA6ORIF]---
      INT_Source  PIR13,DMA6ORIF, PIE13,DMA6ORIE, IPR13,DMA6ORIP
    endif
    ifdef DMA6DCNTIF  ;----{ DMA6DCNT DMA6 Destination Count }----------[PIR13, DMA6DCNTIF]---
      INT_Source  PIR13,DMA6DCNTIF, PIE13,DMA6DCNTIE, IPR13,DMA6DCNTIP
    endif
    ifdef DMA6SCNTIF  ;----{ DMA6SCNT DMA6 Source Count }----------[PIR13, DMA6SCNTIF]---
      INT_Source  PIR13,DMA6SCNTIF, PIE13,DMA6SCNTIE, IPR13,DMA6SCNTIP
    endif
    ifdef U5IF  ;----{ U5 USART5 Interrupt }----------[PIR13, U5IF]---
      INT_Source  PIR13,U5IF, PIE13,U5IE, IPR13,U5IP
    endif
    ifdef U5EIF  ;----{ U5E USART5 Framing Error }----------[PIR13, U5EIF]---
      INT_Source  PIR13,U5EIF, PIE13,U5EIE, IPR13,U5EIP
    endif
    ifdef U5TXIF  ;----{ U5TX USART5 Transmit }----------[PIR13, U5TXIF]---
      INT_Source  PIR13,U5TXIF, PIE13,U5TXIE, IPR13,U5TXIP
    endif
    ifdef U5RXIF  ;----{ U5RX USART5 Receive }----------[PIR13, U5RXIF]---
      INT_Source  PIR13,U5RXIF, PIE13,U5RXIE, IPR13,U5RXIP
    endif
; -=-   PIR14 Interrupts: -=-
    ifdef NCO3IF  ;----{ NCO3 NCO3 Interrupt }----------[PIR14, NCO3IF]---
      INT_Source  PIR14,NCO3IF, PIE14,NCO3IE, IPR14,NCO3IP
    endif
    ifdef CM2IF  ;----{ CM2 Comparator2  Interrupt }----------[PIR14, CM2IF]---
      INT_Source  PIR14,CM2IF, PIE14,CM2IE, IPR14,CM2IP
    endif
    ifdef CLC7IF  ;----{ CLC7 Configurable Logic Cell 7 }----------[PIR14, CLC7IF]---
      INT_Source  PIR14,CLC7IF, PIE14,CLC7IE, IPR14,CLC7IP
    endif
; -=-   PIR15 Interrupts: -=-
    ifdef TMR6IF  ;----{ TMR6 TMR6-T6PR Match }----------[PIR15, TMR6IF]---
      INT_Source  PIR15,TMR6IF, PIE15,TMR6IE, IPR15,TMR6IP
    endif
    ifdef CRCIF  ;----{ CRC Cyclic Redundancy Check }----------[PIR15, CRCIF]---
      INT_Source  PIR15,CRCIF, PIE15,CRCIE, IPR15,CRCIP
    endif
    ifdef CLC8IF  ;----{ CLC8 Configurable Logic Cell 8 }----------[PIR15, CLC8IF]---
      INT_Source  PIR15,CLC8IF, PIE15,CLC8IE, IPR15,CLC8IP
    endif
    ifdef NVMIF  ;----{ NVM EEPROM/FLASH Write Operation }----------[PIR15, NVMIF]---
      INT_Source  PIR15,NVMIF, PIE15,NVMIE, IPR15,NVMIP
    endif
    list
  endm
  ENDASM
#endif      ' (DT_INTS_18XV_FAMILY = "Q43")

;-------------------------
;---- 18FxxK42 family ----
;-------------------------
#if (DT_INTS_18XV_FAMILY = "K42")
  ASM
    nolist
    ; create alias symbols for comparators C1 and C2
    #define CM1IF C1IF
    #define CM1IE C1IE
    #define CM1IP C1IP
    #define CM2IF C2IF
    #define CM2IE C2IE
    #define CM2IP C2IP
    #define CM1_INT C1_INT
    #define CM2_INT C2_INT

; -=-   PIR0 Interrupts: -=-
    #define IOC_INT     PIR0, IOCIF     ;-- IOC INT Port Change 
    #define CRC_INT     PIR0, CRCIF     ;-- Cyclic Redundancy Check
    #define SCAN_INT    PIR0, SCANIF    ;
    #define NVM_INT     PIR0, NVMIF     ;-- EEPROM/FLASH Write Operation   
    #define CSW_INT     PIR0, CSWIF     ;
    #define OSF_INT     PIR0, OSFIF     ;-- Oscillator Fail
    #define HLVD_INT    PIR0, HLVDIF    ;-- High/Low Voltage Detect 
    #define SW_INT      PIR0, SWIF      ;
; -=-   PIR1 Interrupts: -=-
    #define SMT1PWA_INT PIR1, SMT1PWAIF     ;
    #define SMT1PRA_INT PIR1, SMT1PRAIF     ; 
    #define SMT1_INT    PIR1, SMT1IF    ;
    #define C1_INT      PIR1, C1IF      ;-- Comparator1
    #define ADT_INT     PIR1, ADTIF     ;
    #define AD_INT      PIR1, ADIF      ;-- A/D Converter 
    #define ZCD_INT     PIR1, ZCDIF     ;Zero Cross Detect
    #define INT0_INT    PIR1, INT0IF    ;-- INT0 External
; -=-   PIR2 Interrupts: -=-
    #define I2C1RX_INT  PIR2, I2C1RXIF ;I2C1 Receive
    #define SPI1_INT    PIR2, SPI1IF   ;Serial Peripherial Interface 1
    #define SPI1TX_INT  PIR2, SPI1TXIF ;Serial Peripherial Interface 1 TX
    #define SPI1RX_INT  PIR2, SPI1RXIF ;Serial Peripherial Interface 1 RX
    #define DMA1A_INT   PIR2, DMA1AIF   ;
    #define DMA1OR_INT  PIR2, DMA1ORIF ;
    #define DMA1DCNT_INT  PIR2, DMA1DCNTIF ;
    #define DMA1SCNT_INT  PIR2, DMA1SCNTIF ;
; -=-   PIR3 Interrupts: -=-
    #define TMR0_INT    PIR3, TMR0IF    ;-- TMR0 Overflow
    #define U1_INT      PIR3, U1IF      ; -- USART1
    #define U1E_INT     PIR3, U1EIF     ; -- USART1
    #define U1TX_INT    PIR3, U1TXIF    ;-- USART1 Transmit 
    #define U1RX_INT    PIR3, U1RXIF    ;-- USART1 Receive 
    #define I2C1E_INT   PIR3, I2C1EIF   ;
    #define I2C1_INT    PIR3, I2C1IF    ;
    #define I2C1TX_INT  PIR3, I2C1TXIF ;
; -=-   PIR4 Interrupts: -=-
    #define CLC1_INT    PIR4, CLC1IF    ;
    #define CWG1_INT    PIR4, CWG1IF    ;
    #define NCO1_INT    PIR4, NCO1IF    ;
    ; --bit 4 unused--
    #define CCP1_INT    PIR4, CCP1IF    ;-- CCP1 
    #define TMR2_INT    PIR4, TMR2IF    ;-- TMR2 to PR2 Match 
    #define TMR1G_INT   PIR4, TMR1GIF   ;-- TMR1 Gate
    #define TMR1_INT    PIR4, TMR1IF    ;-- TMR1 Overflow 
; -=-   PIR5 Interrupts: -=-
    #define I2C2TX_INT  PIR5, I2C2TXIF  ;
    #define I2C2RX_INT  PIR5, I2C2RXIF  ;
    #define DMA2A_INT   PIR5, DMA2AIF   ;
    #define DMA2OR_INT  PIR5, DMA2ORIF  ;
    #define DMA2DCNT_INT  PIR5, DMA2DCNTIF ;
    #define DMA2SCNT_INT  PIR5, DMA2SCNTIF ;
    #define C2_INT      PIR5, C2IF      ;-- Comparator2 
    #define INT1_INT    PIR5, INT1IF    ;-- INT1 External
; -=-   PIR6 Interrupts: -=-
    #define TMR3G_INT   PIR6, TMR3GIF   ;-- TMR3 Gate
    #define TMR3_INT    PIR6, TMR3IF    ;-- TMR3 Overflow 
    #define U2_INT      PIR6, U2IF      ; -- USART2 
    #define U2E_INT     PIR6, U2EIF     ; -- USART2 
    #define U2TX_INT    PIR6, U2TXIF    ;-- USART2 Transmit
    #define U2RX_INT    PIR6, U2RXIF    ;-- USART2 Receive
    #define I2C2E_INT   PIR6, I2C2EIF   ;
    #define I2C2_INT    PIR6, I2C2IF    ;
; -=-   PIR7 Interrupts: -=-
    ; --bit 7 unused--
    ; --bit 6 unused--
    #define INT2_INT    PIR7, INT2IF    ;-- INT2 External 
    #define CLC2_INT    PIR7, CLC2IF    ;
    #define CWG2_INT    PIR7, CWG2IF    ;
    ; --bit 2 unused--
    #define CCP2_INT    PIR7, CCP2IF    ;-- CCP2 
    #define TMR4_INT    PIR7, TMR4IF    ;-- TMR4 Overflow
; -=-   PIR8 Interrupts: -=-
    #define TMR5G_INT   PIR8, TMR5GIF   ;-- TMR5 Gate
    #define TMR5_INT    PIR8, TMR5IF    ;-- TMR5 Overflow
    ; --bit 5 unused--
    ; --bit 4 unused--
    ; --bit 3 unused--
    ; --bit 2 unused--
    ; --bit 1 unused--
    ; --bit 0 unused--
; -=-   PIR9 Interrupts: -=-
    ; --bit 7 unused--
    ; --bit 6 unused--
    ; --bit 5 unused--
    ; --bit 4 unused--
    #define CLC3_INT    PIR9, CLC3IF    ;
    #define CWG3_INT    PIR9, CWG3IF    ;
    #define CCP3_INT    PIR9, CCP3IF    ;-- CCP3 
    #define TMR6_INT    PIR9, TMR6IF    ;-- TMR6 Overflow
; -=-   PIR10 Interrupts: -=-
    ; --bit 7 unused--
    ; --bit 6 unused--
    ; --bit 5 unused--
    ; --bit 4 unused--
    ; --bit 3 unused--
    ; --bit 2 unused--
    #define CLC4_INT    PIR10, CLC4IF   ;
    #define CCP4_INT    PIR10, CCP4IF   ;-- CCP4 
    list
  ENDASM
#endif

;-- GetIntInfo (**device specific**)
#if (DT_INTS_18XV_FAMILY = "K42")
  ASM
  GetIntInfo  macro  IflagReg, IflagBit
    nolist
    INT_Flag_Reg = -1
    INT_Flag_Bit = -1
    INT_Enable_Reg = -1
    INT_Enable_Bit = -1
    ifdef  IPR1
      INT_Priority_Reg = -1     ; cannot change priority (ie INT0_INT on some devices)
      INT_Priority_Bit = -1
    endif
    IntFound = 0
  
; -=- 18FxxK42 family -=-
; -=-   PIR0 Interrupts: -=-
    ifdef IOCIF   ;----{ Port Change Interrupt }-----------[PIR0, IOCIF]---
      INT_Source  PIR0,IOCIF, PIE0,IOCIE, IPR0,IOCIP
    endif
    ifdef CRCIF  ;----{ Cyclic Redundancy Check Interrupt Flag }-----[PIR0, CRCIF]---
      INT_Source  PIR0,CRCIF, PIE0,CRCIE, IPR0,CRCIP
    endif
    ifdef SCANIF  ;----{ SCAN Interrupt Flag }-----------[PIR0, SCANIF]---
      INT_Source  PIR0,SCANIF, PIE0,SCANIE, IPR0,SCANIP
    endif
    ifdef NVMIF    ;----{ EEPROM/FLASH Write Operation Interrupt } [PIR0, NVMIF]---
      INT_Source  PIR0,NVMIF, PIE0,NVMIE, IPR0,NVMIP
    endif
    ifdef CSWIF  ;----{ Clock Switch Interrupt Flag }-----------[PIR0, CSWIF]---
      INT_Source  PIR0,CSWIF, PIE0,CSWIE, IPR0,CSWIP
    endif
    ifdef OSFIF  ;----{ Osc Fail Interrupt Flag }-----------[PIR0, OSFIF]---
      INT_Source  PIR0,OSFIF, PIE0,OSFIE, IPR0,OSFIP
    endif
    ifdef HLVDIF  ;----{ High/Low Voltage Detect Interrupt }--[PIR0, HLVDIF]---
      INT_Source  PIR0,HLVDIF, PIE0,HLVDIE, IPR0,HLVDIP
    endif
    ifdef SWIF  ;----{ Software Interrupt Flag }-----------[PIR0, SWIF]---
      INT_Source  PIR0,SWIF, PIE0,SWIE, IPR0,SWIP
    endif
; -=-   PIR1 Interrupts: -=-
    ifdef SMT1PWAIF  ;----{ SMT 1 Pulse-Width Acquisition Interrupt Flag }----[PIR1, SMT1PWAIF]---
      INT_Source  PIR1,SMT1PWAIF, PIE1,SMT1PWAIE, IPR1,SMT1PWAIP
    endif
    ifdef SMT1PRAIF  ;----{ SMT 1 Period Acquisition Interrupt Flag }----[PIR1, SMT1PRAIF]---
      INT_Source  PIR1,SMT1PRAIF, PIE1,SMT1PRAIE, IPR1,SMT1PRAIP
    endif
    ifdef SMT1IF  ;----{ Signal Measurement Timer 1 Interrupt Flag }----[PIR1, SMT1IF]---
      INT_Source  PIR1,SMT1IF, PIE1,SMT1IE, IPR1,SMT1IP
    endif
    ifdef C1IF    ;----{ Comparator 1 Interrupt }-----------------[PIR1, C1IF]---
      INT_Source  PIR1,C1IF, PIE1,C1IE, IPR1,C1IP
    endif
    ifdef ADTIF    ;----{ A/D Threshold Interrupt }--------------[PIR1, ADTIF]---
      INT_Source  PIR1,ADTIF, PIE1,ADTIE, IPR1,ADTIP
    endif
    ifdef ADIF    ;----{ A/D Converter Interrupt }--------------[PIR1, ADIF]---
      INT_Source  PIR1,ADIF, PIE1,ADIE, IPR1,ADIP
    endif
    ifdef ZCDIF  ;----{ Zero Cross Detect Flag }-----------[PIR1, ZCDIF]---
      INT_Source  PIR1,ZCDIF, PIE1,ZCDIE, IPR1,ZCDIP
    endif
    ifdef INT0IF  ;----{ INT0 External Interrupt }----------[PIR1, INT0IF]---
      INT_Source  PIR1,INT0IF, PIE1,INT0IE, IPR1, INT0IP
    endif
; -=-   PIR2 Interrupts: -=-
    ifdef I2C1RXIF   ;----{ I2C 1 Receive Interrupt }---[PIR2, I2C1RXIF]---
      INT_Source  PIR2,I2C1RXIF, PIE2,I2C1RXIE, IPR2,I2C1RXIP
    endif
    ifdef SPI1IF   ;----{ Synchronous Serial Port 1 Interrupt }---[PIR2, SPI1IF]---
      INT_Source  PIR2,SPI1IF, PIE2,SPI1IE, IPR2,SPI1IP
    endif
    ifdef SPI1TXIF   ;----{ SPI Transmit Interrupt }---[PIR2, SPI1TXIF]---
      INT_Source  PIR2,SPI1TXIF, PIE2,SPI1TXIE, IPR2,SPI1TXIP
    endif
    ifdef SPI1RXIF   ;----{ SPI Receive Interrupt }---[PIR2, SPI1RXIF]---
      INT_Source  PIR2,SPI1RXIF, PIE2,SPI1RXIE, IPR2,SPI1RXIP
    endif
    ifdef DMA1AIF    ;----{ Direct Memory Access 1 Interrupt } [PIR2, DMA1AIF]---
      INT_Source  PIR2,DMA1AIF, PIE2,DMA1AIE, IPR2,DMA1AIP
    endif
    ifdef DMA1ORIF    ;----{ DMA 1 Over Run Interrupt } [PIR2, DMA1ORIF]---
      INT_Source  PIR2,DMA1ORIF, PIE2,DMA1ORIE, IPR2,DMA1ORIP
    endif
    ifdef DMA1DCNTIF    ;----{ DMA 1 Designation Count Interrupt } [PIR2, DMA1DCNTIF]---
      INT_Source  PIR2,DMA1DCNTIF, PIE2,DMA1DCNTIE, IPR2,DMA1DCNTIP
    endif
    ifdef DMA1SCNTIF    ;----{ DMA 1 Source Count Interrupt } [PIR2, DMA1SCNTIF]---
      INT_Source  PIR2,DMA1SCNTIF, PIE2,DMA1SCNTIE, IPR2,DMA1SCNTIP
    endif
; -=-   PIR3 Interrupts: -=-
    ifdef TMR0IF  ;----{ TMR0 Overflow Interrupt }----------[PIR3, TMR0IF]---
      INT_Source  PIR3,TMR0IF, PIE3,TMR0IE, IPR3,TMR0IP
    endif
    ifdef U1IF    ;----{ UART1 Interrupt }-------------[PIR3, U1IF]---
      INT_Source  PIR3,U1IF, PIE3,U1IE, IPR3,U1IP
    endif
    ifdef U1EIF    ;----{ UART1 Framing Error Interrupt }-------------[PIR3, U1EIF]---
      INT_Source  PIR3,U1EIF, PIE3,U1EIE, IPR3,U1EIP
    endif
    ifdef U1TXIF  ;----{ USART1 Transmit Interrupt }-------------[PIR3, U1TXIF]---
      INT_Source  PIR3,U1TXIF, PIE3,U1TXIE, IPR3,U1TXIP
    endif
    ifdef U1RXIF   ;----{ USART1 Receive Interrupt }------------[PIR3, U1RXIF]---
      INT_Source  PIR3,U1RXIF, PIE3,U1RXIE, IPR3,U1RXIP
    endif
    ifdef I2C1EIF   ;----{ I2C 1 Error Interrupt }---[PIR3, I2C1EIF]---
      INT_Source  PIR3,I2C1EIF, PIE3,I2C1EIE, IPR3,I2C1EIP
    endif
    ifdef I2C1IF   ;----{ I2C 1 Interrupt }---[PIR3, I2C1IF]---
      INT_Source  PIR3,I2C1IF, PIE3,I2C1IE, IPR3,I2C1IP
    endif
    ifdef I2C1TXIF   ;----{ I2C 1 Transmit Interrupt }---[PIR3, I2C1TXIF]---
      INT_Source  PIR3,I2C1TXIF, PIE3,I2C1TXIE, IPR3,I2C1TXIP
    endif
; -=-   PIR4 Interrupts: -=-
    ifdef CLC1IF  ;----{ Configurable Logic Cell 1 Flag }-----------[PIR4, CLC1IF]---
      INT_Source  PIR4,CLC1IF, PIE4,CLC1IE, IPR4,CLC1IP
    endif
    ifdef CWG1IF  ;----{ Complimentary Waveform Generagor 1 Flag }----[PIR4, CWG1IF]---
      INT_Source  PIR4,CWG1IF, PIE4,CWG1IE, IPR4,CWG1IP
    endif
    ifdef NCO1IF  ;----{ CNumerically Controlled Oscillator 1 Flag }----[PIR4, NCO1IF]---
      INT_Source  PIR4,NCO1IF, PIE4,NCO1IE, IPR4,NCO1IP
    endif
    ifdef CCP1IF  ;----{ CCP1 Interrupt }---------------------[PIR4, CCP1IF]---
      INT_Source  PIR4,CCP1IF, PIE4,CCP1IE, IPR4,CCP1IP
    endif
    ifdef TMR2IF  ;----{ TMR2 to PR2 Match Interrupt }--------[PIR4, TMR2IF]---
      INT_Source  PIR4,TMR2IF, PIE4,TMR2IE, IPR4,TMR2IP
    endif
    ifdef TMR1GIF  ;----{ TMR1 Gate Interrupt }------------[PIR4, TMR1GIF]---
      INT_Source  PIR4,TMR1GIF, PIE4,TMR1GIE, IPR4,TMR1GIP
    endif
    ifdef TMR1IF  ;----{ TMR1 Overflow Interrupt }------------[PIR4, TMR1IF]---
      INT_Source  PIR4,TMR1IF, PIE4,TMR1IE, IPR4,TMR1IP
    endif
; -=-   PIR5 Interrupts: -=-
    ifdef I2C2TXIF   ;----{ I2C 2 Transmit Interrupt }---[PIR5, I2C2TXIF]---
      INT_Source  PIR5,I2C2TXIF, PIE5,I2C2TXIE, IPR5,I2C2TXIP
    endif
    ifdef I2C2RXIF   ;----{ I2C 2 Receive Interrupt }---[PIR5, I2C2RXIF]---
      INT_Source  PIR5,I2C2RXIF, PIE5,I2C2RXIE, IPR5,I2C2RXIP
    endif
    ifdef DMA2AIF    ;----{ Direct Memory Access 2 Interrupt } [PIR5, DMA2AIF]---
      INT_Source  PIR5,DMA2AIF, PIE5,DMA2AIE, IPR5,DMA2AIP
    endif
    ifdef DMA2ORIF    ;----{ DMA 2 Over Run Interrupt } [PIR5, DMA2ORIF]---
      INT_Source  PIR5,DMA2ORIF, PIE5,DMA2ORIE, IPR5,DMA2ORIP
    endif
    ifdef DMA2DCNTIF    ;----{ DMA 2 Designation Count Interrupt } [PIR5, DMA2DCNTIF]---
      INT_Source  PIR5,DMA2DCNTIF, PIE5,DMA2DCNTIE, IPR5,DMA2DCNTIP
    endif
    ifdef DMA2SCNTIF    ;----{ DMA 2 Source Count Interrupt } [PIR5, DMA2SCNTIF]---
      INT_Source  PIR5,DMA2SCNTIF, PIE5,DMA2SCNTIE, IPR5,DMA2SCNTIP
    endif
    ifdef C2IF    ;----{ Comparator 2 Interrupt }-----------------[PIR5, C2IF]---
      INT_Source  PIR5,C2IF, PIE5,C2IE, IPR5,C2IP
    endif
    ifdef INT1IF  ;----{ INT1 External Interrupt }---------[PIR5, INT1IF]---
      INT_Source  PIR5,INT1IF, PIE5,INT1IE, IPR5,INT1IP
    endif
; -=-   PIR6 Interrupts: -=-
    ifdef TMR3GIF  ;----{ TMR3 Gate Interrupt }------------[PIR6, TMR3GIF]---
      INT_Source  PIR6,TMR3GIF, PIE6,TMR3GIE, IPR6,TMR3GIP
    endif
    ifdef TMR3IF  ;----{ TMR3 Overflow Interrupt }------------[PIR6, TMR3IF]---
      INT_Source  PIR6,TMR3IF, PIE6,TMR3IE, IPR6,TMR3IP
    endif
    ifdef U2IF   ;----{ UART2 Interrupt }-----------[PIR6, U2IF]---
      INT_Source  PIR6,U2IF, PIE6,U2IE, IPR6,U2IP
    endif
    ifdef U2EIF   ;----{ UART2 Framing Error Interrupt }-----------[PIR6, U2EIF]---
      INT_Source  PIR6,U2EIF, PIE6,U2EIE, IPR6,U2EIP
    endif
    ifdef U2TXIF   ;----{ USART2 Transmit Interrupt }-----------[PIR6, U2TXIF]---
      INT_Source  PIR6,U2TXIF, PIE6,U2TXIE, IPR6,U2TXIP
    endif
    ifdef U2RXIF   ;----{ USART2 Receive Interrupt }------------[PIR6, U2RXIF]---
      INT_Source  PIR6,U2RXIF, PIE6,U2RXIE, IPR6,U2RXIP
    endif
    ifdef I2C2EIF   ;----{ I2C 2 Error Interrupt }---[PIR6, I2C2EIF]---
      INT_Source  PIR6,I2C2EIF, PIE6,I2C2EIE, IPR6,I2C2EIP
    endif  
    ifdef I2C2IF   ;----{ I2C 2 Interrupt }---[PIR6, I2C2IF]---
      INT_Source  PIR6,I2C2IF, PIE6,I2C2IE, IPR6,I2C2IP
    endif
; -=-   PIR7 Interrupts: -=-
    ifdef INT2IF  ;----{ INT2 External Interrupt }---------[PIR7, INT2IF]---
      INT_Source  PIR7,INT2IF, PIE7,INT2IE, IPR7,INT2IP
    endif
    ifdef CLC2IF  ;----{ Configurable Logic Cell 2 Flag }-----------[PIR7, CLC2IF]---
      INT_Source  PIR7,CLC2IF, PIE7,CLC2IE, IPR7,CLC2IP
    endif
    ifdef CWG2IF  ;----{ Complimentary Waveform Generagor 2 Flag }----[PIR7, CWG2IF]---
      INT_Source  PIR7,CWG2IF, PIE7,CWG2IE, IPR7,CWG2IP
    endif
    ifdef CCP2IF  ;----{ CCP2 Interrupt Flag }----------------[PIR7, CCP2IF]---
      INT_Source  PIR7,CCP2IF, PIE7,CCP2IE, IPR7,CCP2IP
    endif
    ifdef TMR4IF  ;----{ TMR4 Overflow Interrupt }------------[PIR7, TMR4IF]---
      INT_Source  PIR7,TMR4IF, PIE7,TMR4IE, IPR7,TMR4IP
    endif
; -=-   PIR8 Interrupts: -=-
    ifdef TMR5GIF  ;----{ TMR5 Gate Interrupt }------------[PIR8, TMR5GIF]---
      INT_Source  PIR8,TMR5GIF, PIE8,TMR5GIE, IPR8,TMR5GIP
    endif
    ifdef TMR5IF  ;----{ TMR5 Overflow Interrupt }------------[PIR8, TMR5IF]---
      INT_Source  PIR8,TMR5IF, PIE8,TMR5IE, IPR8,TMR5IP
    endif
; -=-   PIR9 Interrupts: -=-
    ifdef CLC3IF  ;----{ Configurable Logic Cell 3 Flag }-----------[PIR9, CLC3IF]---
      INT_Source  PIR9,CLC3IF, PIE9,CLC3IE, IPR9,CLC3IP
    endif
    ifdef CWG3IF  ;----{ Complimentary Waveform Generagor 3 Flag }----[PIR9, CWG3IF]---
      INT_Source  PIR9,CWG3IF, PIE9,CWG3IE, IPR9,CWG3IP
    endif
    ifdef CCP3IF  ;----{ CCP3 Interrupt Flag }----------------[PIR9, CCP3IF]---
      INT_Source  PIR9,CCP3IF, PIE9,CCP3IE, IPR9,CCP3IP
    endif
    ifdef TMR6IF  ;----{ TMR6 Overflow Interrupt }------------[PIR9, TMR6IF]---
      INT_Source  PIR9,TMR6IF, PIE9,TMR6IE, IPR9,TMR6IP
    endif
; -=-   PIR10 Interrupts: -=-
    ifdef CLC4IF  ;----{ Configurable Logic Cell 4 Flag }-----------[PIR10, CLC4IF]---
      INT_Source  PIR10,CLC4IF, PIE10,CLC4IE, IPR10,CLC4IP
    endif
    ifdef CCP4IF  ;----{ CCP4 Interrupt Flag }----------------[PIR10, CCP4IF]---
      INT_Source  PIR10,CCP4IF, PIE10,CCP4IE, IPR10,CCP4IP
    endif
    list
  endm
  ENDASM
#endif      ' (DT_INTS_18XV_FAMILY = "K42")

;-------------------------
;---- 18FxxK83 family ----
;-------------------------
#if (DT_INTS_18XV_FAMILY = "K83")
  ASM
    nolist
    ; create alias symbols for comparators C1 and C2
    #define CM1IF C1IF
    #define CM1IE C1IE
    #define CM1IP C1IP
    #define CM2IF C2IF
    #define CM2IE C2IE
    #define CM2IP C2IP
    #define CM1_INT C1_INT
    #define CM2_INT C2_INT

; -=-   PIR0 Interrupts: -=-
    #define IOC_INT     PIR0, IOCIF     ;-- IOC INT Port Change 
    #define CRC_INT     PIR0, CRCIF     ;-- Cyclic Redundancy Check
    #define SCAN_INT    PIR0, SCANIF    ;
    #define NVM_INT     PIR0, NVMIF     ;-- EEPROM/FLASH Write Operation   
    #define CSW_INT     PIR0, CSWIF     ;
    #define OSF_INT     PIR0, OSFIF     ;-- Oscillator Fail
    #define HLVD_INT    PIR0, HLVDIF    ;-- High/Low Voltage Detect 
    #define SW_INT      PIR0, SWIF      ;
; -=-   PIR1 Interrupts: -=-
    #define SMT1PWA_INT PIR1, SMT1PWAIF     ;
    #define SMT1PRA_INT PIR1, SMT1PRAIF     ; 
    #define SMT1_INT    PIR1, SMT1IF    ;
    #define C1_INT      PIR1, C1IF      ;-- Comparator1
    #define ADT_INT     PIR1, ADTIF     ;
    #define AD_INT      PIR1, ADIF      ;-- A/D Converter 
    #define ZCD_INT     PIR1, ZCDIF     ;Zero Cross Detect
    #define INT0_INT    PIR1, INT0IF    ;-- INT0 External
; -=-   PIR2 Interrupts: -=-
    #define I2C1RX_INT  PIR2, I2C1RXIF ;I2C1 Receive
    #define SPI1_INT    PIR2, SPI1IF   ;Serial Peripherial Interface 1
    #define SPI1TX_INT  PIR2, SPI1TXIF ;Serial Peripherial Interface 1 TX
    #define SPI1RX_INT  PIR2, SPI1RXIF ;Serial Peripherial Interface 1 RX
    #define DMA1A_INT   PIR2, DMA1AIF  ;
    #define DMA1OR_INT  PIR2, DMA1ORIF ;
    #define DMA1DCNT_INT  PIR2, DMA1DCNTIF ;
    #define DMA1SCNT_INT  PIR2, DMA1SCNTIF ;
; -=-   PIR3 Interrupts: -=-
    #define TMR0_INT    PIR3, TMR0IF    ;-- TMR0 Overflow
    #define U1_INT      PIR3, U1IF      ; -- USART1
    #define U1E_INT     PIR3, U1EIF     ; -- USART1
    #define U1TX_INT    PIR3, U1TXIF    ;-- USART1 Transmit 
    #define U1RX_INT    PIR3, U1RXIF    ;-- USART1 Receive 
    #define I2C1E_INT   PIR3, I2C1EIF   ;
    #define I2C1_INT    PIR3, I2C1IF    ;
    #define I2C1TX_INT  PIR3, I2C1TXIF ;
; -=-   PIR4 Interrupts: -=-
    #define INT1_INT    PIR4, INT1IF    ;-- INT1 External
    #define CLC1_INT    PIR4, CLC1IF    ;
    #define CWG1_INT    PIR4, CWG1IF    ;
    #define NCO1_INT    PIR4, NCO1IF    ;
    #define CCP1_INT    PIR4, CCP1IF    ;-- CCP1 
    #define TMR2_INT    PIR4, TMR2IF    ;-- TMR2 to PR2 Match 
    #define TMR1G_INT   PIR4, TMR1GIF   ;-- TMR1 Gate
    #define TMR1_INT    PIR4, TMR1IF    ;-- TMR1 Overflow 
; -=-   PIR5 Interrupts: -=-
    #define IRX_INT     PIR5, IRXIF     ;-- CAN
    #define WAK_INT     PIR5, WAKIF     ;-- CAN
    #define ERR_INT     PIR5, ERRIF     ;-- CAN
    #define TXB2_INT    PIR5, TXB2IF    ;-- CAN
    #define TXB1_INT    PIR5, TXB1IF    ;-- CAN
    #define TXB0_INT    PIR5, TXB0IF    ;-- CAN
    #define RXB1_INT    PIR5, RXB1IF    ;-- CAN
    #define RXB0_INT    PIR5, RXB0IF    ;-- CAN
; -=-   PIR6 Interrupts: -=-
    #define DMA2A_INT     PIR6, DMA2AIF   ;
    #define DMA2OR_INT    PIR6, DMA2ORIF  ;
    #define DMA2DCNT_INT  PIR6, DMA2DCNTIF ;
    #define DMA2SCNT_INT  PIR6, DMA2SCNTIF ;
    #define SMT2PWA_INT   PIR6, SMT2PWAIF  ;
    #define SMT2PRA_INT   PIR6, SMT2PRAIF  ;
    #define SMT2_INT      PIR6, SMT2IF     ;
    #define C2_INT        PIR6, C2IF      ;-- Comparator2 
; -=-   PIR7 Interrupts: -=-
    #define U2_INT      PIR7, U2IF      ; -- USART2 
    #define U2E_INT     PIR7, U2EIF     ; -- USART2 
    #define U2TX_INT    PIR7, U2TXIF    ;-- USART2 Transmit
    #define U2RX_INT    PIR7, U2RXIF    ;-- USART2 Receive
    #define I2C2E_INT   PIR7, I2C2EIF   ;
    #define I2C2_INT    PIR7, I2C2IF    ;
    #define I2C2TX_INT  PIR7, I2C2TXIF  ;
    #define I2C2RX_INT  PIR7, I2C2RXIF  ;
; -=-   PIR8 Interrupts: -=-
    #define TMR5_INT    PIR8, TMR5IF    ;-- TMR5 Overflow
    #define INT2_INT    PIR8, INT2IF    ;-- INT2 External 
    #define CLC2_INT    PIR8, CLC2IF    ;
    #define CWG2_INT    PIR8, CWG2IF    ;
    #define CCP2_INT    PIR8, CCP2IF    ;-- CCP2 
    #define TMR4_INT    PIR8, TMR4IF    ;-- TMR4 Overflow
    #define TMR3G_INT   PIR8, TMR3GIF   ;-- TMR3 Gate
    #define TMR3_INT    PIR8, TMR3IF    ;-- TMR3 Overflow 
; -=-   PIR9 Interrupts: -=-
    ; --bit 7 unused--
    #define CLC4_INT    PIR9, CLC4IF    ;
    #define CCP4_INT    PIR9, CCP4IF    ;-- CCP4 
    #define CLC3_INT    PIR9, CLC3IF    ;
    #define CWG3_INT    PIR9, CWG3IF    ;
    #define CCP3_INT    PIR9, CCP3IF    ;-- CCP3 
    #define TMR6_INT    PIR9, TMR6IF    ;-- TMR6 Overflow
    #define TMR5G_INT   PIR8, TMR5GIF   ;-- TMR5 Gate
    list
  ENDASM
#endif

;-- GetIntInfo (**device specific**)
#if (DT_INTS_18XV_FAMILY = "K83")
  ASM
  GetIntInfo  macro  IflagReg, IflagBit
    nolist
    INT_Flag_Reg = -1
    INT_Flag_Bit = -1
    INT_Enable_Reg = -1
    INT_Enable_Bit = -1
    ifdef  IPR1
      INT_Priority_Reg = -1     ; cannot change priority (ie INT0_INT on some devices)
      INT_Priority_Bit = -1
    endif
    IntFound = 0
  
; -=- 18FxxK83 family -=-
; -=-   PIR0 Interrupts: -=-
    ifdef IOCIF   ;----{ Port Change Interrupt }-----------[PIR0, IOCIF]---
      INT_Source  PIR0,IOCIF, PIE0,IOCIE, IPR0,IOCIP
    endif
    ifdef CRCIF  ;----{ Cyclic Redundancy Check Interrupt Flag }-----[PIR0, CRCIF]---
      INT_Source  PIR0,CRCIF, PIE0,CRCIE, IPR0,CRCIP
    endif
    ifdef SCANIF  ;----{ SCAN Interrupt Flag }-----------[PIR0, SCANIF]---
      INT_Source  PIR0,SCANIF, PIE0,SCANIE, IPR0,SCANIP
    endif
    ifdef NVMIF    ;----{ EEPROM/FLASH Write Operation Interrupt } [PIR0, NVMIF]---
      INT_Source  PIR0,NVMIF, PIE0,NVMIE, IPR0,NVMIP
    endif
    ifdef CSWIF  ;----{ Clock Switch Interrupt Flag }-----------[PIR0, CSWIF]---
      INT_Source  PIR0,CSWIF, PIE0,CSWIE, IPR0,CSWIP
    endif
    ifdef OSFIF  ;----{ Osc Fail Interrupt Flag }-----------[PIR0, OSFIF]---
      INT_Source  PIR0,OSFIF, PIE0,OSFIE, IPR0,OSFIP
    endif
    ifdef HLVDIF  ;----{ High/Low Voltage Detect Interrupt }--[PIR0, HLVDIF]---
      INT_Source  PIR0,HLVDIF, PIE0,HLVDIE, IPR0,HLVDIP
    endif
    ifdef SWIF  ;----{ Software Interrupt Flag }-----------[PIR0, SWIF]---
      INT_Source  PIR0,SWIF, PIE0,SWIE, IPR0,SWIP
    endif
; -=-   PIR1 Interrupts: -=-
    ifdef SMT1PWAIF  ;----{ SMT 1 Pulse-Width Acquisition Interrupt Flag }----[PIR1, SMT1PWAIF]---
      INT_Source  PIR1,SMT1PWAIF, PIE1,SMT1PWAIE, IPR1,SMT1PWAIP
    endif
    ifdef SMT1PRAIF  ;----{ SMT 1 Period Acquisition Interrupt Flag }----[PIR1, SMT1PRAIF]---
      INT_Source  PIR1,SMT1PRAIF, PIE1,SMT1PRAIE, IPR1,SMT1PRAIP
    endif
    ifdef SMT1IF  ;----{ Signal Measurement Timer 1 Interrupt Flag }----[PIR1, SMT1IF]---
      INT_Source  PIR1,SMT1IF, PIE1,SMT1IE, IPR1,SMT1IP
    endif
    ifdef C1IF    ;----{ Comparator 1 Interrupt }-----------------[PIR1, C1IF]---
      INT_Source  PIR1,C1IF, PIE1,C1IE, IPR1,C1IP
    endif
    ifdef ADTIF    ;----{ A/D Threshold Interrupt }--------------[PIR1, ADTIF]---
      INT_Source  PIR1,ADTIF, PIE1,ADTIE, IPR1,ADTIP
    endif
    ifdef ADIF    ;----{ A/D Converter Interrupt }--------------[PIR1, ADIF]---
      INT_Source  PIR1,ADIF, PIE1,ADIE, IPR1,ADIP
    endif
    ifdef ZCDIF  ;----{ Zero Cross Detect Flag }-----------[PIR1, ZCDIF]---
      INT_Source  PIR1,ZCDIF, PIE1,ZCDIE, IPR1,ZCDIP
    endif
    ifdef INT0IF  ;----{ INT0 External Interrupt }----------[PIR1, INT0IF]---
      INT_Source  PIR1,INT0IF, PIE1,INT0IE, IPR1, INT0IP
    endif
; -=-   PIR2 Interrupts: -=-
    ifdef I2C1RXIF   ;----{ I2C 1 Receive Interrupt }---[PIR2, I2C1RXIF]---
      INT_Source  PIR2,I2C1RXIF, PIE2,I2C1RXIE, IPR2,I2C1RXIP
    endif
    ifdef SPI1IF   ;----{ Synchronous Serial Port 1 Interrupt }---[PIR2, SPI1IF]---
      INT_Source  PIR2,SPI1IF, PIE2,SPI1IE, IPR2,SPI1IP
    endif
    ifdef SPI1TXIF   ;----{ SPI Transmit Interrupt }---[PIR2, SPI1TXIF]---
      INT_Source  PIR2,SPI1TXIF, PIE2,SPI1TXIE, IPR2,SPI1TXIP
    endif
    ifdef SPI1RXIF   ;----{ SPI Receive Interrupt }---[PIR2, SPI1RXIF]---
      INT_Source  PIR2,SPI1RXIF, PIE2,SPI1RXIE, IPR2,SPI1RXIP
    endif
    ifdef DMA1AIF    ;----{ Direct Memory Access 1 Interrupt } [PIR2, DMA1AIF]---
      INT_Source  PIR2,DMA1AIF, PIE2,DMA1AIE, IPR2,DMA1AIP
    endif
    ifdef DMA1ORIF    ;----{ DMA 1 Over Run Interrupt } [PIR2, DMA1ORIF]---
      INT_Source  PIR2,DMA1ORIF, PIE2,DMA1ORIE, IPR2,DMA1ORIP
    endif
    ifdef DMA1DCNTIF    ;----{ DMA 1 Designation Count Interrupt } [PIR2, DMA1DCNTIF]---
      INT_Source  PIR2,DMA1DCNTIF, PIE2,DMA1DCNTIE, IPR2,DMA1DCNTIP
    endif
    ifdef DMA1SCNTIF    ;----{ DMA 1 Source Count Interrupt } [PIR2, DMA1SCNTIF]---
      INT_Source  PIR2,DMA1SCNTIF, PIE2,DMA1SCNTIE, IPR2,DMA1SCNTIP
    endif
; -=-   PIR3 Interrupts: -=-
    ifdef TMR0IF  ;----{ TMR0 Overflow Interrupt }----------[PIR3, TMR0IF]---
      INT_Source  PIR3,TMR0IF, PIE3,TMR0IE, IPR3,TMR0IP
    endif
    ifdef U1IF    ;----{ UART1 Interrupt }-------------[PIR3, U1IF]---
      INT_Source  PIR3,U1IF, PIE3,U1IE, IPR3,U1IP
    endif
    ifdef U1EIF    ;----{ UART1 Framing Error Interrupt }-------------[PIR3, U1EIF]---
      INT_Source  PIR3,U1EIF, PIE3,U1EIE, IPR3,U1EIP
    endif
    ifdef U1TXIF  ;----{ USART1 Transmit Interrupt }-------------[PIR3, U1TXIF]---
      INT_Source  PIR3,U1TXIF, PIE3,U1TXIE, IPR3,U1TXIP
    endif
    ifdef U1RXIF   ;----{ USART1 Receive Interrupt }------------[PIR3, U1RXIF]---
      INT_Source  PIR3,U1RXIF, PIE3,U1RXIE, IPR3,U1RXIP
    endif
    ifdef I2C1EIF   ;----{ I2C 1 Error Interrupt }---[PIR3, I2C1EIF]---
      INT_Source  PIR3,I2C1EIF, PIE3,I2C1EIE, IPR3,I2C1EIP
    endif
    ifdef I2C1IF   ;----{ I2C 1 Interrupt }---[PIR3, I2C1IF]---
      INT_Source  PIR3,I2C1IF, PIE3,I2C1IE, IPR3,I2C1IP
    endif
    ifdef I2C1TXIF   ;----{ I2C 1 Transmit Interrupt }---[PIR3, I2C1TXIF]---
      INT_Source  PIR3,I2C1TXIF, PIE3,I2C1TXIE, IPR3,I2C1TXIP
    endif
; -=-   PIR4 Interrupts: -=-
    ifdef INT1IF  ;----{ INT1 External Interrupt }---------[PIR4, INT1IF]---
      INT_Source  PIR4,INT1IF, PIE4,INT1IE, IPR4,INT1IP
    endif
    ifdef CLC1IF  ;----{ Configurable Logic Cell 1 Flag }-----------[PIR4, CLC1IF]---
      INT_Source  PIR4,CLC1IF, PIE4,CLC1IE, IPR4,CLC1IP
    endif
    ifdef CWG1IF  ;----{ Complimentary Waveform Generagor 1 Flag }----[PIR4, CWG1IF]---
      INT_Source  PIR4,CWG1IF, PIE4,CWG1IE, IPR4,CWG1IP
    endif
    ifdef NCO1IF  ;----{ CNumerically Controlled Oscillator 1 Flag }----[PIR4, NCO1IF]---
      INT_Source  PIR4,NCO1IF, PIE4,NCO1IE, IPR4,NCO1IP
    endif
    ifdef CCP1IF  ;----{ CCP1 Interrupt }---------------------[PIR4, CCP1IF]---
      INT_Source  PIR4,CCP1IF, PIE4,CCP1IE, IPR4,CCP1IP
    endif
    ifdef TMR2IF  ;----{ TMR2 to PR2 Match Interrupt }--------[PIR4, TMR2IF]---
      INT_Source  PIR4,TMR2IF, PIE4,TMR2IE, IPR4,TMR2IP
    endif
    ifdef TMR1GIF  ;----{ TMR1 Gate Interrupt }------------[PIR4, TMR1GIF]---
      INT_Source  PIR4,TMR1GIF, PIE4,TMR1GIE, IPR4,TMR1GIP
    endif
    ifdef TMR1IF  ;----{ TMR1 Overflow Interrupt }------------[PIR4, TMR1IF]---
      INT_Source  PIR4,TMR1IF, PIE4,TMR1IE, IPR4,TMR1IP
    endif
; -=-   PIR5 Interrupts: -=-
    ifdef IRXIF      ;----{ CAN Interrupt }
      INT_Source  PIR5,IRXIF, PIE5,IRXIE, IPR5,IRXIP
    endif
    ifdef WAKIF      ;----{ CAN Interrupt }
      INT_Source  PIR5,WAKIF, PIE5,WAKIE, IPR5,WAKIP
    endif
    ifdef ERRIF      ;----{ CAN Interrupt }
      INT_Source  PIR5,ERRIF, PIE5,ERRIE, IPR5,ERRIP
    endif
    ifdef TXB2IF      ;----{ CAN Interrupt }
      INT_Source  PIR5,TXB2IF, PIE5,TXB2IE, IPR5,TXB2IP
    endif
    ifdef TXB1IF      ;----{ CAN Interrupt }
      INT_Source  PIR5,TXB1IF, PIE5,TXB1IE, IPR5,TXB1IP
    endif
    ifdef TXB0IF      ;----{ CAN Interrupt }
      INT_Source  PIR5,TXB0IF, PIE5,TXB0IE, IPR5,TXB0IP
    endif
    ifdef RXB1IF      ;----{ CAN Interrupt }
      INT_Source  PIR5,RXB1IF, PIE5,RXB1IE, IPR5,RXB1IP
    endif
    ifdef RXB0IF      ;----{ CAN Interrupt }
      INT_Source  PIR5,RXB0IF, PIE5,RXB0IE, IPR5,RXB0IP
    endif
; -=-   PIR6 Interrupts: -=-
    ifdef DMA2AIF    ;----{ Direct Memory Access 2 Interrupt } [PIR6, DMA2AIF]---
      INT_Source  PIR6,DMA2AIF, PIE6,DMA2AIE, IPR6,DMA2AIP
    endif
    ifdef DMA2ORIF    ;----{ DMA 2 Over Run Interrupt } [PIR6, DMA2ORIF]---
      INT_Source  PIR6,DMA2ORIF, PIE6,DMA2ORIE, IPR6,DMA2ORIP
    endif
    ifdef DMA2DCNTIF    ;----{ DMA 2 Designation Count Interrupt } [PIR6, DMA2DCNTIF]---
      INT_Source  PIR6,DMA2DCNTIF, PIE6,DMA2DCNTIE, IPR6,DMA2DCNTIP
    endif
    ifdef DMA2SCNTIF    ;----{ DMA 2 Source Count Interrupt } [PIR6, DMA2SCNTIF]---
      INT_Source  PIR6,DMA2SCNTIF, PIE6,DMA2SCNTIE, IPR6,DMA2SCNTIP
    endif
    ifdef SMT2PWAIF  ;----{ SMT 2 Pulse-Width Acquisition Interrupt Flag }----[PIR6, SMT2PWAIF]---
      INT_Source  PIR6,SMT2PWAIF, PIE6,SMT2PWAIE, IPR6,SMT2PWAIP
    endif
    ifdef SMT2PRAIF  ;----{ SMT 2 Period Acquisition Interrupt Flag }----[PIR6, SMT2PRAIF]---
      INT_Source  PIR6,SMT2PRAIF, PIE6,SMT2PRAIE, IPR6,SMT2PRAIP
    endif
    ifdef SMT2IF  ;----{ Signal Measurement Timer 2 Interrupt Flag }----[PIR6, SMT2IF]---
      INT_Source  PIR6,SMT2IF, PIE6,SMT2IE, IPR6,SMT2IP
    endif
    ifdef C2IF    ;----{ Comparator 2 Interrupt }-----------------[PIR6, C2IF]---
      INT_Source  PIR6,C2IF, PIE6,C2IE, IPR6,C2IP
    endif
; -=-   PIR7 Interrupts: -=-
    ifdef U2IF   ;----{ UART2 Interrupt }-----------[PIR7, U2IF]---
      INT_Source  PIR7,U2IF, PIE7,U2IE, IPR7,U2IP
    endif
    ifdef U2EIF   ;----{ UART2 Framing Error Interrupt }-----------[PIR7, U2EIF]---
      INT_Source  PIR7,U2EIF, PIE7,U2EIE, IPR7,U2EIP
    endif
    ifdef U2TXIF   ;----{ USART2 Transmit Interrupt }-----------[PIR7, U2TXIF]---
      INT_Source  PIR7,U2TXIF, PIE7,U2TXIE, IPR7,U2TXIP
    endif
    ifdef U2RXIF   ;----{ USART2 Receive Interrupt }------------[PIR7, U2RXIF]---
      INT_Source  PIR7,U2RXIF, PIE7,U2RXIE, IPR7,U2RXIP
    endif
    ifdef I2C2EIF   ;----{ I2C 2 Error Interrupt }---[PIR7, I2C2EIF]---
      INT_Source  PIR7,I2C2EIF, PIE7,I2C2EIE, IPR7,I2C2EIP
    endif  
    ifdef I2C2IF   ;----{ I2C 2 Interrupt }---[PIR7, I2C2IF]---
      INT_Source  PIR7,I2C2IF, PIE7,I2C2IE, IPR7,I2C2IP
    endif
    ifdef I2C2TXIF   ;----{ I2C 2 Transmit Interrupt }---[PIR7, I2C2TXIF]---
      INT_Source  PIR7,I2C2TXIF, PIE7,I2C2TXIE, IPR7,I2C2TXIP
    endif
    ifdef I2C2RXIF   ;----{ I2C 2 Receive Interrupt }---[PIR7, I2C2RXIF]---
      INT_Source  PIR7,I2C2RXIF, PIE7,I2C2RXIE, IPR7,I2C2RXIP
    endif
; -=-   PIR8 Interrupts: -=-
    ifdef TMR5IF  ;----{ TMR5 Overflow Interrupt }------------[PIR8, TMR5IF]---
      INT_Source  PIR8,TMR5IF, PIE8,TMR5IE, IPR8,TMR5IP
    endif
    ifdef INT2IF  ;----{ INT2 External Interrupt }---------[PIR8, INT2IF]---
      INT_Source  PIR8,INT2IF, PIE8,INT2IE, IPR8,INT2IP
    endif
    ifdef CLC2IF  ;----{ Configurable Logic Cell 2 Flag }-----------[PIR8, CLC2IF]---
      INT_Source  PIR8,CLC2IF, PIE8,CLC2IE, IPR8,CLC2IP
    endif
    ifdef CWG2IF  ;----{ Complimentary Waveform Generagor 2 Flag }----[PIR8, CWG2IF]---
      INT_Source  PIR8,CWG2IF, PIE8,CWG2IE, IPR8,CWG2IP
    endif
    ifdef CCP2IF  ;----{ CCP2 Interrupt Flag }----------------[PIR8, CCP2IF]---
      INT_Source  PIR8,CCP2IF, PIE8,CCP2IE, IPR8,CCP2IP
    endif
    ifdef TMR4IF  ;----{ TMR4 Overflow Interrupt }------------[PIR8, TMR4IF]---
      INT_Source  PIR8,TMR4IF, PIE8,TMR4IE, IPR8,TMR4IP
    endif
    ifdef TMR3GIF  ;----{ TMR3 Gate Interrupt }------------[PIR8, TMR3GIF]---
      INT_Source  PIR8,TMR3GIF, PIE8,TMR3GIE, IPR8,TMR3GIP
    endif
    ifdef TMR3IF  ;----{ TMR3 Overflow Interrupt }------------[PIR8, TMR3IF]---
      INT_Source  PIR8,TMR3IF, PIE8,TMR3IE, IPR8,TMR3IP
    endif
; -=-   PIR9 Interrupts: -=-
    ifdef CLC4IF  ;----{ Configurable Logic Cell 4 Flag }-----------[PIR9, CLC4IF]---
      INT_Source  PIR9,CLC4IF, PIE9,CLC4IE, IPR9,CLC4IP
    endif
    ifdef CCP4IF  ;----{ CCP4 Interrupt Flag }----------------[PIR9, CCP4IF]---
      INT_Source  PIR9,CCP4IF, PIE9,CCP4IE, IPR9,CCP4IP
    endif
    ifdef CLC3IF  ;----{ Configurable Logic Cell 3 Flag }-----------[PIR9, CLC3IF]---
      INT_Source  PIR9,CLC3IF, PIE9,CLC3IE, IPR9,CLC3IP
    endif
    ifdef CWG3IF  ;----{ Complimentary Waveform Generagor 3 Flag }----[PIR9, CWG3IF]---
      INT_Source  PIR9,CWG3IF, PIE9,CWG3IE, IPR9,CWG3IP
    endif
    ifdef CCP3IF  ;----{ CCP3 Interrupt Flag }----------------[PIR9, CCP3IF]---
      INT_Source  PIR9,CCP3IF, PIE9,CCP3IE, IPR9,CCP3IP
    endif
    ifdef TMR6IF  ;----{ TMR6 Overflow Interrupt }------------[PIR9, TMR6IF]---
      INT_Source  PIR9,TMR6IF, PIE9,TMR6IE, IPR9,TMR6IP
    endif
    ifdef TMR5GIF  ;----{ TMR5 Gate Interrupt }------------[PIR9, TMR5GIF]---
      INT_Source  PIR9,TMR5GIF, PIE9,TMR5GIE, IPR9,TMR5GIP
    endif
    list
  endm
  ENDASM
#endif      ' (DT_INTS_18XV_FAMILY = "K83")

' to add support for other devices, copy the two sections above and edit them
' to match your device
;----------------------------------------------------------------------------
;---- ** end of device specific section **-----------------------------------
;----------------------------------------------------------------------------

;----[if not using Low Priority INTS, create a dummy handler]----------------
  ASM
  ifndef  USE_LOWPRIORITY
    INT_ENTRY_L:
      retfie 1
  else
    if (USE_LOWPRIORITY != 1)
      INT_ENTRY_L:
        retfie 1
    endif
  endif
  ENDASM
  
;---[stay compatible with the 14-bit version]---------------------------------
  ASM
  INT_FINISH_H   macro
  endm
  ENDASM
  
;---[create the High Priority interrupt processor]----------------------------
  ASM
  INT_CREATE_H  macro
    list
    local OverCREATE
    goto OverCREATE
    Priority = H    

  ; high-priority intr entry point
  INT_ENTRY_H:  
    SAVE_TABLEPTR H                 ; save HP TBLPTR regs
    bsf      _InHPint, 0
  List_Start_H:
    bcf      _Serviced_H, 0         ; clear serviced flag
    RESET_BANK
    ifdef INT_LIST_H
      INT_LIST_H                    ; expand the users list of HP INT handlers
    else
      ifdef INT_LIST
        INT_LIST                    ; expand the 16F Compatible List
      else
        error "INT_CREATE_H - INT_LIST or INT_LIST_H not found"
      endif
    endif
    btfsc    _Serviced_H, 0         ; if anything was serviced
    goto     List_Start_H           ; go around, and check again
  
    ifdef REENTERPBP_18XV           ; is ReEnterPBP-18xv included
      call _RestorePBP_H            ; restore HP PBP system vars saved by INT_Handler
    endif
  
  INT_EXIT_H:
    bcf      _InHPint, 0
    RESTORE_TABLEPTR H              ; restore HP TBLPTR regs
    RESET_BANK                      ; set PREV_BANK = 0 (BSR will be over-written by retfie)
    retfie    1                     ; return from interrupt (restores context from shadow registers)
  
  OverCREATE:
    INTERRUPT_ENABLE 1              ; enable interrupts
  endm
  
;---[stay compatible with the 14-bit version]---------------------------------
  INT_CREATE  macro
    INT_CREATE_H
  endm
  ENDASM
  
;---[create the Low Priority interrupt processor]-----------------------------
  ASM
  INT_CREATE_L  macro
    list
    local OverCREATE
    goto OverCREATE
    ifdef USE_LOWPRIORITY
      if (USE_LOWPRIORITY != 1)
        error "'DEFINE USE_LOWPRIORITY 1' required for Low Priority Interrupts"
      endif
    else
      error "'DEFINE USE_LOWPRIORITY 1' required for Low Priority Interrupts"
    endif
    Priority = L

  ; low-priority intr entry point
  INT_ENTRY_L:
    SAVE_TABLEPTR L                 ; save LP TBLPTR regs
    bcf      _InHPint, 0
  List_Start_L:
    bcf      _Serviced_L, 0
    RESET_BANK
    ifdef INT_LIST_L
      INT_LIST_L                    ; expand the users list of HP INT handlers
    else
      error "INT_CREATE_L - INT_LIST_L not defined, can not create"
    endif
    btfsc    _Serviced_L, 0         ; if anything was serviced
    goto     List_Start_L           ; go around, and check again
  
    ifdef REENTERPBP_18XVLP         ; is ReEnterPBP-18xvLPs included
      call _RestorePBP_L            ; restore LP PBP system Vars
    endif
  
  INT_EXIT_L:
    RESTORE_TABLEPTR L              ; restore LP TBLPTR regs
    RESET_BANK                      ; set PREV_BANK = 0 (BSR will be over-written by retfie)
    retfie    1                     ; return from interrupt (restores context from shadow registers)

  OverCREATE:
    INTERRUPT_ENABLE 0              ; enable Low Priority interrupts
  endm
  ENDASM
  
  ASM
;---[find assigned priority for specified INT Source]------------------------
  INT_Count = 0     ; count of INT_Handlers added

  FindIntPriority  macro IntFlagReg, IntFlagBit
    local LoopCount
    nolist
    PrFound = 0
    LoopCount = 1
    while (LoopCount <= INT_Count)
      if (IntFlagReg == PrList#v(LoopCount)R)
        if (IntFlagBit == PrList#v(LoopCount)B)
          list
          Priority = PrList#v(LoopCount)Priority
          PrFound = 1
          LoopCount = INT_Count     ; exit loop
        endif
      endif
      LoopCount += 1
    endw
  endm
  ENDASM
  
  ASM
;---[add an interrupt source to the user's list of INT Handlers]--------------
  INT_Handler  macro  IntFlagReg, IntFlagBit, IntHandlerLabel, IntType, ResetIF
    list
    local AfterSave, AfterUser, NoInt
    INT_Count += 1
    PrList#v(INT_Count)R = IntFlagReg
    PrList#v(INT_Count)B = IntFlagBit
    PrList#v(INT_Count)Priority = Priority
    GetIntInfo   IntFlagReg, IntFlagBit

    if (IntFound == YES)
      SET_BANK INT_Enable_Reg
      btfss    INT_Enable_Reg, INT_Enable_Bit     ; if INT IE is enabled
      goto     NoInt
      btfss    INT_Flag_Reg, INT_Flag_Bit         ; and the IF flag set?
      goto     NoInt
      if (Priority == H)
        bsf      _Serviced_H, 0
      else
        bsf      _Serviced_L, 0
      endif
      ifdef NO_CLRWDT
        if (NO_CLRWDT != 1)
          CLRWDT
        endif
      else
        CLRWDT
      endif
  
      if (IntType == PBP)               ; if INT handler is PBP
        if (Priority == H)
          ifdef REENTERPBP_18XV
            call  _SavePBP_H            ; save PBP system vars for HP INT
          else
            error "ReEnterPBP-18xv must be INCLUDEd to use High Priority PBP interrupts"
          endif
        else    ; Priority = L
          ifdef REENTERPBP_18XVLP
            call  _SavePBP_L            ; save PBP system vars for LP INT
          else
            error "ReEnterPBP-18xvLP must be INCLUDEd to use Low Priority PBP interrupts"
          endif
        endif
      endif
  
    AfterSave:
      call  IntHandlerLabel             ; call the users INT handler
       
    AfterUser:
      if (ResetIF == YES)               ; reset interrupt IF flag (if specified)
        SET_BANK INT_Flag_Reg
        bcf      INT_Flag_Reg, INT_Flag_Bit 
      endif
    else
      error  "INT_HANDLER - Interrupt Source not found"
    endif

  NoInt:
  endm
  ENDASM
  
  ASM
;---[returns from an interrupt subroutine]-----------------------------------
  INT_RETURN  macro
    RESET_BANK
    return
  endm    

;---[set interrupt source priority]-------------------------------------------
  INT_PRIORITY  macro  IntFlagReg, IntFlagBit
    GetIntInfo   IntFlagReg, IntFlagBit
    if (IntFound == YES)
      FindIntPriority  IntFlagReg, IntFlagBit
      if (PrFound == 1)
        if (INT_Priority_Reg != -1)
          if (Priority == H)
            SET_BANK INT_Priority_Reg  
            bsf  INT_Priority_Reg, INT_Priority_Bit
          else
            if (Priority == L)
              SET_BANK INT_Priority_Reg  
              bcf  INT_Priority_Reg, INT_Priority_Bit
            else
              error "INT_PRIORITY - Invalid Priority Specified"
            endif
          endif
        else
          if (Priority == L)
            error "INT0_INT can NOT be assigned to Low Priority"
          endif
        endif
      else
        error "INT_PRIORITY - Priority State Not Found"
      endif
    else
        error  "INT_PRIORITY - Interrupt Source not found"
    endif
  endm
  

;---[enable an interrupt source]----------------------------------------------
  INT_ENABLE  macro  IntFlagReg, IntFlagBit
    GetIntInfo   IntFlagReg, IntFlagBit
    if (IntFound == YES)
      INT_PRIORITY IntFlagReg, IntFlagBit          ; set IPR int priority reg
      SET_BANK INT_Flag_Reg 
      bcf     INT_Flag_Reg, INT_Flag_Bit           ; clear the int IF flag first 
      bsf     INT_Enable_Reg, INT_Enable_Bit       ; and enable the INT source  
    else
        error  "INT_ENABLE - Interrupt Source not found"
    endif
  endm
  
;---[disable an interrupt source]---------------------------------------------
  INT_DISABLE  macro  IntFlagReg, IntFlagBit
    GetIntInfo   IntFlagReg, IntFlagBit
    if (IntFound == YES)
      SET_BANK INT_Enable_Reg 
      bcf     INT_Enable_Reg, INT_Enable_Bit      ; disable the INT source  
    else
      error "INT_DISABLE - Interrupt Source not found"
    endif
  endm
  
;---[clear an interrupt flag]-------------------------------------------------
; note: not all interrupt IF bits are read-write... some are read-only
  INT_CLEAR  macro  IntFlagReg, IntFlagBit
    GetIntInfo   IntFlagReg, IntFlagBit
    if (IntFound == YES)
      SET_BANK INT_Flag_Reg 
      bcf     INT_Flag_Reg, INT_Flag_Bit          ; clear the IF INT flag
    else
      error "INT_CLEAR - Interrupt Source not found"
    endif
  endm

;---[set an interrupt flag/generate software interrupt]-----------------------
; note: not all interrupt IF bits are read-write... some are read-only
  INT_SET  macro  IntFlagReg, IntFlagBit
    GetIntInfo   IntFlagReg, IntFlagBit
    if (IntFound == YES)
      SET_BANK INT_Flag_Reg 
      bsf     INT_Flag_Reg, INT_Flag_Bit          ; set the IF INT flag
    else
      error "INT_SET - Interrupt Source not found"
    endif
  endm

  ENDASM
  
  
;---[module end]--------------------------------------------------------------
  OVER_DT_INTS_18:
  ENABLE DEBUG
  
