;*****************************************************************************
;*  Name    : umc_loader_k40.asm                                             *
;*  Author  : David John Barker                                              *
;*  Date    : 04/08/2011                                                     *
;*  Version : 1.0 - Initial release                                          *
;*          : BETA CHANGES                                                   *
;*          : 26-5-2018 - version for 18FxxK40                               *
;*                      - add TEST_UART define                               *
;*                      - add program size check                             *
;*                      - add RCSTA_VAL                                      *
;*          : 26-08-2011 - check for TXSTA1 rather than TXSTA                *
;*          : 26-08-2011 - added SPBRG_ and SPBRGH_                          *
;*          : 17-08-2011 - check for EEPGD                                   *
;*                       - added #ifdef ENABLE_EEPROM_READ                   *
;*          : 10-08-2011 - BAUDCON_, BRG16                                   *
;*          :            - TXSTA_, TRMT                                      * 
;*          : 15-08-2011 - clrf TBLPTRU                                      *
;*  Notes   : Parts of this code based around the Tiny Bootloader            *
;*          : claudiu.chiculita@ugal.ro                                      *
;*          : http://www.etc.ugal.ro/cchiculita/software/picbootloader.htm   *  
;*****************************************************************************
   radix                              DEC ; default radix is DECIMAL

;=============================================================================
; USER CONFIGURATION SECTION
;=============================================================================
   processor                        18F27K40
   #define DEVICE_CLOCK             16000000
   #define BAUDRATE                 115200

; #define TEST_UART adds a poweron uart check routine that you can use
; to verify that the uart is working properly. it sends an initial "U>" sequence
; and then loops echoing received chars. test with a serial terminal emulator.
;#define TEST_UART 1        ; uncomment to add uart test code
#ifdef TEST_UART
  messg "UART test code enabled"
#endif

; enable EEPROM READ
#define ENABLE_EEPROM_READ 1

;
; insert any device specific information inside here - for example, some 
; devices (such as the 18F1220) the USART pins default to analog and so must be 
; set to digital for comms to work
;
; the following assumes using uart1 with PORTC RC6(TX output) and RC7(RX input) pins
#define TX_PIN_   6
#define RX_PIN_   7
#define PPSLOCKED_ 0  ; PPSLOCK bit 0
#define GIE_  7       ; INTCON bit 7

UserConfig macro
   ; setup K40 internal osc
   banksel OSCCON1
   movlw  0x60               ; NOSC=110 (HFINTOSC), NDIV=1:1       
   movwf  OSCCON1
  #if DEVICE_CLOCK == 8000000
   movlw  0x03               ; 8MHz
   movwf  OSCFRQ
  #else 
  #if DEVICE_CLOCK == 16000000
   movlw  0x05               ; 16MHz
   movwf  OSCFRQ
  #else
  #if DEVICE_CLOCK == 32000000
   movlw  0x06               ; 32MHz
   movwf  OSCFRQ
  #else
  #if DEVICE_CLOCK == 64000000
   movlw  0x08               ; 64MHz
   movwf  OSCFRQ
  #else
   error "unsupported HFINTOSC frequency"
  #endif
  #endif
  #endif
  #endif

   ; set TX and RX pins to digital mode (ANC6, ANC7 = 0)
   banksel ANSELC
   bcf ANSELC, RX_PIN_        ; RC6/TX
   bcf ANSELC, TX_PIN_        ; RC7/RX
   
   ; set normal slewrate for RC6/RC7 (SLRCx = 0)
   banksel SLRCONC
   bcf SLRCONC, RX_PIN_        ; RC6/TX
   bcf SLRCONC, TX_PIN_        ; RC7/RX
   
   ; set uart TX and RX pin directions (these will over-ridden by the usart as req'd)
   banksel TRISC
   bsf TRISC, TX_PIN_         ; RC6/TX output
   bsf TRISC, RX_PIN_         ; RC7/RX input

   ; assign pps
   ; first, make sure PPSLOCK is unlocked (just in case)
   bcf  INTCON, GIE_
   banksel  PPSLOCK
   movlw  0x55
   movwf  PPSLOCK
   movlw  0xAA
   movwf  PPSLOCK
   bcf    PPSLOCK, PPSLOCKED_
   ; now assign uart1 pps pins
   movlw  0x17        ; RX1PPS=0x17 - select PORTC<7> as EUSART1 RX input
   movwf  RX1PPS
   movlw  0x09        ; RC6PPS=0x09 - select PORTC<6> as EUSART1 TX output
   movwf  RC6PPS
   
   banksel 0
   endm

;
; K40 configuration fuse setup
;
  config FEXTOSC = OFF       ; Oscillator not enabled
  config RSTOSC = HFINTOSC_1MHZ ; HFINTOSC with HFFRQ = 4 MHz and CDIV = 4:1
  config CLKOUTEN = OFF      ; CLKOUT function is disabled
  config CSWEN = ON          ; Writing to NOSC and NDIV is allowed
  config FCMEN = OFF         ; Fail-Safe Clock Monitor disabled
  config MCLRE = EXTMCLR     ; If LVP = 0  MCLR pin is MCLR; If LVP = 1  RE3 pin function is MCLR 
  config PWRTE = ON          ; Power up timer enabled
  config LPBOREN = OFF       ; ULPBOR disabled
  config BOREN = ON          ; Brown-out Reset enabled according to SBOREN
  config BORV = VBOR_270     ; Brown-out Reset Voltage (VBOR) set to 2.70V
  config ZCD = OFF           ; ZCD disabled. ZCD can be enabled by setting the ZCDSEN bit of ZCDCON
  config PPS1WAY = OFF       ; PPSLOCK bit can be set and cleared repeatedly (subject to the unlock sequence)
  config STVREN = ON         ; Stack full/underflow will cause Reset
  config DEBUG = OFF         ; Background debugger disabled
  config XINST = OFF         ; Extended Instruction Set and Indexed Addressing Mode disabled
  config WDTCPS = WDTCPS_31  ; Divider ratio 1:65536; software control of WDTPS
  config WDTE = OFF           ; WDT disabled
  config WDTCWS = WDTCWS_7   ; window always open (100%); software control; keyed access not required
  config WDTCCS = SC         ; Software Control
  config WRTC = OFF          ; configuration registers (300000-30000Bh) not write-protected
  config WRTB = OFF          ; Boot Block (000000-0007FFh) not write-protected
  config WRTD = OFF          ; Data EEPROM not write-protected
  config SCANE = ON          ; Scanner module is available for use  SCANMD bit can control the module
  config LVP = OFF           ; HV on MCLR/VPP must be used for programming
  config CP = OFF            ; UserNVM code protection disabled
  config CPD = OFF           ; DataNVM code protection disabled
  config EBTRB = OFF         ; Boot Block (000000-0007FFh) not protected from table reads executed in other blocks

#ifdef __18F24K40
  config WRT0 = OFF          ; Block 0 (000800-001FFFh) not write-protected
  config WRT1 = OFF          ; Block 1 (002000-003FFFh) not write-protected
  config EBTR0 = OFF         ; Block 0 (000800-001FFFh) not protected from table reads executed in other blocks
  config EBTR1 = OFF         ; Block 1 (002000-003FFFh) not protected from table reads executed in other blocks
#endif
#ifdef __18F25K40
  config WRT0 = OFF          ; Block 0 (000800-001FFFh) not write-protected
  config WRT1 = OFF          ; Block 1 (002000-003FFFh) not write-protected
  config WRT2 = OFF          ; Block 2 (004000-005FFFh) not write-protected
  config WRT3 = OFF          ; Block 3 (006000-007FFFh) not write-protected
  config EBTR0 = OFF         ; Block 0 (000800-001FFFh) not protected from table reads executed in other blocks
  config EBTR1 = OFF         ; Block 1 (002000-003FFFh) not protected from table reads executed in other blocks
  config EBTR2 = OFF         ; Block 2 (004000-005FFFh) not protected from table reads executed in other blocks
  config EBTR3 = OFF         ; Block 3 (006000-007FFFh) not protected from table reads executed in other blocks
#endif
#ifdef __18F26K40
  config WRT0 = OFF          ; Block 0 (000800-003FFFh) not write-protected
  config WRT1 = OFF          ; Block 1 (004000-007FFFh) not write-protected
  config WRT2 = OFF          ; Block 2 (008000-00BFFFh) not write-protected
  config WRT3 = OFF          ; Block 3 (00C000-00FFFFh) not write-protected
  config EBTR0 = OFF         ; Block 0 (000800-003FFFh) not protected from table reads executed in other blocks
  config EBTR1 = OFF         ; Block 1 (004000-007FFFh) not protected from table reads executed in other blocks
  config EBTR2 = OFF         ; Block 2 (008000-00BFFFh) not protected from table reads executed in other blocks
  config EBTR3 = OFF         ; Block 3 (00C000-00FFFFh) not protected from table reads executed in other blocks
#endif
#ifdef __18F27K40
  config WRT0 = OFF          ; Block 0 (000800-003FFFh) not write-protected
  config WRT1 = OFF          ; Block 1 (004000-007FFFh) not write-protected
  config WRT2 = OFF          ; Block 2 (008000-00BFFFh) not write-protected
  config WRT3 = OFF          ; Block 3 (00C000-00FFFFh) not write-protected
  config WRT4 = OFF          ; Block 4 (010000-013FFFh) not write-protected
  config WRT5 = OFF          ; Block 5 (014000-017FFFh) not write-protected
  config WRT6 = OFF          ; Block 6 (018000-01BFFFh) not write-protected
  config WRT7 = OFF          ; Block 7 (01C000-01FFFFh) not write-protected
  config EBTR0 = OFF         ; Block 0 (000800-003FFFh) not protected from table reads executed in other blocks
  config EBTR1 = OFF         ; Block 1 (004000-007FFFh) not protected from table reads executed in other blocks
  config EBTR2 = OFF         ; Block 2 (008000-00BFFFh) not protected from table reads executed in other blocks
  config EBTR3 = OFF         ; Block 3 (00C000-00FFFFh) not protected from table reads executed in other blocks
  config EBTR4 = OFF         ; Block 4 (010000-013FFFh) not protected from table reads executed in other blocks
  config EBTR5 = OFF         ; Block 5 (014000-017FFFh) not protected from table reads executed in other blocks
  config EBTR6 = OFF         ; Block 6 (018000-01BFFFh) not protected from table reads executed in other blocks
  config EBTR7 = OFF         ; Block 7 (01C000-01FFFFh) not protected from table reads executed in other blocks
#endif

; it's not recommended you change these options unless you know what you are doing...
#define DEFAULT_BAUDRATE            19200 ; initial startup baudrate (default is 19200)
#define USE_BRGH                    1     ; high baudrate select (default is ON) 
#define USE_BRGH16                  1     ; 16 bit SPBRG support (default is OFF)

; import the device information file - you may need to add devices to this file...
;#include                            devices.inc
;*****************************************************************************
;* device specific settings...                                               *
;* DEVICE_ID - the microchip device ID                                       *
;* SEVICE_SERIES - for example, 16F or 18F                                   *
;* DEVICE_ROM - size of device ROM (in bytes)                                *
;* DEVICE_BLOCK_ERASE - size of device erase block (in bytes)                *
;* DEVICE_BLOCK_WRITE - size of the device write block size (in bytes)       *
;* is 1 if true, 0 if false                                                  *
;* DEVICE_EEPROM - size of onboard EEPROM                                    * 
;*****************************************************************************

#define DEVICE_IS_16F               0x01
#define DEVICE_IS_18F               0x04
#define DEVICE_IS_18FJ              0x05

; device info from DS40001772C
#ifdef  __18F24K40
   messg "building for 18F24K40"
   #include                         p18F24K40.inc   
   #define DEVICE_ID                0x69C0
   #define DEVICE_SERIES            DEVICE_IS_18F
   #define DEVICE_ROM               0x04000
   #define DEVICE_BLOCK_ERASE       64
   #define DEVICE_BLOCK_WRITE       64
   #define DEVICE_EEPROM            256
#endif
#ifdef  __18F25K40
   messg "building for 18F25K40"
   #include                         p18F25K40.inc   
   #define DEVICE_ID                0x69A0
   #define DEVICE_SERIES            DEVICE_IS_18F
   #define DEVICE_ROM               0x08000
   #define DEVICE_BLOCK_ERASE       64
   #define DEVICE_BLOCK_WRITE       64
   #define DEVICE_EEPROM            256
#endif
#ifdef  __18F26K40
   messg "building for 18F26K40"
   #include                         p18F26K40.inc   
   #define DEVICE_ID                0x6980
   #define DEVICE_SERIES            DEVICE_IS_18F
   #define DEVICE_ROM               0x10000
   #define DEVICE_BLOCK_ERASE       64
   #define DEVICE_BLOCK_WRITE       64
   #define DEVICE_EEPROM            1024
#endif
#ifdef  __18F27K40
   messg "building for 18F27K40"
   #include                         p18F27K40.inc   
   #define DEVICE_ID                0x6960
   #define DEVICE_SERIES            DEVICE_IS_18F
   #define DEVICE_ROM               0x20000
   #define DEVICE_BLOCK_ERASE       128
   #define DEVICE_BLOCK_WRITE       128
   #define DEVICE_EEPROM            1024
#endif
#ifdef  __18F45K40
   messg "building for 18F45K40"
   #include                         p18F45K40.inc   
   #define DEVICE_ID                0x6940
   #define DEVICE_SERIES            DEVICE_IS_18F
   #define DEVICE_ROM               0x08000
   #define DEVICE_BLOCK_ERASE       64
   #define DEVICE_BLOCK_WRITE       64
   #define DEVICE_EEPROM            1024
#endif
#ifdef  __18F46K40
   messg "building for 18F46K40"
   #include                         p18F46K40.inc   
   #define DEVICE_ID                0x6920
   #define DEVICE_SERIES            DEVICE_IS_18F
   #define DEVICE_ROM               0x10000
   #define DEVICE_BLOCK_ERASE       64
   #define DEVICE_BLOCK_WRITE       64
   #define DEVICE_EEPROM            1024
#endif
#ifdef  __18F47K40
   messg "building for 18F47K40"
   #include                         p18F47K40.inc   
   #define DEVICE_ID                0x6900
   #define DEVICE_SERIES            DEVICE_IS_18F
   #define DEVICE_ROM               0x20000
   #define DEVICE_BLOCK_ERASE       128
   #define DEVICE_BLOCK_WRITE       128
   #define DEVICE_EEPROM            1024
#endif

;=============================================================================
; END USER CONFIGURATION SECTION
;=============================================================================

; BRG16   BRGH    USARTMODE   FORMULA
; 0       0       8 Bit       (OSC / BAUDRATE / 64) - 1
#if (USE_BRGH16 == 0) && (USE_BRGH == 0)
#define MULT                        64
#define TXSTA_VALUE                 0x20
#endif

; BRG16   BRGH    USARTMODE   FORMULA
; 0       1       8 Bit       (OSC / BAUDRATE / 16) - 1
#if (USE_BRGH16 == 0) && (USE_BRGH == 1)
#define MULT                        16
#define TXSTA_VALUE                 0x24
#endif

; BRG16   BRGH    USARTMODE   FORMULA
; 1       0       16 Bit      (OSC / BAUDRATE / 4) - 1
#if (USE_BRGH16 == 1) && (USE_BRGH == 0)
#define MULT                        4
#define TXSTA_VALUE                 0x20
#endif

; BRG16   BRGH    USARTMODE   FORMULA
; 1       1       16 Bit      (OSC / BAUDRATE / 4) - 1
#if (USE_BRGH16 == 1) && (USE_BRGH == 1)
#define MULT                        4
#define TXSTA_VALUE                 0x24
#endif

; added 26-5-2018
#define RCSTA_VAL                   0x90  ;SPEN = 1, CREN = 1 (enable, continuous receive)

; formula is (OSC / BAUDRATE / MULT) - 1. However, we need to round the result
; so that the SPBRG value is correct
#define MULT_DIV_2                  MULT / 2
#define DEFAULT_SPBRG_VALUE         ((DEVICE_CLOCK / DEFAULT_BAUDRATE + MULT_DIV_2) / MULT - 1)
#define USER_SPBRG_VALUE            ((DEVICE_CLOCK / BAUDRATE + MULT_DIV_2) / MULT - 1)

; alias K40 USART1 registers    ; (26-08-2011)
#define PIR_          PIR3
#define RCIF_         RC1IF
#define TXIF_         TX1IF
#define TXSTA_        TX1STA
#define RCSTA_        RC1STA
#define SPBRG_        SP1BRG
#define SPBRGH_       SP1BRGH
#define TXREG_        TX1REG
#define RCREG_        RC1REG
#define BAUDCON_      BAUD1CON

; set the minimum loader size...
#define LOADER_SIZE_MIN             512 ; was 384
#define LOADER_SIZE                 LOADER_SIZE_MIN

; if the loader size is not an integer multiple of the erase block size, we
; need to set the loader size so it falls on an erase block boundary. For example
; EraseBlock == 64, LoaderSize := 384    (6 * 64)   - no change needed
; EraseBlock == 128, LoaderSize := 384   (3 * 128)  - change needed
; EraseBlock == 1024, LoaderSize := 1024 (1 * 1024) - change needed
#if DEVICE_BLOCK_ERASE > 0                ; if zero, then ignore
#if LOADER_SIZE % DEVICE_BLOCK_ERASE > 0  ; if modulus not zero, we need a new loader size
#undefine LOADER_SIZE
#define LOADER_SIZE                 DEVICE_BLOCK_ERASE * (LOADER_SIZE_MIN / DEVICE_BLOCK_ERASE + 1)
#endif
#endif

; now create the loader and user code origin...
#define ORG_LOADER                  DEVICE_ROM - LOADER_SIZE
#define ORG_LOADER_USER             ORG_LOADER - 8

; loader commands - these are bit field identifiers
; data input values are shown in comments
#define CMD_EEPROM_READ             0x00              ; 0x01
#define CMD_EEPROM_WRITE            0x01              ; 0x02
#define CMD_ROM_ERASE               0x02              ; 0x04
#define CMD_ROM_READ                0x03              ; 0x08
#define CMD_ROM_WRITE               0x04              ; 0x10
#define CMD_BAUDRATE                0x05              ; 0x20

; addition information and commands...
#define FIRMWARE_VERSION            0x10
#define LOADER_CONNECT              0x80
#define LOADER_ACK                  LOADER_CONNECT
#define LOADER_GO                   0x81
#define LOADER_ERROR_CRC            0x8A
#define LOADER_ERROR_VERIFY         0x8B

 ; program variables
 #define BUFFER_SIZE                DEVICE_BLOCK_WRITE
 CBLOCK 0x00
    CRC:        1
    Command:    1
    Ack:        1
    Index:      1
    EEVerify:   1
    Timeout_01: 1
    Timeout_02: 1
    Timeout_03: 1
    Buffer:     BUFFER_SIZE
 endc

   ; reset vector
   org    0x0000
   goto   LoaderStart

   ; relocated reset vector...
   org    ORG_LOADER_USER
LoaderStartUser

   ; loader start
   org    ORG_LOADER
LoaderStart
   ; user configuration macro...
   UserConfig

   ; set baudrate...
   movlw     (DEFAULT_SPBRG_VALUE) & 0xFF
   movwf     SPBRG_               ; (26-08-2011)   
   #if USE_BRGH16 == 1
   movlw     (DEFAULT_SPBRG_VALUE >> 8) & 0xFF
   movwf     SPBRGH_              ; (26-08-2011)
   bsf       BAUDCON_, BRG16      ; (10-08-2011)
   #endif

   ; init com to default - normally 19200 baud...
   movlw     TXSTA_VALUE          ; transmission config
   movwf     TXSTA_               ; into transmit status
   movlw     RCSTA_VAL            ; receive config
   movwf     RCSTA_               ; into receive status

  #ifdef TEST_UART
   rcall    TestUart
  #endif   

   ; can we connect to remote PC...
   rcall  ReadByte                ; read a byte
   sublw  LOADER_CONNECT          ; subtract connection ID
   bnz    LoaderStartUser         ; if not zero, run user code

   ; acknowledge...
   movlw  LOADER_CONNECT
   rcall  WriteByte

   ; wait for go before sending device information...
WaitForGo
   rcall ReadByte
   sublw LOADER_GO
   bnz   WaitForGo

   ; firmware version...
   movlw  FIRMWARE_VERSION
   rcall  WriteByte

   ; baudrate...
   movlw  (BAUDRATE >> 16) & 0xFF
   rcall  WriteByte
   movlw  (BAUDRATE >> 8) & 0xFF
   rcall  WriteByte
   movlw  (BAUDRATE) & 0xFF
   rcall  WriteByte

   ; device series...
   movlw  (DEVICE_SERIES) & 0xFF
   rcall  WriteByte

   ; device ID...
   movlw  (DEVICE_ID >> 8) & 0xFF
   rcall  WriteByte
   movlw  (DEVICE_ID) & 0xFF
   rcall  WriteByte

   ; the loader origin - this defines the loader protection point - note
   ; that the PC application will offset this value by 8 bytes to take
   ; into account the relocatable user code address
   movlw  (ORG_LOADER >> 16) & 0xFF
   rcall  WriteByte
   movlw  (ORG_LOADER >> 8) & 0xFF
   rcall  WriteByte
   movlw  (ORG_LOADER) & 0xFF
   rcall  WriteByte

   ; erase block size...
   movlw  (DEVICE_BLOCK_ERASE >> 8) & 0xFF
   rcall  WriteByte
   movlw  (DEVICE_BLOCK_ERASE) & 0xFF
   rcall  WriteByte

   ; write block size...
   movlw  (DEVICE_BLOCK_WRITE) & 0xFF
   rcall  WriteByte

   ; init ack
   movlw  LOADER_ACK
   movwf  Ack

; main program loop
LoaderMain

   ; get command...
   rcall  ReadByte                ; read a byte
   movwf  Command                 ; set command
   clrf   CRC                     ; clear CRC

   ; switch to user defined baudrate - the loader starts in the default
   ; 19200 baud - this enables you to crank up the speed after initial
   ; startup handshaking has taken place...
   #if BAUDRATE != DEFAULT_BAUDRATE
   btfss  Command, CMD_BAUDRATE   ; check command
   bra    NoBaudrate              ; no baudrate change

   ; set baudrate...
   movlw     (USER_SPBRG_VALUE) & 0xFF
   movwf     SPBRG_               ; (26-08-2011)      
   #if USE_BRGH16 == 1
   movlw     (USER_SPBRG_VALUE >> 8) & 0xFF
   movwf     SPBRGH_              ; (26-08-2011)
   bsf       BAUDCON_, BRG16      ; (10-08-2011)
   #endif
   bra    LoaderMain              ; no ack (settle time) loop back
   #endif
NoBaudrate

   ; read upper byte of ROM address and high
   ; byte of EEPROM address...
   rcall  ReadByte                ; read a byte
   movwf  TBLPTRU                 ; move into table pointer upper
   #if    DEVICE_EEPROM > 256     ; if we have extended EEPROM
   movwf  NVMADRH            ; then move value into EEPROM address high
   #endif

   ; read high byte of ROM address and low
   ; byte of EEPROM addresss...
   rcall  ReadByte                ; read a byte
   movwf  TBLPTRH                 ; move into table pointer high
   #if    DEVICE_EEPROM > 0       ; if we have onboard EEPROM
   movwf  NVMADRL                 ; then move value into EEPROM address low
   #endif

   ; read low byte of ROM address and
   ; data EEPROM byte...
   rcall  ReadByte                ; read a byte
   movwf  TBLPTRL                 ; move into table pointer low
   #if    DEVICE_EEPROM > 0       ; if we have onboard EEPROM
   movwf  NVMDAT             ; then move value into EEPROM data
   movwf  EEVerify                ; store - used to verify the written data
   #endif

   ; EEPROM READ...
   #ifdef ENABLE_EEPROM_READ
   #if    DEVICE_EEPROM > 0       ; if device has onboard EEPROM
   btfsc  Command, CMD_EEPROM_READ; check command
   rcall  EEPROMRead              ; execute EEPROM read
   #endif
   #endif

   ; EEPROM WRITE...
   #if    DEVICE_EEPROM > 0       ; if device has onboard EEPROM
   btfsc  Command,CMD_EEPROM_WRITE; check command
   rcall  EEPROMWrite             ; execute EEPROM write
   #endif

   ; CODE ERASE...
   btfsc  Command, CMD_ROM_ERASE  ; check command
   rcall  ROMErase                ; execute program code erase

   ; CODE READ...
   btfsc  Command, CMD_ROM_READ   ; check command
   rcall  ROMRead                 ; execute program code read

   ; CODE WRITE...
   btfsc  Command, CMD_ROM_WRITE  ; check command
   rcall  ROMWrite                ; execute program code write

   ; write ack...
   movf   Ack, W                  ; ack byte to w reg
   rcall  WriteByte               ; send
   bra    LoaderMain              ; then loop back

; verify error...
VerifyError
   movlw  LOADER_ERROR_VERIFY     ; set flag
   rcall  WriteByte               ; send ack
   bra    LoaderMain              ; loop back to main

; CRC error...
CRCError
   movlw  LOADER_ERROR_CRC        ; set flag
   rcall  WriteByte               ; send ack
   bra    LoaderMain              ; loop back to main

; execute user code...
RunUserCode
   #if DEVICE_ROM > 0x10000       ; 15-08-2011
   clrf   TBLPTRU                 ; clear table pointer upper
   #endif
   clrf   STKPTR                  ; reset stack pointer
   clrf   RCSTA_                  ; reset receive status and control register
   clrf   TXSTA_                  ; reset transmit status and control register
   goto   LoaderStartUser         ; execute user code

;*****************************************************************************
; Name    : WriteByte                                                        *
; Purpose : Write a byte to the hardware USART                               *
;*****************************************************************************
WriteByte
   clrwdt                         ; clear WDT
   btfss  TXSTA_, TRMT            ; can we send data (10-08-2010)
   bra    WriteByte               ; no - loop back
   movwf  TXREG_                  ; send data
   return

;*****************************************************************************
; Name    : ReadByte                                                         *
; Purpose : Read a byte from the hardware USART. Return in WREG              *
;         : If a timeout occurs, the routine will branch to 'RunUserCode'    *
;         : TIMEOUT                                                          *
;         : Assume a clock of 20MHz, then one instruction takes              *
;         : 1 / 20 000 000 * 4 = 2us to execute                              *
;         : Timeout_03 loop costs 6 cycles                                   *
;         : Timeout_02 loop costs 4 cycles                                   *
;         : Timeout_01 loop costs 4 cycles                                   *
;         : 256 * 6          = 1536 cycles                                   *
;         : 256 * 1536 + 4   = 393220 cycles                                 *
;         : 20 * 393220 + 4  = 7864404 cycles                                *
;         : 7864404 * 2us = 1.57 seconds - this is constant through varying  *
;         : clock speeds                                                     *
;*****************************************************************************
ReadByte
   movlw  (DEVICE_CLOCK / 1000000)
   movwf  Timeout_01
TimeoutLoop_01                    ;  4 cycle cost, excluding inner...
   clrf   Timeout_02              ; (1)
TimeoutLoop_02                    ;  4 cycle cost, excluding inner...
   clrf   Timeout_03              ; (1)
TimeoutLoop_03                    ;  6 cycle cost - if nothing received...
   clrwdt                         ; (1)
   banksel PIR_
   btfss  PIR_, RCIF_             ; (1) if not skipped
   bra    NotReceived             ; (2)
   banksel 0
   movf   RCREG_, W               ; return in WREG
   addwf  CRC,F                   ; calculate the CRC checksum
   return
NotReceived
   banksel 0
   decfsz Timeout_03,F            ; (1) if not skipped
   bra    TimeoutLoop_03          ; (2)
   decfsz Timeout_02,F            ; (1) if not skipped
   bra    TimeoutLoop_02          ; (2)
   decfsz Timeout_01,F            ; (1) if not skipped
   bra    TimeoutLoop_01          ; (2)
   bra    RunUserCode


;*****************************************************************************
; Name    : TestUart                                                         *
; Purpose : uart loopback test                                               *
;*****************************************************************************
; uart test routine (conditional)
#ifdef TEST_UART
TestUart
   movlw  0x55           ; display 'U>'
   rcall  WriteByte
   movlw  0x3e
   rcall  WriteByte
echo_loop:                ; read and echo all chars
   rcall  GetByte
   rcall  WriteByte
   goto   echo_loop
   return
 
GetByte
wait_char:
   clrwdt
   btfsc  RCSTA_, OERR    ; check for overrun error
   bra    handle_oerr
   banksel PIR_
   btfss  PIR_, RCIF_             ; (1) if not skipped
   bra    wait_char
   movf   RCREG_, W               ; return in WREG
   banksel 0
   return
handle_oerr:
   bcf    RCSTA_, CREN      ; toggle CREN to clear OERR
   movf   RCREG_, W
   bsf    RCSTA_, CREN
   bra    wait_char
#endif       ; ifdef TEST_UART

;*****************************************************************************
; Name    : EEPROMRead                                                       *
; Purpose : Read a single byte from onboard EEPROM                           *
;*****************************************************************************
#ifdef ENABLE_EEPROM_READ
#if DEVICE_EEPROM > 0
EEPROMRead
   bcf    NVMCON1, NVMREG1      ; point to data memory
   bcf    NVMCON1, NVMREG0
   bsf    NVMCON1, RD           ; EEProm read
   movf   NVMDAT, W               ; W <- EEDATA
   rcall  WriteByte               ; send
   return
#endif
#endif

;*****************************************************************************
; Name    : EEPROMWrite                                                      *
; Purpose : Write a single byte of data to onboard EEPROM                    *
;*****************************************************************************
#if DEVICE_EEPROM > 0
EEPROMWrite
   ; CRC check...
   rcall  ReadByte                ; read a byte
   tstfsz CRC                     ; test for zero
   bra    CRCError                ; not zero, so error

   ; write data...
   bcf    NVMCON1, NVMREG1      ; point to data memory
   bcf    NVMCON1, NVMREG0
   rcall  NVMUnlock             ; required write sequence

   ; wait For write completion
EEPromWriteWait
   btfsc  NVMCON1, WR             ; is Bit WR of EECON1 Clear?
   GoTo   EEPromWriteWait         ; no, loop back
   bcf    NVMCON1, WREN            ; disable writes

   ; now verify the data
   bsf    NVMCON1, RD             ; EEProm read
   movf   EEVerify, W             ; data byte received
   cpfseq NVMDAT                  ; compare
   bra    VerifyError             ; failed - set ack
   return
#endif

;*****************************************************************************
; Name    : ROMErase                                                         *
; Purpose : Erase a block of device ROM                                      *
;*****************************************************************************
ROMErase
   bsf    NVMCON1, NVMREG1      ; point to program memory
   bcf    NVMCON1, NVMREG0
   bsf    NVMCON1, FREE
   rcall  NVMUnlock             ; required write sequence
   bcf    NVMCON1, WREN            ; Disable writes
   bcf    NVMCON1, FREE
   return

;*****************************************************************************
; Name    : ROMRead                                                          *
; Purpose : Read a single byte from ROM                                      *
;*****************************************************************************
ROMRead
   bsf    NVMCON1, NVMREG1      ; point to program memory
   bcf    NVMCON1, NVMREG0
   tblrd  *+                      ; perform table read (low byte)
   movf   TABLAT, W               ; W <- TABLAT
   rcall  WriteByte               ; send
   return

;*****************************************************************************
; Name    : ROMWrite                                                         *
; Purpose : Write data held in data buffer to ROM                            *
;*****************************************************************************
ROMWrite
   bsf    NVMCON1, NVMREG1      ; point to program memory
   bcf    NVMCON1, NVMREG0
   lfsr   0, Buffer               ; load buffer address
   movlw  BUFFER_SIZE             ; load buffer size
   movwf  Index                   ; into index

ReadData:
   rcall  ReadByte                ; read a byte
   movwf  POSTINC0                ; save to buffer
   decfsz Index                   ; dec index
   bra    ReadData                ; loop back if not zero

   ; CRC check...
   rcall  ReadByte                ; read a byte
   tstfsz CRC                     ; test for zero
   bra    CRCError                ; not zero, so error

   ; write the data into holding registers...
   lfsr   0, Buffer               ; load buffer address
   movlw  BUFFER_SIZE             ; loader buffer size
   movwf  Index                   ; into index

LoadData:
   movf   POSTINC0, W
   movwf  TABLAT                  ; TABLAT <- data
   tblwt  *+                      ; write to holding register
   decfsz Index                   ; dec index
   bra    LoadData                ; loop back if not zero

   ; write the data...
   tblrd  *-                      ; point back, need this before write...

   bsf    NVMCON1, NVMREG1      ; point to program memory
   bcf    NVMCON1, NVMREG0
   rcall  NVMUnlock
   bcf    NVMCON1, WREN            ; Disable writes

   ; verify the write...
   movf   POSTDEC0, W             ; dec buffer pointer
   movlw  BUFFER_SIZE             ; load buffer size
   movwf  Index                   ; into index

VerifyData:
   tblrd  *-                      ; read data from ROM
   movf   POSTDEC0, W             ; read data from RAM
   cpfseq TABLAT                  ; compare values
   bra    VerifyError             ; failed, set ack
   decfsz Index                   ; dec index
   bra    VerifyData              ; loop back if not zero
   return

;*****************************************************************************
; Name    : NVMUnlock                                                        *
; Purpose : A required write sequence for EEPROM and CODE writes             *
;*****************************************************************************
NVMUnlock       
   banksel NVMCON1
   bsf    NVMCON1, WREN
   movlw  0x55                    ; W <- $55
   movwf  NVMCON2                 ; EECON2 <- W
   movlw  0xAA                    ; W <- $AA
   movwf  NVMCON2                 ; EECON2 <- W
   bsf    NVMCON1, WR              ; set WR Bit For write
   nop
   return

; check to make sure program fits 26-5-2018
#if $ > DEVICE_ROM
  error "program code exceeds flash size. change LOADER_SIZE_MIN"
#endif
   end
