;------------------------------------------------------------------
;	FILE:		vp6_lcd.asm
;	AUTHOR:		Tom Perme
;	COMPANY:	Microchip Technology, Inc.
;	DEVICE:		PIC16F690
;
;	DESCRIP:	This project is intended to demonstrate
;				how to use the 0.6V internal reference
;				voltage to measure Vdd voltage supplied.
;
;				'MeasureVdd' is the primary function.  But first,
;				a part must be calibrated at a known voltage.
;				Also pay attention to the constant VREFPLUS
;				which is the voltage at which calibration is run,
;				and StoreCalibData which stores the calibration
;				data to EEPROM.
;------------------------------------------------------------------
;  The software supplied herewith by Microchip Technology Incorporated
;  (the "Company") for its PICmicro Microcontroller is intended and
;  supplied to you, the Company's customer, for use solely and
;  exclusively on Microchip PICmicro Microcontroller products. The
;  software is owned by the Company and/or its supplier, and is
;  protected under applicable copyright laws. All rights are reserved.
;  Any use in violation of the foregoing restrictions may subject the
;  user to criminal sanctions under applicable laws, as well as to
;  civil liability for the breach of the terms and conditions of this
;  license.
; 
;  THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES,
;  WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
;  TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
;  PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
;  IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
;  CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
;********************************************************************
;  File Description:
; 
; 
; 
;  Change History:
;  Author			Date		Description
;  ------------------------------------------------------------------
;  Tom Perme		12 Dec 2006	Created.
;
;********************************************************************


	
	#include <p16F690.inc>			; Include Part Specific definitions/labels

	errorlevel -302					; Don't show bank warnings in build output
	

	; Config Bits
	; Turn off unneeded features.
	__config 	_FCMEN_OFF & _IESO_OFF & _BOR_OFF & _CP_OFF & _MCLRE_OFF & _PWRTE_OFF & _WDT_OFF & _INTRC_OSC_NOCLKOUT


;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
; Defines
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


; Declare variables
uvars udata 0x20
DelayCount		res 1				; Counter for delays
DelayCount2		res 1				; Counter for delays
temp			res 1				; misc. use local variable
SaveLSb			res 1				; For saving LSB in calibration
VuCount_H		res 1				; High byte of ADC result
VuCount_L		res 1				; Low byte of ADC result
Vdd_H			res 1				; Final result as Vdd*100
Vdd_L			res 1				; .. uses two bytes
EE_Data			res 1				; Local bytes used for
EE_Addr			res 1				; EEPROM reads/writes.

; multiply variables
mulcnd 			res 1				; 8 bit multiplicand
mulplr 			res 1				; 8 bit multiplier
Prod_H 			res 1				; High byte of the 16 bit result
Prod_L  		res 1				; Low byte of the 16 bit result
count   		res 1				; loop counter

; divide variables
ACCaLO  		res 1				; denominator
ACCaHI  		res 1				; horrible horrible names
ACCbLO  		res 1				; numerator, and 16-bit output
ACCbHI  		res 1
ACCcLO  		res 1				; 16-bit remainder
ACCcHI  		res 1				
ACCdLO  		res 1				; used internal to divide routine
ACCdHI  		res	1


; lcd variables
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
DataByte		res 1				; Byte of data to send to LCD
H_byte  		res 1				; High and low byte inputs for 
L_byte  		res 1				; bintobcd routine dec # < 65535
R0      		res 1       		; Result byte outputs
R1      		res 1
R2      		res 1		

; Declare pinouts (LCD)
#define DATA_PORT	PORTC			; Parallel PORT for data.
#define DATA_TRIS	TRISC			; TRIS register for data port.	
#define	D4			PORTC, 0		; Use 4-bit communication       
#define D5			PORTC, 1		; Declare pins, Note D4 is not	
#define D6			PORTC, 2		; contiguous to D5, D6, D7.
#define D7			PORTC, 3

#define CTL_PORT	PORTA			; Port for output control lines
#define CTL_TRIS	TRISA			; I/O reg for ctl lines port
#define REGSEL		PORTA, 5		; Register Select pin def'n
;#define READWRITEo	PORTA, 2		; Read/Write_not pin def'n      ; tied to ground (always writing to LCD)
#define ENABLE		PORTA, 4		; Enable pin def'n
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



; EEPROM variables (0..255)
#define VP6CALVALH			0x01	; Addresses for placing H and L
#define VP6CALVALL			0x02	; bytes into EEPROM
	;
	; For calibrating Vdd voltages > 3V, only VP6CALVALL is required


; Declare constants					
constant VREFPLUS 		= d'400' 		; Vdd used to meas VP6 during calibration
constant VREFCORRECTED 	= VREFPLUS/2	; Corrected for implementation use
	;
	;  Limits on value:		300 < VREFPLUS < 510  (2.50V to 5.10V)
	;
	;  The voltage used to take the calibration measurement must be
	;  in this range for the mathematical implementation to be assured.
	;  NOTE: HV parts must have VREFPLUS < Voltage for Shunt Regulator (5.0V)
	;        4.00V is a good choice for all parts


; Initialization definitions
#define ADCON0_INIT 	b'10110101' ; Turn on, R-Jstfy, Ch VP6, ...
#define ADCON1_INIT	 	b'01110000'	; Select Frc clock source.




;---------------------------------------------------------------------
;	Begin Program Code
;---------------------------------------------------------------------

STARTUP code
	goto 	Main					; On power-up begin at label "Main"
	nop
	nop
	nop
	goto	ISR						; Int. vector @ loc'n STARTUP+4



;.....................................................................
; Main Program Code
;.....................................................................

PROG0 code							; For linker to place code

Main:

Initializations:
	
	; The A/D must be initialized with the proper settings
	;
	; Bank 1
	banksel	TRISC
	clrf	TRISC					; PORTC = digital o/p 	(can be removed)
	clrf	TRISA					; PORTA = outputs, force to be non-floating.
	bsf		TRISA, 3				; set RA3 as input

	; ADC Settings
	banksel	ADCON1
	movlw 	ADCON1_INIT				
	movwf 	ADCON1					; Set ADC clock source

	banksel	ADCON0
	movlw 	ADCON0_INIT
	movwf 	ADCON0					; Enable ADC, channel selects, L/R justification

	; Analog Pins
	banksel	ANSEL
	clrf	ANSEL					; Set all AN pins as digital i/o
	clrf	ANSELH
	
	; Clear outputs
	banksel	PORTC
	clrf	PORTC					; clear portc
	clrf	PORTA
	
	
	; Delay 1 second
	movlw	d'255'					; Delay ~1 second before
	call	Delay_ms				; doing anything so that
	movlw	d'255'					; enough time allowed for prgming
	call	Delay_ms				; that eeprom not modified.
	movlw	d'255'					;   else
	call	Delay_ms				; can accidentally enter calib
	movlw	d'255'					; mode and overwrite calib data
	call	Delay_ms				; when using _MCLRE_OFF


	

	; If button is down, 
	; ENTER CALIBRATION MODE
	btfsc	PORTA, 3				; Enter CALIBRATION MODE
	goto	$+4						; skip next 3 instr - don't run calib
	movlw	0x06
	movwf	PORTC					; Signify calibration mode ON
	call	StoreCalibData			; Run calibration (assumes Vdd=4.00V)
	

	; WAIT FOR STARTUP BUTTON
	; (Signifies program running)
	clrf	PORTC
	bsf		PORTC, 0				; Show 0x9 on Pickit2 LPC board LEDs
	bsf		PORTC, 3				; Indicates ready to start - await button.
	call	WaitForButton			; On press begin program.

	
	
	; INITIALIZE LCD
	call 	LCDInit					; Prepare LCD for use
	call	LCDClear				; Clear the display
	call	LCDHome					; Put cursor to home, (0,0)



MainLoop:


	call 	MeasureVdd				; Returns Vdd_H : Vdd_L


	; Display some helpful debug data
	; to ensure everything is in order.

	; Disp Text #1	: Line 1 Col 1
	movlw	0x00
	call	LCDGoto
	movlw	'V'
	call	SendData
	movlw	'd'
	call	SendData
	movlw	'd'
	call	SendData
	movlw	'='
	call	SendData				; Vdd=###
	movf	Vdd_H, W				; Display MeasureVdd result
	movwf	H_byte
	movf	Vdd_L, W
	movwf	L_byte				
	call	LCDDisplayDecimal		; Put # on screen


	; Disp Text #2	: Line 1 Col N (such that right is flush)
	movlw	0x0A
	call	LCDGoto
	movlw	'C'
	call	SendData
	movlw	'V'
	call	SendData
	movlw	'='
	call	SendData
	banksel	EE_Addr
	movlw	VP6CALVALL
	movwf	EE_Addr
	call	ReadEEDAT				
	banksel EE_Data					; CV=###
	movlw	d'0'					; Display calibration value
	movwf	H_byte					
	movf	EE_Data, W				
	movwf	L_byte				
	call	LCDDisplayDecimal		; Put # onto screen.


	; Disp Text #3	: Line 2 Col 1
	movlw	0x40
	call	LCDGoto
	movlw	'A'
	call	SendData
	movlw	'D'
	call	SendData
	movlw	'C'
	call	SendData
	movlw	'='
	call	SendData				; ADC=###
	movf 	VuCount_H, W			; Display MeasureVdd ADC conversion
	movwf	H_byte
	movf	VuCount_L, W			; Put unknown voltage count 
	movwf	L_byte					; to H & L bytes
	call	LCDDisplayDecimal		; Put # onto screen.


	
	goto 	MainLoop



;-----------------------------------------------------------;
;   Routines												;
;-----------------------------------------------------------;



;*********************************************************************
;  MEASURE VDD Routines
;
;		The following functions are used to measure Vdd.
;*********************************************************************

MeasureVdd:
	; Calls all functions to take one measurement of Vdd

	call 	SetChannelVP6			; Select input to ADC
	call 	RunAtoDconversion		; Result VuCount_H : VuCount_L
	call	CalibrateResult			; Result Vdd_H : Vdd_L

	banksel 0						; leave in bank 0
	return							;! end MeasureVdd


AcqTime:
	; Device clock configured to run @ 4MHz
	; Must wait ~7.5us worst case (datasheet) --> 30 clock cycles
	; 1 instr cycle = 4 clk cycle --> I require > 7.5 instr cyc

	goto	$+1						; +2us  (Each goto will take
	goto	$+1						; +2us	two instruction cycles
	goto 	$+1						; +2us	to get to next instruction.)
	return							;! (+2us for return --> 8us Total)


SetChannelVP6:
	; If your program uses the ADC other than to measure Vdd,
	; make sure the channel select is set for the 0.6V input.

	banksel	ADCON0				
	bsf		ADCON0, CHS3	
	bsf		ADCON0, CHS2
	bcf		ADCON0, CHS1
	bsf		ADCON0, CHS0			; CHS<3:0> = 1101
	return							;! end SetChannelVP6


RunAtoDconversion:
	; Run a single conversion through the ADC.

	call 	AcqTime					; Wait for charge cap to reach s.s.
	banksel ADCON0
	bsf 	ADCON0, GO				; Start a sample.
	btfsc	ADCON0, GO				; Wait until completed.
	goto	$-1						; On exit: result in ADRESH & ADRESL

	; Put ADRESH:L to variables
	movf	ADRESH, W
	banksel	VuCount_H
	movwf	VuCount_H				; VuCount_H = ADRESH
	banksel	ADRESL					; go to adresl bank
	movf	ADRESL, W				; W = ADRESL
	banksel	VuCount_L				; go to res_l bank
	movwf	VuCount_L				; VuCount_L = ADRESL
	
	return							;! end RunAtoDconversion


CalibrateResult:

	; 5-step process

	; 1) Save LSb of VuCount_L
	; 2) Multiply  VREFCORRECTED * VP6CALVAL 
	; 3) Divide 16-bit mpy result by 16-bit reading of ADC unknown voltage
	; 4) Multiply 16-bit result by 2 (shift left once)
	; 5) Restore/Correct for loss of LSb from /2
	;    (Steps 1&5 are due to implementation, not theory.)

	; 1) Save VuCount_L LSb	
	clrf	SaveLSb					; Clear all bits of saving byte
	btfsc	VuCount_L, 0			; If VuCount_L<0> is high, set bit in
	bsf		SaveLSb, 0				; .. SaveLSb to use at end of routine.
	; JUSTIFICATION of 1 & 5
	; Doing steps 1 & 5 can actually round up when not supposed to, but
	; they provide 'better' results more than they hurt.  To stay true to the
	; mathematics, take out steps 1 & 5, and always round down.  OR to follow
	; the mathematics exactly and keep that LSb, do a 16x16 multiply of Vref*VP6CALVAL, 
	; which is possibly greater than 16-bit result for worst case on 0.6V. Then perform
	; a 24-bit/16-bit division.  However, the decrease in tolerance is marginal.
	; (This roundoff has error up to ~10mV, e.g. if it rounds 2.001 to 3.)
	;
	; Rounding up gives odd numbers which is better to see as a user than only
	; even numbers, which is a false perception for the same accuracy.


	; 2) Multiply

	; READ 0.6V CALIBRATION VALUE
	banksel	EE_Addr
	movlw	VP6CALVALL
	movwf	EE_Addr
	call	ReadEEDAT				; EE_Data = VP6CALVAL
	

	; Multiply  VrefCorr * VP6CALVAL
	movf	EE_Data, W				
	movwf	mulplr
	movlw	VREFCORRECTED			; Precompiled constant set by user
	movwf	mulcnd					; Perform:  VP6CALVAL * (Vref*100/2)
	call	Multiply				; Result in Prod_H & Prod_L

	

	; 3) Divide

	movf	Prod_H, W				; Prepare the numerator (mpy result)
	movwf	ACCbHI
	movf	Prod_L, W
	movwf	ACCbLO

	banksel VuCount_H				; Prepare the denominator (ADC reading)
	movf	VuCount_H, W
	movwf	ACCaHI
	movf	VuCount_L, W
	movwf	ACCaLO
									; Perform division
	call	Divide					; Result in ACCbHI & ACCbLO


	; 4) Multiply x2

	bcf		STATUS, C				; Clear carry to push into LSb
	rlf		ACCbLO, F				; Rotate LSByte, C=old MSb
	rlf		ACCbHI, F				; Rotate MSByte, bring in carry as LSb
	
	; Rename to have significance
	movf	ACCbLO, W
	movwf	Vdd_L					; Vdd_L = low byte 
	movf	ACCbHI, W
	movwf	Vdd_H					; Vdd_H = high byte


	; 5) Restore LSb 
	; It was lost from divide/mult by 2/2
	btfsc	SaveLSb, 0				; Check if LSb is a one
	bsf		Vdd_L, 0				; Yes, Set Vdd_L - else.. already zero

	return							;! end CalibrateResult




;*********************************************************************
;  EEPROM Routines
;
;		Required
;*********************************************************************

;.....................................................................
;  WriteEEDAT
;
;  Inputs:
;	 EE_Data : The data byte to be written to EEPROM.
;	 EE_Addr : Where in EEPROM to write the data byte.
;.....................................................................
WriteEEDAT:

	; Move variables to registers
	banksel EE_Data	
	movf	EE_Data, W			
	banksel	EEDAT
	movwf	EEDAT					; EEDAT  = EE_Data
	banksel	EE_Addr
	movf	EE_Addr, W				
	banksel	EEADR
	movwf	EEADR					; EEADR  = EE_Addr
	
	; Prep for write
	banksel	EECON1
	bcf		EECON1, EEPGD			; Use for DATA memory
	bsf		EECON1, WREN			; Allow writing

	bcf		INTCON, GIE				; Disable interrupts
	btfsc	INTCON, GIE				; during write process.
	goto	$-2						; Make sure has taken effect

	; Required Sequence ---
	movlw	0x55					
	movwf	EECON2					; Write 0x55
	movlw	0xAA					
	movwf	EECON2					; Write 0xAA
	bsf		EECON1, WR				; Set WR to begin write
	; ---------------------


	; Clear settings
	bsf		INTCON, GIE				; Re-enable interrupts
	bcf		EECON1, WREN			; Disable writes


	; Wait until write complete
	btfsc	EECON1, WR				; Check if WR complete
	goto	$-1						; Once clear.. exit routine

	banksel	0						; Leave flow in bank 0
	return
	


;.....................................................................
;  ReadEEDAT
;
;  Inputs:
;	 EE_Addr : specifies the address of EEPROM to read.
;
;  Outputs:
;	 EE_Data : the read value from EEPROM is placed here.
;.....................................................................
ReadEEDAT:
	
	banksel	EE_Addr
	movf	EE_Addr, W
	banksel	EEADR					
	movwf	EEADR					; EEADR = EE_Addr

	banksel	EECON1
	bcf		EECON1, EEPGD			; Use DATA memory
	bsf		EECON1, RD				; Read EEPROM
	
	banksel	EEDAT
	movf	EEDAT, W				; W = EEDAT
	banksel	EE_Data
	movwf	EE_Data					; EE_Data = W

	banksel 0						; Leave flow in Bank 0
	return



;.....................................................................
;  StoreCalibData
;
;  ASSUMES: Vdd = a stable voltage specified
;			as VREFPLUS for taking the calibration
;			value.  The user must set this before
;			entering calibration mode on the physical
;			device, and the same value should be set
;			as the constant VREFPLUS.
;
;   For Vdd's specified with VREFPLUS constant at beginning,
;	only low byte is required to be stored.  
;
;   (High byte is shown for completeness, and later potential
;	modification, although it is currently unused.)
;.....................................................................
StoreCalibData:
	; STORE CALIBRATION VALUES

	; Take a measurment of 0.6V as input to ADC
	call	MeasureVdd

	; Store measured high byte of VP6CALVAL
	banksel	EE_Data
	movf	VuCount_H, W
	movwf	EE_Data
	movlw	VP6CALVALH
	movwf	EE_Addr
	call	WriteEEDAT				 
	
	; Store measured low byte of VP6CALVAL
	banksel	EE_Data
	movf	VuCount_L, W
	movwf	EE_Data
	movlw	VP6CALVALL
	movwf	EE_Addr				
	call	WriteEEDAT

	return



;*********************************************************************
;  Delay Routines
;
;		Required for LCD and for exmaple application start up delay.
;*********************************************************************

;---------------------------------------------------------------------
;  Delay_us
;
;  Inputs:
;	 W = # microseconds to delay  {0..255}
;
;		Crude delay will delay roughly the specified number of
;	microseconds, but is not exact in order to keep code small. 
;
;		It will Guarantee delay is longer than 1-byte value input.
;---------------------------------------------------------------------
Delay_us:
								; +2 us to enter subroutine
	; Time based on instruction cycles.

	movwf	DelayCount			; Move # us to delay into DelayCount
	bcf		STATUS, C			; Clear C so rrf shifts in 0 to MSb
	rrf		DelayCount, F		; Divide # us to delay by 2.
	bcf		STATUS, C			; Clear C again
	rrf		DelayCount, F		; Divide # us again by 2, tot div by 4

	; Loop Eq. = 4*N-1
	nop							; +1 null instr
	decfsz	DelayCount, F		; +1(2br) Perform Looping.
	goto 	$-2					; +2 Go backwards $-#, # instructions

	; Delayed 4*N-1 + 7 usec thus far.
	; Will use two more usec on exit of routine.

	; OVERALL EQUATION OF ROUTINE, Input #us = Count
	; Delay(us) = INTEGER(Count/4)*4+8

	return						; +2 us to exit subroutine



;---------------------------------------------------------------------
;  Delay_ms
;
;  Inputs:
;	 W = # milliseconds to delay {0..255}
;
;		Will delay roughly # of ms, and will ensure delay is 
;	greater than supplied number. Note that if 0 is supplied,
;	it will wait at least 1 ms. Based on 4MHz clock.
;---------------------------------------------------------------------
Delay_ms:

	movwf	DelayCount2			; Move # ms to delay into DelayCount
	
	; Loop Eq. = dominated by large delay
	movlw	d'250'				
	call	Delay_us			; Run 250 us Delay
	call	Delay_us			; do so four times
	call	Delay_us			; such that ea. pass
	call	Delay_us			; of loop delays 1ms.
	decfsz	DelayCount2, F		; Decrement Counter to itself, skip if zero
	goto	$-6
	
	return



;*********************************************************************
;  MATH ROUTINES
;
;		Required for Measuring Vdd
;*********************************************************************


;.....................................................................
;  Multiply
;
;	mulcnd*mulplr = Prod_H : Prod_L
; 
;	optimized for code size (looped)
;.....................................................................
Multiply:
	clrf    Prod_H				; Clear outputs
	clrf    Prod_L				;  "
	movlw   8
	movwf   count
	movf    mulcnd, W
	bcf     STATUS, C    		; Clear the carry bit in the status Reg.
mloop:
	rrf     mulplr, F
	btfsc   STATUS, C
	addwf   Prod_H, F
	rrf     Prod_H, F
	rrf     Prod_L, F
	decfsz  count, F
	goto    mloop

	return


;*********************************************************************
;  Divide
;
;	Double precision divide.
;
;   Division : ACCb(16 bits) / ACCa(16 bits) -> ACCb(16 bits) with
;                                               Remainder in ACCc (16 bits)
;
;*********************************************************************
Divide:

	call    ud16setup
	clrf    ACCcHI
	clrf    ACCcLO
ud16dloop:
	bcf     STATUS, C
	rlf     ACCdLO, F
	rlf     ACCdHI, F
	rlf     ACCcLO, F
	rlf     ACCcHI, F
	movf    ACCaHI, W
	subwf   ACCcHI, W          	; check if a>c
	btfss   STATUS, Z
	goto    ud16nochk
	movf    ACCaLO, W
	subwf   ACCcLO, W        	; if msb equal then check lsb
ud16nochk:
	btfss   STATUS, C    		; carry set if c>a
	goto    ud16nogo
	movf    ACCaLO, W        	; c-a into c
	subwf   ACCcLO, F
	btfss   STATUS, C
	decf    ACCcHI, F
	movf    ACCaHI, W
	subwf   ACCcHI, F
	bsf     STATUS, C    		;shift a 1 into b (result)
ud16nogo:
	rlf     ACCbLO, F
	rlf     ACCbHI, F
	decfsz  temp, F         	; loop untill all bits checked
	goto    ud16dloop
	return


ud16setup:
	movlw   .16             	; for 16 shifts
	movwf   temp
	movf    ACCbHI, W          	; move ACCb to ACCd
	movwf   ACCdHI
	movf    ACCbLO, W
	movwf   ACCdLO
	clrf    ACCbHI
	clrf    ACCbLO
	return


;*******************************************************************






;*********************************************************************
;  LCD Control Routines
;*********************************************************************


;---------------------------------------------------------------------
;  SendData
;
;  Inputs:
;	 W = byte to send
;		
;		Sends a byte located in the W register to the LCD display.
;---------------------------------------------------------------------
SendData:
	
	; Process based on HANTRONIX 
	; character LCD module interface
	banksel	CTL_PORT
	bsf		REGSEL				; Set to write data
	banksel	DataByte
	movwf	DataByte			; Move W to variable DataByte
	call	SendByte			; Put the instruction to LCD

	return						; Return from subroutine



;---------------------------------------------------------------------
;  SendInstruction
;
;  Inputs:
;	 W = byte to send
;		
;		Sends a byte located in the W register to the LCD display.
;---------------------------------------------------------------------
SendInstruction:

	; Process based on HANTRONIX 
	; character LCD module interface

	banksel	CTL_PORT
	bcf		REGSEL				; Clear to write instruction
	banksel	DataByte
	movwf	DataByte			; Move W to variable DataByte
	call	SendByte			; Put the instruction to LCD
	
	return						; Return from subroutine



;---------------------------------------------------------------------
;  SendNibble
;
;  Inputs:
;	 W = W<7:4> = nibble to send, <3:0> may be garbage
;  Expects:
;	 REGSEL bit shall be set prior to calling SendNibble because
;	 SendNibble will be used both for data and instructions.
;---------------------------------------------------------------------
SendNibble:
	banksel	CTL_PORT
	bsf		ENABLE				; Enable x-fr of data MCU->LCD_module
	andlw	0xF0				; Clear low nibble of W
	
	movwf	temp				; Store copy of nibble to send
	; Set Data lines
	; Set contiguous bits
	movlw	b'11110000'			; load 'anti-mask'
	andwf	DATA_PORT, F		; clear bits to be set
	swapf	temp, W
	andlw	b'00001111'			; get lower 4 bits only
	iorwf	DATA_PORT, F
	
	nop							; give min delay before lower ENABLE
	bcf		ENABLE				; Lower enable signal while data valid

	return



;---------------------------------------------------------------------
;  SendByte
;
;  Inputs:
;	 'DataByte' = variable byte to send
;  Expects:
;	 REGSEL bit shall be set prior to calling to say instr/data byte.
;---------------------------------------------------------------------
SendByte:

	banksel	DataByte
	movf	DataByte, W			; W = DataByte
	andlw	0xF0				; Get DataByte<7:4>
	call	SendNibble			; Send upper nibble
 
	banksel	DataByte
	swapf	DataByte, W			; W = DB<3:0>:DB<7:4>
	andlw	0xF0				; Get DataByte<3:0>
	call	SendNibble			

	movlw	d'51'				; 110% of time for instr on module to execute.
	call	Delay_us			; special instructions may req more time
								; .. provide those individually after use
	return 


	

;*********************************************************************
;  LCD Routines
;*********************************************************************

;---------------------------------------------------------------------
;  LCDInit
;---------------------------------------------------------------------
LCDInit:

	; Set control port directions
	banksel	CTL_TRIS			; Ensure proper bank
	bcf		REGSEL				; Set as ouptut
;	bcf		READWRITEo			; Set as output
	bcf		ENABLE				; Set as output

	; Set data port directions
	banksel DATA_TRIS			
	bcf		D4					; Set as output
	bcf		D5					; Set as output
	bcf		D6					; Set as output
	bcf		D7					; Set as output
	clrf	DATA_PORT
	

	; Set initial values
	banksel	CTL_PORT
	bcf		REGSEL				; Set line low
;	bcf		READWRITEo			; Set line low
	bcf		ENABLE				; Set line low

	banksel	DATA_PORT
	clrf	DATA_PORT			; Set all data lines low

	banksel 0					; return to bank 0


	; Using HANTRONIX LCD controller module
	;
	; Set up LCD for use with 8-bit data mode
	;   Software Initialization
	movlw	d'22'				; Need to wait > 20ms after power applied
	call	Delay_ms			; before doing anything LCD related

	; Ctrl Seq 1
	bcf		REGSEL
;	bcf		READWRITEo
	movlw	b'00110000'			; Send Required Ctrl Sequence #1,  
	call	SendNibble			; This sequnce has no specified meaning.
	movlw	d'6'				; this instruction takes > 4.1ms to execute
	call	Delay_ms

	; Ctrl Seq 2
	bcf		REGSEL
	movlw	b'00110000'			; Send Required Ctrl Sequence #2, no real meaning
	call	SendNibble
	movlw	d'200'				; this instruction takes 100us to execute
	call	Delay_us

	; Ctrl Seq 3
	bcf		REGSEL
	movlw	b'00110000'			; Send Required Ctrl Sequence #3, no real meaning
	call	SendNibble
	movlw	d'200'				; this takes 100 us to execute
	call	Delay_us	

	; Send Initialization Values
	bcf		REGSEL
	movlw	b'00100000'			; function set.  4 bit, 2 lines.
	call	SendNibble
	movlw	d'50'
	call	Delay_us

	movlw	b'00101100'			; 2 lines, 5x7 dots
	call	SendInstruction		; std delay handled
	
	movlw	b'00001100'			; display on, cursor off, blink off
	call	SendInstruction		; std delay handled

	movlw	b'00000001'			; clear display return cursor home
	call	SendInstruction
	movlw	d'3'				; time to execute > 1.64 ms
	call	Delay_ms

	movlw	b'00000110'			; set position to auto increment
	call	SendInstruction		; std delay handled
	

	return
	


;---------------------------------------------------------------------
;  LCDGoto
;
;  Inputs:
;	 W = Display Data Address to put cursor.
;
;			 Addresses(values)    (Columns on screen)       
;	Row 1:   0x00 0x01 .. .. .. .. .. .. .. 0x26 0x27 
;	Row 2:   0x40 0x41 .. .. .. .. .. .. .. 0x66 0x67 
;              1   2   3  4  5  .. so on ..  39   40   Nth display postion
;---------------------------------------------------------------------
LCDGoto:
	iorlw	b'10000000'			; ensure upper most bit set for cmd
	bcf		REGSEL				; select instruction registers
	call	SendInstruction		; Send the command.
	return


;---------------------------------------------------------------------
;  LCDHome
;
;		Sends cursor to position #1 (upper leftmost).
;---------------------------------------------------------------------
LCDHome:
	movlw	0x00
	goto	LCDGoto				; Send cursor to start position
	return
	

;---------------------------------------------------------------------
;  LCDClear
;
;		Clears the display.
;---------------------------------------------------------------------
LCDClear:
	bcf		REGSEL				; select non-character registers
	movlw	b'00000001'			; clear screen
	call	SendInstruction
	movlw	d'2'				; wait for command to finish
	call	Delay_ms
	bsf		REGSEL				; pick character registers
	return


;---------------------------------------------------------------------
;  LCDDisplayDecimal
;
;  Inputs:
;	 H_byte, L_Byte : expected to have high and low byte values for
;					  the value to be displayed (decimal value
;					  stored in hex)	
;
;		This function will display the value of H_Byte & L_Byte
;	on the LCD screen in decimal format.
;---------------------------------------------------------------------
LCDDisplayDecimal:
	call	bintobcd			; convert the numbers of H, L bytes
	
	; Results start from low nibble of R2 = MSdigit .. R0low = LSdigit

; 2 MSDigits commented out (always 0 in example)
; -----------------------------------
;	movfw	R2					; fetch most signif. digit first, dig5
;	andlw	0x0F				; Mask off nibble
;	addlw	0x30				; Convert to an ASCII character
;	call	SendData
;
;	swapf	R1, W				; W = dig3:dig4
;	andlw	0x0F				; Grab only dig4
;	addlw	0x30				; convert to ASCII
;	call	SendData

	movfw	R1					; W = dig4:dig3
	andlw	0x0F				; Grab only low nibble
	addlw	0x30				; convert to ASCII
	call	SendData

	swapf	R0, W				; W = dig1:dig2
	andlw	0x0F				; Grab only low nibble
	addlw	0x30				; convert to ASCII
	call	SendData

	movfw	R0					; W = dig2:dig1
	andlw	0x0F				; Grab only low nibble
	addlw	0x30				; convert to ASCII
	call	SendData

	return




;---------------------------------------------------------------------
; ===== BINARY TO BCD CONVERSION =====
;      This routine converts a 16 Bit binary Number to a 5 Digit
; BCD Number. 
;      The 16 bit binary number is input in locations H_byte and
; L_byte with the high byte in H_byte.
;       The 5 digit BCD number is returned in R0, R1 and R2 with R2
;	being the containing the most significant digit in its lower nibble.
;
;  		Pictorally:
;	R2H   R2L   R1H   R1L   R0H   R0L
;    -    dig5  dig4  dig3  dig2  dig1
;---------------------------------------------------------------------
bintobcd:
Bin16BCD:  
	bcf     STATUS,C         	; clear the carry bit in STATUS
	movlw   .16
	movwf   temp+1
	clrf    R0
	clrf    R1
	clrf    R2
b16b_loop16  
	rlf     L_byte, F
	rlf     H_byte, F
	rlf     R0, F
	rlf     R1, F
	rlf     R2, F

	decfsz  temp+1, F
	goto    adjDEC
	return

adjDEC  
	movlw   R0
	movwf   FSR			
	call    adjBCD
	
	movlw   R1
	movwf   FSR			
	call    adjBCD

	movlw   R2
	movwf   FSR			
	call    adjBCD
	goto    b16b_loop16

adjBCD  
	movlw   0x03
	addwf   INDF,W
	movwf   temp
	btfsc   temp,3          	; test if result > 7
	movwf   INDF
	movlw   0x30
	addwf   INDF,W
	movwf   temp
	btfsc   temp,7          	; test if result > 7
	movwf   INDF            	; save as MSD
	return



;*********************************************************************
;  MISC
;*********************************************************************

;---------------------------------------------------------------------
;  WaitForButton
;---------------------------------------------------------------------
WaitForButton:
	; Await button press
	btfsc	PORTA, 3		
	goto	$-1				; check that button is down
	movlw	d'20'
	call	Delay_us		; debounce delay 20us
	btfss	PORTA, 3
	goto	$-1				; check that button released
	movlw	d'20'
	call	Delay_us		; debounce delay 20us
	return





;---------------------------------------------------------------------
;  Interrupt Service Routine
;---------------------------------------------------------------------

ISR:								; Were interrupts used,
	retfie							; service them here.




;.....................................................................
; STOP - The End
	end


