;------------------------------------------------------------------
;	FILE:		vp6_measonly.asm
;	AUTHOR:		Tom Perme
;	COMPANY:	Microchip Technology, Inc.
;	DEVICE:		PIC16F690
;
;	DESCRIP:	This project is intended to demonstrate
;				how to use the 0.6V internal reference
;				voltage to measure Vdd voltage supplied.
;
;				'MeasureVdd' is the primary function.  But first,
;				a part must be calibrated at a known voltage.
;				Also pay attention to the constant VREFPLUS
;				which is the voltage at which calibration is run,
;				and StoreCalibData which stores the calibration
;				data to EEPROM.
;------------------------------------------------------------------
;  The software supplied herewith by Microchip Technology Incorporated
;  (the "Company") for its PICmicro Microcontroller is intended and
;  supplied to you, the Company's customer, for use solely and
;  exclusively on Microchip PICmicro Microcontroller products. The
;  software is owned by the Company and/or its supplier, and is
;  protected under applicable copyright laws. All rights are reserved.
;  Any use in violation of the foregoing restrictions may subject the
;  user to criminal sanctions under applicable laws, as well as to
;  civil liability for the breach of the terms and conditions of this
;  license.
; 
;  THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES,
;  WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
;  TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
;  PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
;  IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
;  CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
;********************************************************************
;  File Description:
; 
; 
; 
;  Change History:
;  Author			Date		Description
;  ------------------------------------------------------------------
;  Tom Perme		12 Dec 2006	Created.
;
;********************************************************************


	
	#include <p16F690.inc>			; Include Part Specific definitions/labels

	errorlevel -302					; Don't show bank warnings in build output
	

	; Config Bits
	; Turn off unneeded features.
	__config _FCMEN_OFF & _IESO_OFF & _BOR_OFF & _CP_OFF & _MCLRE_OFF & _PWRTE_OFF & _WDT_OFF & _INTRC_OSC_NOCLKOUT


;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
; Defines
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

; Declare pinouts


; Declare variables
uvars udata 0x20
DelayCount		res 1				; Counter for delays
DelayCount2		res 1				; Counter for delays
temp			res 1				; misc. use local variable
SaveLSb			res 1				; For saving LSB in calibration
VuCount_H		res 1				; High byte of ADC result
VuCount_L		res 1				; Low byte of ADC result
Vdd_H			res	1				; Final result as Vdd*100
Vdd_L			res 1				; .. uses two bytes
EE_Data			res 1				; Local bytes used for
EE_Addr			res 1				; EEPROM reads/writes.

; multiply variables
mulcnd 			res 1				; 8 bit multiplicand
mulplr 			res 1				; 8 bit multiplier
Prod_H 			res 1				; High byte of the 16 bit result
Prod_L  		res 1				; Low byte of the 16 bit result
count   		res 1				; loop counter

; divide variables
ACCaLO  		res 1				; denominator
ACCaHI  		res 1				; horrible horrible names
ACCbLO  		res 1				; numerator, and 16-bit output
ACCbHI  		res 1
ACCcLO  		res 1				; 16-bit remainder
ACCcHI  		res 1				
ACCdLO  		res 1				; used internal to divide routine
ACCdHI  		res	1



; EEPROM variables (0..255)
#define VP6CALVALH			0x01	; Addresses for placing H and L
#define VP6CALVALL			0x02	; bytes into EEPROM
	;
	; For calibrating Vdd voltages > 3V, only VP6CALVALL is required


; Declare constants					
constant VREFPLUS 		= d'400' 		; Vdd used to meas VP6 during calibration
constant VREFCORRECTED 	= VREFPLUS/2	; Corrected for implementation use
	;
	;  Limits on value:		300 < VREFPLUS < 510  (2.50V to 5.10V)
	;
	;  The voltage used to take the calibration measurement must be
	;  in this range for the mathematical implementation to be assured.
	;  NOTE: HV parts must have VREFPLUS < Voltage for Shunt Regulator (5.0V)
	;        4.00V is a good choice for all parts


; Initialization definitions
#define ADCON0_INIT 	b'10110101' ; Turn on, R-Jstfy, Ch VP6, ...
#define ADCON1_INIT	 	b'01110000'	; Select Frc clock source.




;---------------------------------------------------------------------
;	Begin Program Code
;---------------------------------------------------------------------

STARTUP code
	goto 	Main					; On power-up begin at label "Main"
	nop
	nop
	nop
	goto	ISR						; Int. vector @ loc'n STARTUP+4



;.....................................................................
; Main Program Code
;.....................................................................

PROG0 code							; For linker to place code

Main:

Initializations:
	
	; The A/D must be initialized with the proper settings

	; ADC Settings
	banksel	ADCON1
	movlw 	ADCON1_INIT				
	movwf 	ADCON1					; Set ADC clock source

	banksel	ADCON0
	movlw 	ADCON0_INIT
	movwf 	ADCON0					; Enable ADC, channel selects, L/R justification




MainLoop:

	; Application Code
	
	goto 	MainLoop



;-----------------------------------------------------------;
;   Routines												;
;-----------------------------------------------------------;



;*********************************************************************
;  MEASURE VDD Routines
;
;		The following functions are used to measure Vdd.
;*********************************************************************

MeasureVdd:
	; Calls all functions to take one measurement of Vdd

	call 	SetChannelVP6			; Select input to ADC
	call 	RunAtoDconversion		; Result VuCount_H : VuCount_L
	call	CalibrateResult			; Result Vdd_H : Vdd_L

	banksel 0						; leave in bank 0
	return							;! end MeasureVdd


AcqTime:
	; Device clock configured to run @ 4MHz
	; Must wait ~7.5us worst case (datasheet) --> 30 clock cycles
	; 1 instr cycle = 4 clk cycle --> I require > 7.5 instr cyc

	goto	$+1						; +2us  (Each goto will take
	goto	$+1						; +2us	two instruction cycles
	goto 	$+1						; +2us	to get to next instruction.)
	return							;! (+2us for return --> 8us Total)


SetChannelVP6:
	; If your program uses the ADC other than to measure Vdd,
	; make sure the channel select is set for the 0.6V input.

	banksel	ADCON0				
	bsf		ADCON0, CHS3	
	bsf		ADCON0, CHS2
	bcf		ADCON0, CHS1
	bsf		ADCON0, CHS0			; CHS<3:0> = 1101
	return							;! end SetChannelVP6


RunAtoDconversion:
	; Run a single conversion through the ADC.

	call 	AcqTime					; Wait for charge cap to reach s.s.
	banksel ADCON0
	bsf 	ADCON0, GO				; Start a sample.
	btfsc	ADCON0, GO				; Wait until completed.
	goto	$-1						; On exit: result in ADRESH & ADRESL

	; Put ADRESH:L to variables
	movf	ADRESH, W
	banksel	VuCount_H
	movwf	VuCount_H				; VuCount_H = ADRESH
	banksel	ADRESL					; go to adresl bank
	movf	ADRESL, W				; W = ADRESL
	banksel	VuCount_L				; go to res_l bank
	movwf	VuCount_L				; VuCount_L = ADRESL
	
	return							;! end RunAtoDconversion


CalibrateResult:

	; 5-step process

	; 1) Save LSb of VuCount_L
	; 2) Multiply  VREFCORRECTED * VP6CALVAL 
	; 3) Divide 16-bit mpy result by 16-bit reading of ADC unknown voltage
	; 4) Multiply 16-bit result by 2 (shift left once)
	; 5) Restore/Correct for loss of LSb from /2
	;    (Steps 1&5 are due to implementation, not theory.)

	; 1) Save VuCount_L LSb	
	clrf	SaveLSb					; Clear all bits of saving byte
	btfsc	VuCount_L, 0			; If VuCount_L<0> is high, set bit in
	bsf		SaveLSb, 0				; .. SaveLSb to use at end of routine.
	; JUSTIFICATION of 1 & 5
	; Doing steps 1 & 5 can actually round up when not supposed to, but
	; they provide 'better' results more than they hurt.  To stay true to the
	; mathematics, take out steps 1 & 5, and always round down.  OR to follow
	; the mathematics exactly and keep that LSb, do a 16x16 multiply of Vref*VP6CALVAL, 
	; which is possibly greater than 16-bit result for worst case on 0.6V. Then perform
	; a 24-bit/16-bit division.  However, the decrease in tolerance is marginal.
	; (This roundoff has error up to ~10mV, e.g. if it rounds 2.001 to 3.)
	;
	; Rounding up gives odd numbers which is better to see as a user than only
	; even numbers, which is a false perception for the same accuracy.


	; 2) Multiply

	; READ 0.6V CALIBRATION VALUE
	banksel	EE_Addr
	movlw	VP6CALVALL
	movwf	EE_Addr
	call	ReadEEDAT				; EE_Data = VP6CALVAL
	

	; Multiply  VrefCorr * VP6CALVAL
	movf	EE_Data, W				
	movwf	mulplr
	movlw	VREFCORRECTED			; Precompiled constant set by user
	movwf	mulcnd					; Perform:  VP6CALVAL * (Vref*100/2)
	call	Multiply				; Result in Prod_H & Prod_L

	

	; 3) Divide

	movf	Prod_H, W				; Prepare the numerator (mpy result)
	movwf	ACCbHI
	movf	Prod_L, W
	movwf	ACCbLO

	banksel VuCount_H				; Prepare the denominator (ADC reading)
	movf	VuCount_H, W
	movwf	ACCaHI
	movf	VuCount_L, W
	movwf	ACCaLO
									; Perform division
	call	Divide					; Result in ACCbHI & ACCbLO


	; 4) Multiply x2

	bcf		STATUS, C				; Clear carry to push into LSb
	rlf		ACCbLO, F				; Rotate LSByte, C=old MSb
	rlf		ACCbHI, F				; Rotate MSByte, bring in carry as LSb
	
	; Rename to have significance
	movf	ACCbLO, W
	movwf	Vdd_L					; Vdd_L = low byte 
	movf	ACCbHI, W
	movwf	Vdd_H					; Vdd_H = high byte


	; 5) Restore LSb 
	; It was lost from divide/mult by 2/2
	btfsc	SaveLSb, 0				; Check if LSb is a one
	bsf		Vdd_L, 0				; Yes, Set Vdd_L - else.. already zero

	return							;! end CalibrateResult




;*********************************************************************
;  EEPROM Routines
;
;		Required
;*********************************************************************

;.....................................................................
;  WriteEEDAT
;
;  Inputs:
;	 EE_Data : The data byte to be written to EEPROM.
;	 EE_Addr : Where in EEPROM to write the data byte.
;.....................................................................
WriteEEDAT:

	; Move variables to registers
	banksel EE_Data	
	movf	EE_Data, W			
	banksel	EEDAT
	movwf	EEDAT					; EEDAT  = EE_Data
	banksel	EE_Addr
	movf	EE_Addr, W				
	banksel	EEADR
	movwf	EEADR					; EEADR  = EE_Addr
	
	; Prep for write
	banksel	EECON1
	bcf		EECON1, EEPGD			; Use for DATA memory
	bsf		EECON1, WREN			; Allow writing

	bcf		INTCON, GIE				; Disable interrupts
	btfsc	INTCON, GIE				; during write process.
	goto	$-2						; Make sure has taken effect

	; Required Sequence ---
	movlw	0x55					
	movwf	EECON2					; Write 0x55
	movlw	0xAA					
	movwf	EECON2					; Write 0xAA
	bsf		EECON1, WR				; Set WR to begin write
	; ---------------------


	; Clear settings
	bsf		INTCON, GIE				; Re-enable interrupts
	bcf		EECON1, WREN			; Disable writes


	; Wait until write complete
	btfsc	EECON1, WR				; Check if WR complete
	goto	$-1						; Once clear.. exit routine

	banksel	0						; Leave flow in bank 0
	return
	


;.....................................................................
;  ReadEEDAT
;
;  Inputs:
;	 EE_Addr : specifies the address of EEPROM to read.
;
;  Outputs:
;	 EE_Data : the read value from EEPROM is placed here.
;.....................................................................
ReadEEDAT:
	
	banksel	EE_Addr
	movf	EE_Addr, W
	banksel	EEADR					
	movwf	EEADR					; EEADR = EE_Addr

	banksel	EECON1
	bcf		EECON1, EEPGD			; Use DATA memory
	bsf		EECON1, RD				; Read EEPROM
	
	banksel	EEDAT
	movf	EEDAT, W				; W = EEDAT
	banksel	EE_Data
	movwf	EE_Data					; EE_Data = W

	banksel 0						; Leave flow in Bank 0
	return



;.....................................................................
;  StoreCalibData
;
;  ASSUMES: Vdd = a stable voltage specified
;			as VREFPLUS for taking the calibration
;			value.  The user must set this before
;			entering calibration mode on the physical
;			device, and the same value should be set
;			as the constant VREFPLUS.
;
;   For Vdd's specified with VREFPLUS constant at beginning,
;	only low byte is required to be stored.  
;
;   (High byte is shown for completeness, and later potential
;	modification, although it is currently unused.)
;.....................................................................
StoreCalibData:
	; STORE CALIBRATION VALUES

	; Take a measurment of 0.6V as input to ADC
	call	MeasureVdd

	; Store measured high byte of VP6CALVAL
	banksel	EE_Data
	movf	VuCount_H, W
	movwf	EE_Data
	movlw	VP6CALVALH
	movwf	EE_Addr
	call	WriteEEDAT				 
	
	; Store measured low byte of VP6CALVAL
	banksel	EE_Data
	movf	VuCount_L, W
	movwf	EE_Data
	movlw	VP6CALVALL
	movwf	EE_Addr				
	call	WriteEEDAT

	return




;*********************************************************************
;  MATH ROUTINES
;
;		Required for Measuring Vdd
;*********************************************************************


;.....................................................................
;  Multiply
;
;	mulcnd*mulplr = Prod_H : Prod_L
; 
;	optimized for code size (looped)
;.....................................................................
Multiply:
	clrf    Prod_H				; Clear outputs
	clrf    Prod_L				;  "
	movlw   8
	movwf   count
	movf    mulcnd, W
	bcf     STATUS, C    		; Clear the carry bit in the status Reg.
mloop:
	rrf     mulplr, F
	btfsc   STATUS, C
	addwf   Prod_H, F
	rrf     Prod_H, F
	rrf     Prod_L, F
	decfsz  count, F
	goto    mloop

	return


;*********************************************************************
;  Divide
;
;	Double precision divide.
;
;   Division : ACCb(16 bits) / ACCa(16 bits) -> ACCb(16 bits) with
;                                               Remainder in ACCc (16 bits)
;
;*********************************************************************
Divide:

	call    setup
	clrf    ACCcHI
	clrf    ACCcLO
dloop  
	bcf     STATUS,C
	rlf     ACCdLO, F
	rlf     ACCdHI, F
	rlf     ACCcLO, F
	rlf     ACCcHI, F
	movf    ACCaHI,W
	subwf   ACCcHI,W          ;check if a>c
	btfss   STATUS,Z
	goto    nochk
	movf    ACCaLO,W
	subwf   ACCcLO,W        ;if msb equal then check lsb
nochk   
	btfss   STATUS,C    ;carry set if c>a
	goto    nogo
	movf    ACCaLO,W        ;c-a into c
	subwf   ACCcLO, F
	btfss   STATUS,C
	decf    ACCcHI, F
	movf    ACCaHI,W
	subwf   ACCcHI, F
	bsf     STATUS,C    ;shift a 1 into b (result)
nogo    
	rlf     ACCbLO, F
	rlf     ACCbHI, F
	decfsz  temp, F         ;loop untill all bits checked
	goto    dloop

	retlw   0

;*********************************************************************

setup   movlw   .16             ; for 16 shifts
	movwf   temp
	movf    ACCbHI,W          ;move ACCb to ACCd
	movwf   ACCdHI
	movf    ACCbLO,W
	movwf   ACCdLO
	clrf    ACCbHI
	clrf    ACCbLO
	retlw   0


;*********************************************************************

neg_A   
	comf    ACCaLO, F       ; negate ACCa ( -ACCa -> ACCa )
	incf    ACCaLO, F
	btfsc   STATUS, Z
	decf    ACCaHI, F
	comf    ACCaHI, F
	retlw   0


;*********************************************************************
; end Divide
;*********************************************************************





;---------------------------------------------------------------------
;  Interrupt Service Routine
;---------------------------------------------------------------------

ISR:								; Were interrupts used,
	retfie							; service them here.




;.....................................................................
; STOP - The End
	end
