'****************************************************************
'*  Name    : ST7032i_10.inc                                    *
'*  Author  : [Alberto Freixanet]                               *
'*  Notice  : Copyright (c) 2012 [AFreixanet 2012]              *
'*          : All Rights Reserved                               *
'*  Date    : 08/12/2012                                        *
'*  Version : 1.0                                               *
'*  Notes   : Driver for I2C Display with ST7032i interface     *
'*          : LiquidCrystal_I2C_ST7032i                         *
'* Compiler : Proton Basic Version 3554                         *
'****************************************************************
'
' Date    : 08/12/2012
' Version : 1.0 for Proton Basic 3554 
'-------------------------------------------------------------------------------
' This library has been modified for Proton Basic language and has been adapted for all 
' Print commands of Proton to interface with a ST7032i LCD chip driver.
'
' Written for the Crownhill Proton Basic compiler version 3.5.5.4 onwards using an 18F device,
'
' And based on a code written in C language for LiquidCrystal_I2C_ST7032i V1.0 for Arduino.
'
'-------------------------------------------------------------------------------
'===============================================================================
' This program is free software, you can redistribute it and/or modify.
' The code is offered as-is and the author take any responsibility for its use.
' This program is distributed in the hope that it will be useful, but WITHOUT ANY
' WARRANTY. However, a mention of the author and compiler would be appreciated.
'
' Este programa es libre, puede redistribuirlo y/o modificarlo.
' Se entrega este cdigo tal como esta y el autor NO se hace responsable de
' su utilizacin. Este programa se distribuye con la esperanza de que sea til,
' pero SIN NINGUNA GARANTA. No obstante citar el autor y el compilador
' sera muy apreciado.

' Ce programme est un logiciel libre, vous pouvez le redistribuer et/ou modifier.
' Ce logiciel est offert tel qu'il est, et son auteur NE prend AUCUNE responsabilit
' pour son utilization. Ce programme est distribu dans l'espoir qu'il sera util,
' mais SANS AUCUNE GARANTIE. Cependant mentionner l'auteur et le compileur serait
' aprci.
'===============================================================================
'-------------------------------------------------------------------------------       
' The ST7032 dot-matrix liquid crystal display controller can display alphanumeric,
' Japanese kana characters, and symbols. It can be configured to drive a dot-matrix
' liquid crystal display under the control of a 4 / 8-bit with 6800-series or 8080-series,
' 3/4-line serial interface microprocessor.
' Since all the functions such as display RAM, character generator ROM/RAM and liquid
' crystal driver, required for driving a dot-matrix liquid crystal display are
' internally provided on one chip, a minimal system can be used with this controller/driver.
' The ST7032 character generator ROM size is 256 5x8dot bits which can be used to
' generate 256 different character fonts (5x8dot)
'
' The ST7032 is suitable for low voltage supply (2.7V to 5.5V) and is perfectly
' suitable for any portable product which is driven by the battery and requires
' low power consumption.
' The ST7032 LCD driver consists of 17 common signal drivers and 80 segment signal drivers.
' The maximum display RAM size can be either 80 characters in 1-line display or 40 
' characters in 2-line display. A single ST7032 can display up to one 16-character 
' line or two 16-character lines.
' The ST7032 dot-matrix LCD driver does not need extra cascaded drivers.
'-------------------------------------------------------------------------------
'===============================================================================
'
' Initializing by Internal Reset Circuit:
' An internal reset circuit automatically initializes the ST7032 when the power
' is turned on.
'
' If the electrical characteristics conditions listed under the table Power Supply 
' Conditions Using Internal Reset Circuit are not met, the internal reset circuit
' will not operate normally and will fail to initialize the ST7032.
'
' When the display powers up, it is configured as follows:

' The following instructions are executed during the initialization. The busy flag
' (BF) is kept in the busy state (BF = 1) until the initialization ends.
' The busy state lasts for 40 ms after VDD rises to stable.
' It just only could write Data or Instruction to ST7032i by the IIC interface.
'
'   1. Display clear
'   2. Function set:
'       DL = 1; 8-bit interface data
'       N = 0; 1-line display
'       DH=0; normal 5x8 font
'       IS=0; use instruction table 0
'   3. Display on/off control:
'       D = 0; Display off
'       C = 0; Cursor off
'       B = 0; Blinking off
'   4. Entry mode set:
'       I/D = 1; Increment by 1
'       S = 0; No shift
'   5. Internal OSC frequency
'       (F2,F1,F0)=(1,0,0)
'   6. ICON control
'       Ion=0; ICON off
'   7. Power control
'       BS=0; 1/5bias
'       Bon=0; booster off
'       Fon=0; follower off
'       (C5,C4,C3,C2,C1,C0)=(1,0,0,0,0,0)
'       (Rab2,Rab1,Rab0)=(0,1,0)
'
' Note, however, that resetting the Amicus18 board doesn't reset the LCD, so we
' can't assume that its in that state when a sketch starts (and the
' LiquidCrystal constructor is called).
'*******************************************************************************
'*******************************************************************************

$ifndef _ST7032i_
    $define _ST7032i_

'-------------------------------------------------------------------------------
    GoTo OverDriverST7032
'*******************************************************************************    
'
    Declare LCD_Type = Alphanumeric
    
'$define LCD_DEBUG

' Define the LCD Voltage from the AMicus18 Board, Here:
$define LCDVoltage5V
'$define LCDVoltage3V
' 
'-------------------------------------------------------------------------------
' Function Name     : Declaration of variables for the LCD ST7032i Driver
' Routine Number    : X001  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Notes             : 
'
    Dim BPF As Word System
    Dim GEN As Byte System
    Dim CUR_Row As Byte System
    Dim CUR_Col As Byte System
'    Dim CRS_First As Byte System
    Dim CRS_Col As Byte System
    Dim aCUR_Col As Byte System
    Dim aCUR_Row As Byte System        
'
    Dim LCD_Flags As Byte System
    Dim CommandFlag As LCD_Flags.0    
    Dim LCD_FunctionSet_IS As LCD_Flags.1
'
    Dim LCD_DataInp As Byte
    Dim LCD_ByteToWrite As Byte
    Dim LCD_InstToWrite As Byte    
'
    Dim LCDDataByte As Byte
    Dim Contrast As Byte
    Dim LCD_Index As Byte
    Dim CUR_Chrs As Byte
    Dim Row As Byte
'
    Dim _displaycontrol As Byte System
'
'only for DEBUG
$ifdef LCD_DEBUG
    Dim CRS_Col_Start As Byte
    Dim CUR_Col_Start As Byte
    Dim CUR_Row_Start As Byte
$endif   
'
    Symbol AddressST7032 = %01111100    ' Bit 0 is the R/W: Always = 0
    Symbol cols = 16                    ' 16 columns display
    Symbol numlines = 2                 ' 2 lines display
    Symbol LCD_Initialized = BPF.0      ' =1 the LCD is Initialized     
'===============================================================================
'-------------------------------------------------------------------------------
' Function Name     : Command/flag definitions
' Routine Number    : X002  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Notes             : 
'
$define LCD_ClearDisplay			$01   ' [1] Command
$define LCD_ReturnHome				$02   ' [2] Command
'
' flags for display entry mode
$define LCD_EntryModeSet			$06   '[3] Command And Setting 
$define LCD_ShiftDisplayToLeft      $07   '[3] Command and Setting  
$define LCD_ShiftDisplayToRight     $05   '[3] Command And Setting
'                      
' flags for display on/off control
$define LCD_DisplayControl	$08  '[4] Command
$define LCD_Display_ON      $04  ' Setting, use | function
$define LCD_Display_OFF     $FB  ' Setting, use & function 
$define LCD_Cursor_ON       $02  ' Setting, use | function
$define LCD_Cursor_OFF      $FD  ' Setting, use & function   
$define LCD_Blink_ON        $01  ' Setting, use | function
$define LCD_Blink_OFF       $FE  ' Setting, use & function 
'
' flags for Functions
$define LCD_FunctionSet_2Line_Basic     $38     '[5] Command and Settings 
$define LCD_FunctionSet_2Line_Extended  $39     '[5] Command And Settings
$define LCD_FunctionSet_1Line_Basic     $32     '[5] Command and Settings 
$define LCD_FunctionSet_1Line_Extended  $33     '[5] Command And Settings
'
$define LCD_SetDDRAMADDR		        $80     '[6]  Command	
'
$define LCD_DisplayCursorSHIFT	$10  '[10] Command 
$define LCD_CursorToLeft        $00  ' Setting
$define LCD_CursorToRight       $04  ' Setting
$define LCD_DisplayToLeft       $08  ' Setting
$define LCD_DisplayToRight      $0C  ' Setting
'                                   
$define LCD_SetCGRAMADDR        $40  '[11] Command
'
$define LCD_Bias_OSC_Control    $10  '[12] Command
$define LCD_Bias1_4 		    $08	 ' Setting				 
$define LCD_Bias1_5 			$F7  ' Setting*
'					
' Internal frequency adjust for VDD = 3.0 V
$ifdef LCDVoltage3V
$define LCD_OSC_122				$00	 ' Setting (ST7032)			
$define LCD_OSC_131				$01	 ' Setting (ST7032)			
$define LCD_OSC_144				$02	 ' Setting (ST7032)			
$define LCD_OSC_161				$03	 ' Setting (ST7032)			
$define LCD_OSC_183				$04	 ' Setting (ST7032) Default			
$define LCD_OSC_221				$05	 ' Setting (ST7032)			
$define LCD_OSC_274				$06	 ' Setting (ST7032)			
$define LCD_OSC_347				$07	 ' Setting (ST7032)
$endif
'			
' Internal frequency adjust for VDD = 5.0 V
$ifdef LCDVoltage5V
$define LCD_OSC_120				$00  ' Setting (ST7032)			
$define LCD_OSC_133				$01  ' Setting (ST7032)			
$define LCD_OSC_149				$02  ' Setting (ST7032)			
$define LCD_OSC_167				$03  ' Setting (ST7032)			
$define LCD_OSC_192				$04  ' Setting (ST7032)* Default			
$define LCD_OSC_227				$05  ' Setting (ST7032)			
$define LCD_OSC_277				$06  ' Setting (ST7032)			
$define LCD_OSC_347				$07  ' Setting (ST7032)
$endif
'
$define ICON_RAMAddressSet		$40  ' [13]Command (ST7032)		
'
$define LCD_ICON_Contrast_HIGH_Byte  $50   '[14] Command
$define LCD_ICON_ON				     $08   ' Setting, use | function
$define LCD_ICON_OFF				 $F7   ' Setting, use & function
$define LCD_Booster_ON				 $04   ' Setting, use | function *
$define LCD_Booster_OFF				 $FB   ' Setting, use & function 
$define LCD_Contrast_HIGH_Byte_MASK  $03   ' Only used for bit masking (ST7032)
'
$define LCD_Follower_Control		 $60   ' [15] Command (ST7032)
$define LCD_Follower_ON				 $08   ' Setting (ST7032)*			
$define LCD_Follower_OFF			 $F7   ' Setting (ST7032)			
$define LCD_Rab_1_00				 $00   ' Setting (ST7032)			
$define LCD_Rab_1_25				 $01   ' Setting (ST7032)			
$define LCD_Rab_1_50				 $02   ' Setting (ST7032)			
$define LCD_Rab_1_80				 $03   ' Setting (ST7032)			
$define LCD_Rab_2_00				 $04   ' Setting (ST7032)*			
$define LCD_Rab_2_50				 $05   ' Setting (ST7032)			
$define LCD_Rab_3_00				 $06   ' Setting (ST7032)			
$define LCD_Rab_3_75				 $07   ' Setting (ST7032)			
'
$define LCD_Contrast_LOW_Byte	     $70   ' [16] Command (ST7032)
$define LCD_Contrast_LOW_Byte_MASK   $0F   ' Only used for bit masking (ST7032)
'
' Init Values after Reset
$ifdef LCDVoltage5V
$define LCD_FunctionSet_Basic_Init      %00111000
$define LCD_FunctionSet_Extended_Init   %00111001
$define LCD_Internal_OSC_Frequency_Init %00010100
$define LCD_Contrast_Set_Init           %01111001
$define LCD_Power_ICON_control_Init     %01010000
$define LCD_Follower_Control_Init       %01101100
$define LCD_Display_On_Init             %00001100
$define LCD_ShiftDisplayToRight_Init    %00000101 
$define LCD_Clear_Display_Init          %00000001
$endif
$ifdef LCDVoltage3V
$define LCD_FunctionSet_Basic_Init      %00111000
$define LCD_FunctionSet_Extended_Init   %00111001
$define LCD_Internal_OSC_Frequency_Init %00010100
$define LCD_Contrast_Set_Init           %01110100
$define LCD_Power_ICON_control_Init     %01010100
$define LCD_Follower_Control_Init       %01101111
$define LCD_Display_On_Init             %00001100
$define LCD_ShiftDisplayToRight_Init    %00000101 
$define LCD_Clear_Display_Init          %00000001
$endif
'
'======= End of command/flag defenitions =======================================
'-------------------------------------------------------------------------------
' Function Name     : Print Commands of the Proton Basic Compiler / Amicus18
' Routine Number    : X003  : Version Number 1.00
' Syntax            : Print LcdCmd,LcdHome  (Command of Proton Basic)
' Notes             : 
'
    Symbol LcdCmd =          $FE    ' Command mode byte
    ' Now Setting Byte
    Symbol LcdClr =          $01    ' Clear Display
    Symbol LcdHome =         $02    ' Return home (beginning of first line)
    Symbol LcdUnderlineOff = $0C    ' Underline Cursor Off
    Symbol LcdUnderlineOn =  $0E    ' Underline Cursor On
    Symbol LcdBlinkingOff =  $0C    ' Blinking Box Cursor Off
    Symbol LcdBlinkingOn =   $0F    ' Blinking Box Cursor On
    Symbol LcdCurMoveLeft =  $10    ' Move Cursor Left one position
    Symbol LcdCurMoveRight = $14    ' Move Cursor Right one position
    Symbol LcdHome2 =        $C0    ' Move cursor to beginning of second line
    Symbol LcdScrollRight =  $1C    ' Scroll entire screen Right one space
    Symbol LcdScrollLeft =   $18    ' Scroll entire screen Left one space
    Symbol LcdDisplayOff =   $08    ' Turn visible LCD off
    Symbol LcdDisplayOn =    $0C    ' Turn visible LCD on
'
'-------------------------------------------------------------------------------
'===============================================================================    
'-------------------------------------------------------------------------------
'*******************************************************************************
' Function Name     : Bypass the compiler's PRINT library subroutine in favour 
'                   : of this subroutine
' Routine Number    : X005  : Version Number 1.00
' Syntax            :  
' Notes             : The code is fully compatible with the Cls, Print and Cursor 
'                   : commands of Proton Basic
' The PRINT subroutine expects the character to display to be in WREG
'
#disable Cls        ' Disable the Cls Low level routine
#Disable Cursor     ' Disable the Cursor Low level routine
#Disable Print      ' Disable the Print Low level routine
'
Asm
LCD@CLS                             ; CLS routine
    GoTo mInitLCD_Sub               ; Running the 'Cls' command
;
LCD@CRS                             ; Cls and Print At,x,y, Routine
    Movwf   CRS_Col,0               ; Load address DDRAM value
    $ifdef LCD_DEBUG
    Movwf   CRS_Col_Start,0         ; Load Column value to DEBUG
    $endif
;    Movff   BPFH,CRS_First          ; CRS routine: Print AT Line,Column: ???
                                    ; Not use : always = value Dec 128 (What is the use?)
EndAsm
    GoTo    mLCDCRS_Sub             ; compute the 'Cls' and 'Print At' cursor commands 

Asm
LCD@CUR                             ; Cursor Line,Column, Cursor routine
    Movwf   CUR_Col,0               ; Load Column value
    $ifdef LCD_DEBUG
    Movwf   CUR_Col_Start,0         ; Load Column value to DEBUG    
    $endif
    Movff   GEN,CUR_Row             ; Load Row value
    $ifdef LCD_DEBUG
    Movff   GEN,CUR_Row_Start       ; Load Row value To Debug            
    $endif
EndAsm
    GoTo    mLCDCUR_Sub             ; compute the 'Cursor' command    

Asm
Print                               ; Print routine
    Movwf   LCD_DataInp             ; Data to print to the display. Not used in my subroutines.
EndAsm
    Wreg_Byte LCD_ByteToWrite           ' Data to print to the display , Byte used in my subroutines.
    If CommandFlag = 0 Then             ' Test Print mode
        If LCD_DataInp = $FE Then       ' $FE = Command. Detect the syncro byte for a next data command
            CommandFlag = 1             ' If the first byte is $FE                     
            GoTo PrintOut11             ' Exit the print routine and wait the second byte
        Else                            ' Command mode
            GoSub LCD_WriteByte_Sub     ' This byte is a data byte
        EndIf
    Else
        CommandFlag = 0                 ' Clear the Command Flag, the next byte would be
                                        ' a data byte if <> $FE (is it correct?)             
        GoSub LCD_WriteCmd_Sub          ' This byte would be a instruction Byte
    EndIf
PrintOut11:
Asm
    Movf LCD_DataInp,W                  ' Original input Print Data
EndAsm
    Return
'
'*******************************************************************************
'*******************************************************************************
' LCD SUBROUTINES AND MACROS:
'-------------------------------------------------------------------------------
' Function Name     : Write one Instruction Byte to the I2C Bus
' Routine Number    : X010  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Gosub LCD_WriteInstruction_Sub
' Notes             : 
'
LCD_WriteInstruction_Sub:        
    HBStart                     ' Send a Start condition
    HBusOut AddressST7032       ' Address of the LCD, and send a Flag Write command
    HBusOut %00000000           ' Send the last command byte
    HBusOut LCD_InstToWrite     ' Send the instruction byte
    HBStop                      ' Send a Stop condition        
    Return
'-------------------------------------------------------------------------------
' Function Name     : Write one Data Byte to the I2C Bus
' Routine Number    : X011  : Version Number 1.00
' Input Variables   : pValue
' Output Variables  : 
' Syntax            : Print_Byte(ASCII)
' Notes             : Send the ASCII byte to the I2C Bus
'
$define Print_Byte(pValue) mPrint_Byte pValue

mPrint_Byte Macro pValue
    #if (Prm_Count > 1)
	   #error "mPrint_Byte - Too many parameters"
       Exitm
    #else
        #if (Prm_Count < 1)
            #error "mPrint_Byte - Too few parameters"
            Exitm
        #else
            #if(Prm_1 != Byte) && (Prm_1 != Num8)
                #error "mPrint_Byte - (Param1) Should be a Byte or Num8"
                Exitm
            #else
                #if (Prm_1 == Byte) 
                    Byte_Byte pValue,LCD_ByteToWrite         
                #endif
                #if (Prm_1 == Num8) 
                    Num_Byte pValue,LCD_ByteToWrite         
                #endif
                GoSub LCD_WriteByte_Sub
            #endif
        #endif
    #endif
Endm
'
LCD_WriteByte_Sub:
    HBStart                     ' Send a Start condition
    HBusOut AddressST7032       ' Address of the LCD, and send a Write command
    HBusOut %01000000           ' The next byte will be a Data
    HBusOut LCD_ByteToWrite     ' Send the Data byte
    HBStop                      ' Send a Stop condition        
    DelayUS 30
    Return
'*******************************************************************************
'*******************************************************************************
'-------------------------------------------------------------------------------
' Function Name     : SetCursor Macro
' Routine Number    : X020  : Version Number 1.00
' Input Variables   : CUR_Row, CUR_Col
' Output Variables  : Return_Var
' Syntax            : SetCursor(Row, Col)
' Notes             : Set cursor position
'
'SetCursor(Row, Col) 
' Delayus 30
' we count rows starting w/0
'$define SetCursor(pValue1, pValue2)  mSetCursor pValue1,pValue2

mSetCursor Macro pValue1,pValue2
    #if (Prm_Count > 2)
	   #error "mSetCursor - Too many parameters"
       Exitm
    #else
        #if (Prm_Count < 2)
            #error "mSetCursor - Too few parameters"
            Exitm
        #else
            #if(Prm_1 != Byte) && (Prm_1 != Num8)
                #error "mSetCursor - (Param1) Should be a Byte or Num8"
                Exitm
            #endif
            #if(Prm_2 != Byte) && (Prm_2 != Num8)
                #error "mSetCursor - (Param2) Should be a Byte Or Num8"
                Exitm                
            #else
                #if (Prm_1 == Byte) 
                    Byte_Byte pValue1,CUR_Row         
                #endif
                #if (Prm_1 == Num8) 
                    Num_Byte pValue1,CUR_Row         
                #endif
                #if (Prm_2 == Byte) 
                    Byte_Byte pValue2,CUR_Col         
                #endif   
                #if (Prm_2 == Num8) 
                    Num_Byte pValue2,CUR_Col         
                #endif                
                GoSub mSetCursor_Sub
                #if (mSetCursor_RETURN != 1) 
                    #error "mSetCursor - Mandatory return parameter misssing" 
                #else 
                    #if (Return_Type != Byte)
                        #error "mSetCursor - Return variable should be a Byte variable"
                        Exitm 
                    #else 
                        #if (Return_Type == Byte) 
                            Byte_Byte LCD_InstToWrite, Return_Var
                        #endif 
                    #endif 
                #endif 
            #endif
        #endif
    #endif
Endm

'#ifdef mSetCursor#REQ
mSetCursor_Sub:
    CUR_Col = CUR_Col - 1
    If CUR_Row = 2 Then
        LCD_InstToWrite = LCD_SetDDRAMADDR | $40 | CUR_Col  ' 2nd Row
    Else
        LCD_InstToWrite = LCD_SetDDRAMADDR | $00 | CUR_Col  ' 1st Row         
    EndIf    
    Return    
'#endif
' 
'*******************************************************************************
'*******************************************************************************
' LCD LIBRARY:
'Basic Functions Library
'-------------------------------------------------------------------------------
' Function Name     : Function Set 2 Lines basic
' Routine Number    : X021  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : FunctionSet_Basic()
' Notes             : Send the FunctionSet_Basic to the I2C Bus
' [5]A
$define FunctionSet_2Line_Basic() mFunctionSet_2LBasic

mFunctionSet_2LBasic Macro
    GoSub mFunctionSet_2LBasic_Sub
Endm

#ifdef mFunctionSet_2LBasic#REQ
mFunctionSet_2LBasic_Sub:
    LCD_InstToWrite = LCD_FunctionSet_2Line_Basic
    GoSub LCD_WriteInstruction_Sub
    LCD_FunctionSet_IS = 0
    DelayUS 30
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Function Set 2 Lines extended
' Routine Number    : X022  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : FunctionSet_Extended()
' Notes             : Send the 'FunctionSet_Extended' to the I2C Bus
' [5]B   
$define FunctionSet_2Line_Extended() mFunctionSet_2LExtended

mFunctionSet_2LExtended Macro
    GoSub mFunctionSet_2LExtended_Sub
Endm

#ifdef mFunctionSet_2LExtended#REQ
mFunctionSet_2LExtended_Sub:
    LCD_InstToWrite = LCD_FunctionSet_2Line_Extended
    GoSub LCD_WriteInstruction_Sub
    LCD_FunctionSet_IS = 1
    DelayUS 30 
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Follower control
' Routine Number    : X023  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Follower_Control()
' Notes             : Send the 'Follower_Control' to the I2C Bus
' [15]              : Runs "FunctionSet_Extended()" first
' 
$define Follower_Control() mFollower_Control

mFollower_Control Macro
    GoSub mFollower_Control_Sub
Endm

#ifdef mFollower_Control#REQ
mFollower_Control_Sub:
    If  LCD_FunctionSet_IS = 0 Then
        'LCD_InstToWrite = LCD_FunctionSet_2Line_Extended
        'GoSub LCD_WriteInstruction_Sub
        'LCD_FunctionSet_IS = 1
        'DelayUS 30
        GoSub mFunctionSet_2LExtended_Sub
    EndIf
$ifdef LCDVoltage5V
'Manufacturer value
    LCD_InstToWrite = LCD_Follower_Control | LCD_Rab_2_00 | LCD_Follower_ON
'    LCD_InstToWrite = (LCD_Follower_Control | LCD_Rab_2_00) & LCD_Follower_Off    
$endif
$ifdef LCDVoltage3V
'Manufacturer value
    LCD_InstToWrite = LCD_Follower_Control | LCD_Rab_3_75 | LCD_Follower_ON
'    LCD_InstToWrite = (LCD_Follower_Control | LCD_Rab_2_00) & LCD_Follower_OFF      
$endif        
    GoSub LCD_WriteInstruction_Sub
    'DelayUS 200
    DelayUS 30
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Internal osc
' Routine Number    : X024  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Internal_OSC_Frequency()
' Notes             : Send the 'Internal_OSC_Frequency' to the I2C Bus
' [12]              : Runs "FunctionSet_Extended()" first
' 
$define Internal_OSC_Frequency() mInternal_OSC

mInternal_OSC Macro
    GoSub mInternal_OSC_Sub
Endm
#ifdef mInternal_OSC#REQ
mInternal_OSC_Sub:
    If  LCD_FunctionSet_IS = 0 Then
        'LCD_InstToWrite = LCD_FunctionSet_2Line_Extended
        'GoSub LCD_WriteInstruction_Sub
        'LCD_FunctionSet_IS = 1
        'DelayUS 30
        GoSub mFunctionSet_2LExtended_Sub
    EndIf
$ifdef LCDVoltage5V
    LCD_InstToWrite = (LCD_Bias_OSC_Control | LCD_OSC_192) & LCD_Bias1_5 
$endif
$ifdef LCDVoltage3V
    LCD_InstToWrite = (LCD_Bias_OSC_Control | LCD_OSC_183) & LCD_Bias1_5 
$endif    
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Contrast Low nible
' Routine Number    : X025  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Contrast_Low_Nible()
' Notes             : Send the 'Contrast_Low_Nible' to the I2C Bus
' [16]              : Runs the "FunctionSet_Extended()" first
' 
$define Contrast_Low_Nible() mContrastLowNible

mContrastLowNible Macro
    GoSub mContrastLowNible_Sub
Endm

#ifdef mContrastLowNible#REQ
mContrastLowNible_Sub:
    If  LCD_FunctionSet_IS = 0 Then
        'LCD_InstToWrite = LCD_FunctionSet_2Line_Extended
        'GoSub LCD_WriteInstruction_Sub
        'LCD_FunctionSet_IS = 1
        'DelayUS 30
        GoSub mFunctionSet_2LExtended_Sub
    EndIf
    LCD_InstToWrite = LCD_Contrast_LOW_Byte | (Contrast & LCD_Contrast_LOW_Byte_MASK)
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Contrast High nible / icon / power
' Routine Number    : X026  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Contrast_High_Nible_Icon_Power()
' Notes             : Send the 'Contrast_High_Nible_Icon_Power' to the I2C Bus
' [14]              : Runs "FunctionSet_Extended()" first 
'
$define Contrast_High_Nible_Icon_Power() mContrastHighNibleIconPower

mContrastHighNibleIconPower Macro
    GoSub mContrast111_Sub
Endm

#ifdef mContrastHighNibleIconPower#REQ
    If  LCD_FunctionSet_IS = 0 Then
        'LCD_InstToWrite = LCD_FunctionSet_2Line_Extended
        'GoSub LCD_WriteInstruction_Sub
        'LCD_FunctionSet_IS = 1
        'DelayUS 30
        GoSub mFunctionSet_2LExtended_Sub
    EndIf
mContrast111_Sub:
    LCD_InstToWrite = (LCD_ICON_Contrast_HIGH_Byte & LCD_ICON_OFF & LCD_Booster_OFF) | ((Contrast >> 4) & LCD_Contrast_HIGH_Byte_MASK)
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    Return
#endif
'-------------------------------------------------------------------------------
'
'************* HIGH LEVEL COMMANDS *********************************************
'
'-------------------------------------------------------------------------------
' Function Name     : Clear display
' Routine Number    : X050  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Clear_Display()
' Notes             : Send the 'Clear_Display' to the I2C Bus
' [1]               : Clear display, Set Cursor position To zero
'                   : This command takes a long time! 2mS
' 
$define Clear_Display() mClear_Display

mClear_Display Macro
    GoSub mClear_Display_Sub
Endm

#ifdef mClear_Display#REQ
mClear_Display_Sub:
    LCD_InstToWrite = LCD_ClearDisplay
    GoSub LCD_WriteInstruction_Sub
    DelayMS 2
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Set Cursor position To zero
' Routine Number    : X051  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Return_Home()
' Notes             : Send the 'Return_Home' to the I2C Bus
' [2]               : This command takes a long time! 2mS
' OK
$define Return_Home() mReturn_Home

mReturn_Home Macro
    GoSub mReturn_Home_Sub
Endm

#ifdef mReturn_Home#REQ
mReturn_Home_Sub:
    LCD_InstToWrite = LCD_ReturnHome
    GoSub LCD_WriteInstruction_Sub
    DelayMS 2
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : SetCursor
' Routine Number    : X052  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Set_Cursor(Row, Col)
' Notes             : Send the 'Set_Cursor' to the I2C Bus
' [6]               : we count rows starting w/0            
'
$define Set_Cursor(pValue1,pValue2)    '
    CUR_Row = pValue1   '
    CUR_Col = pValue2   '
    GoSub mSetCursor_Sub    '        
    GoSub LCD_WriteInstruction_Sub '
    DelayUS 30
    
'-------------------------------------------------------------------------------
' Function Name     : Turn the display On
' Routine Number    : X053  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Display_On()
' Notes             : Send the 'Display_On' to the I2C Bus
' [4]A
$define Display_On() mDisplay_On

mDisplay_On Macro
    GoSub mDisplay_On_Sub
Endm

#ifdef mDisplay_On#REQ
mDisplay_On_Sub:
    _displaycontrol = _displaycontrol | LCD_Display_ON
    LCD_InstToWrite = LCD_DisplayControl | _displaycontrol
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Display_Off
' Routine Number    : X054  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Display_Off()
' Notes             : Send the 'Display_Off' to the I2C Bus
' [4]B   
$define Display_Off() mNO_Display

mNO_Display Macro
    GoSub mNO_Display_Sub
Endm

#ifdef mNO_Display#REQ
mNO_Display_Sub:
    _displaycontrol = _displaycontrol & LCD_Display_OFF
    LCD_InstToWrite = LCD_DisplayControl | _displaycontrol
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Turns the underline Cursor On
' Routine Number    : X055  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Cursor_On()
' Notes             : Send the 'Cursor_On' to the I2C Bus
' [4]          
$define Cursor_On() mCursor_On

mCursor_On Macro
    GoSub mCursor_On_Sub
Endm

#ifdef mCursor_On#REQ
mCursor_On_Sub:
    _displaycontrol = _displaycontrol | LCD_Cursor_ON
    LCD_InstToWrite = LCD_DisplayControl | _displaycontrol
    GoSub LCD_WriteInstruction_Sub    
    DelayUS 30
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Cursor_Off
' Routine Number    : X056  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Cursor_Off()
' Notes             : Send the 'Cursor_Off' to the I2C Bus
' [4]           
$define Cursor_Off() mNO_Cursor

mNO_Cursor Macro
    GoSub mNO_Cursor_Sub
Endm

#ifdef mNO_Cursor#REQ
mNO_Cursor_Sub:
    _displaycontrol = _displaycontrol & LCD_Cursor_OFF
    LCD_InstToWrite = LCD_DisplayControl | _displaycontrol
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Turn Cursor blinking On
' Routine Number    : X057  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Cursor_Blinking_On()
' Notes             : Send the 'Cursor_Blinking_On_Off' to the I2C Bus
' [4]           
$define Cursor_Blinking_On() mCursor_Blinking_On

mCursor_Blinking_On Macro
    GoSub mCursor_Blinking_On_Sub
Endm

#ifdef mCursor_Blinking_On#REQ
mCursor_Blinking_On_Sub:
    _displaycontrol = _displaycontrol | LCD_Blink_ON
    LCD_InstToWrite = LCD_DisplayControl | _displaycontrol
    GoSub LCD_WriteInstruction_Sub '
    DelayUS 30
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : NO Blink
' Routine Number    : X058  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : NO_Blink()
' Notes             : Send the 'NO_Blink' to the I2C Bus
' [4]       
$define NO_Blink() mNO_Blink

mNO_Blink Macro
    GoSub mNO_Blink_Sub
Endm

#ifdef mNO_Blink#REQ
mNO_Blink_Sub:
    _displaycontrol = _displaycontrol & LCD_Blink_OFF
    LCD_InstToWrite = LCD_DisplayControl | _displaycontrol
    GoSub LCD_WriteInstruction_Sub '
    DelayUS 30
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Clear Line 1
' Routine Number    : X059  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Clear_Line1()
' Notes             : Send the 'Clear_Line1' to the I2C Bus
' []                : This command takes a long time! 0,5mS
' Write a "SPACE" to the LCD column 1 to column 16, Row1, and place the cursor in column1, Row1.
$define Clear_Line1() mClear_Line1

mClear_Line1 Macro
    GoSub mClear_Line1_Sub
Endm

#ifdef mClear_Line1#REQ
mClear_Line1_Sub:
    Set_Cursor(1,1)         ' Set the Cursor to the line1, Column1
    LCD_Index = 0
    Repeat
        Print_Byte($20)     ' Send a SPACE to the LCD
        Inc LCD_Index       ' Increment the index
    Until LCD_Index > 15
    Set_Cursor(1,1)         ' Set the Cursor to the line1, Column1    
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Clear Line 2 
' Routine Number    : X060  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Clear_Line2()
' Notes             : Send the 'Clear_Line2' to the I2C Bus
' []                : This command takes a long time! 0,5mS
' Write a "SPACE" to the LCD column 1 to column 16, Row2, and place the cursor in column1, Row2.
$define Clear_Line2() mClear_Line2

mClear_Line2 Macro
    GoSub mClear_Line2_Sub
Endm

#ifdef mClear_Line2#REQ
mClear_Line2_Sub:
    Set_Cursor(2,1)         ' Set the Cursor to the Row2, Column1
    LCD_Index = 0
    Repeat
        Print_Byte($20)     ' Send a SPACE to the LCD
        Inc LCD_Index       ' Increment the index
    Until LCD_Index > 15
    Set_Cursor(2,1)         ' Set the Cursor to the Row2, Column1
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Clear number of Chareacters on line 1 or 2
' Routine Number    : X061  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Clear_LCD_Chrs(Row,Column,Characters)
' Notes             : Send the 'Clear_LCD_Chrs' to the I2C Bus
' []                : This command takes a long time! 0,5mS
$define Clear_LCD_Chrs(pValue1,pValue2,pValue3) mClear_Chrs pValue1,pValue2,pValue3

mClear_Chrs Macro pValue1,pValue2
    #if (Prm_Count > 3)
	   #error "mClear_Chrs - Too many parameters"
       Exitm
    #else
        #if (Prm_Count < 3)
            #error "mClear_Chrs - Too few parameters"
            Exitm
        #else
            #if(Prm_1 != Byte) && (Prm_1 != Num8)
                #error "mClear_Chrs - (Param1) Should be a Byte or Num8"
                Exitm
            #endif
            #if(Prm_2 != Byte) && (Prm_2 != Num8)
                #error "mClear_Chrs - (Param2) Should be a Byte Or Num8"
                Exitm
            #endif
            #if(Prm_3 != Byte) && (Prm_3 != Num8)
                #error "mClear_Chrs - (Param3) Should be a Byte Or Num8"
                Exitm                                
            #else
                #if (Prm_1 == Byte) 
                    Byte_Byte pValue1,aCUR_Row         
                #endif
                #if (Prm_1 == Num8) 
                    Num_Byte pValue1,aCUR_Row         
                #endif
                #if (Prm_2 == Byte) 
                    Byte_Byte pValue2,aCUR_Col         
                #endif   
                #if (Prm_2 == Num8) 
                    Num_Byte pValue2,aCUR_Col         
                #endif                 
                #if (Prm_3 == Byte) 
                    Byte_Byte pValue3,CUR_Chrs         
                #endif   
                #if (Prm_3 == Num8) 
                    Num_Byte pValue3,CUR_Chrs         
                #endif                
                GoSub mClear_Chrs_Sub
            #endif
        #endif
    #endif
Endm

#ifdef mClear_Chrs#REQ
mClear_Chrs_Sub:
    Set_Cursor(aCUR_Row,aCUR_Col)   ' Set the Cursor to the line= Row, Column1
    LCD_Index = 1
    Repeat
        Print_Byte($20)             ' Send a SPACE to the LCD
        Inc LCD_Index               ' Increment the index
    Until LCD_Index > CUR_Chrs      ' Until the last number of characters deleted
    Set_Cursor(aCUR_Row,aCUR_Col)   ' Set the Cursor to the Origin position.
    Return
#endif
  
'-------------------------------------------------------------------------------
' These commands scroll the display without changing the RAM
' LCD_FunctionSet
'-------------------------------------------------------------------------------
' Function Name     : scroll Cursor Left 
' Routine Number    : X070  : Version Number 1.00
' Input Variables   : 
' Output Variables  : Send the 'Scroll_Display_Left' to the I2C Bus
' Syntax            : Scroll_Display_Left()
' Notes             : Runs first IS = 0
' [10]A             : Runs "FunctionSet_Basic()" first
' Notes: Cursor Or Display Shift : Shift Cursor to the right AC = AC-1
'        Address= $01 : Bit3= S/C=0, Bit2= R/L=0, with Function Set: IS=0
' Without writing or reading of display data, shift right/left cursor position or display.
' This instruction is used to correct or search display data. During 2-line mode display,
' cursor moves the 2nd line after 40th digit of 1st line. Note that display shift is performed
' simultaneously in all line. When displayed data is shifted repeatedly, each line shifted
' individually. When display shift is performed, the contents of address counter are not
' changed.
' 
$define Scroll_Cursor_Left() mScrollCursorLeft

mScrollCursorLeft Macro
    GoSub mScrollCursorLeft_Sub
Endm

#ifdef mScrollCursorLeft#REQ    
mScrollCursorLeft_Sub:
' Runs "FunctionSet_Basic()" first
    If  LCD_FunctionSet_IS = 1 Then
        'LCD_InstToWrite = LCD_FunctionSet_2Line_Basic
        'GoSub LCD_WriteInstruction_Sub
        'LCD_FunctionSet_IS = 0
        'DelayUS 30
        GoSub mFunctionSet_2LBasic_Sub
    EndIf
    LCD_InstToWrite = LCD_DisplayCursorSHIFT | LCD_CursorToLeft
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30    
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : scroll Cursor Right 
' Routine Number    : X071  : Version Number 1.00
' Input Variables   : 
' Output Variables  : Send the 'Scroll_Display_Right' to the I2C Bus
' Syntax            : Scroll_Display_Right()
' Notes             : IS = 0
' [10]B             : Runs "FunctionSet_Basic()" first
' Notes: Cursor Or Display Shift : Shift Cursor to the right AC = AC+1
'        Address= $01 : Bit3= S/C=0, Bit2= R/L=1, with Function Set: IS=0
' Without writing or reading of display data, shift right/left cursor position or display.
' This instruction is used to correct or search display data. During 2-line mode display,
' cursor moves the 2nd line after 40th digit of 1st line. Note that display shift is performed
' simultaneously in all line. When displayed data is shifted repeatedly, each line shifted
' individually. When display shift is performed, the contents of address counter are not
' changed.
' 
$define Scroll_Cursor_Right() mScrollCursorRight

mScrollCursorRight Macro
    GoSub mScrollCursorRight_Sub
Endm

#ifdef mScrollCursorRight#REQ    
mScrollCursorRight_Sub:
' Runs "FunctionSet_Basic()" first
    If  LCD_FunctionSet_IS = 1 Then
        'LCD_InstToWrite = LCD_FunctionSet_2Line_Basic
        'GoSub LCD_WriteInstruction_Sub
        'LCD_FunctionSet_IS = 0
        'DelayUS 30
        GoSub mFunctionSet_2LBasic_Sub
    EndIf
    LCD_InstToWrite = LCD_DisplayCursorSHIFT | LCD_CursorToRight
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30    
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : scroll Display Left 
' Routine Number    : X072  : Version Number 1.00
' Input Variables   : 
' Output Variables  : Send the 'Scroll_Display_Left' to the I2C Bus
' Syntax            : Scroll_Display_Left()
' Notes             : IS = 0, Runs "FunctionSet_Basic()" first
' [10]A             : Set the moving direction of cursor and display
' I/D: Increment/Decrement of DDRAM address (cursor or blink)
'   When I/D = 1, cursor/blink moves to right and DDRAM address is increased by 1.
'   When I/D = 0, cursor/blink moves to left and DDRAM address is decreased by 1.
'       CGRAM operates the same as DDRAM, when read from or write to CGRAM.
' S: Shift entire display
'   When DDRAM read (CGRAM read/write) operation or S=0, shift of entire display is not
'   performed.
' If S=1 and DDRAM write operation, shift entire display is performed according to I/D value
' (I/D=1: shift left, I/D=0: shift right).
'
$define Scroll_Display_Left() mScrollDisplayLeft

mScrollDisplayLeft Macro
    GoSub mScrollDisplayLeft_Sub
Endm

#ifdef mScrollDisplayLeft#REQ    
mScrollDisplayLeft_Sub:
' Runs "FunctionSet_Basic()" first
    If  LCD_FunctionSet_IS = 1 Then
        'LCD_InstToWrite = LCD_FunctionSet_2Line_Basic
        'GoSub LCD_WriteInstruction_Sub
        'LCD_FunctionSet_IS = 0
        'DelayUS 30
        GoSub mFunctionSet_2LBasic_Sub
    EndIf
    LCD_InstToWrite = LCD_DisplayCursorSHIFT | LCD_DisplayToLeft
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30    
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Scroll Display Right
' Routine Number    : X073  : Version Number 1.00
' Input Variables   : 
' Output Variables  : Send the 'Scroll_Display_Right' to the I2C Bus
' Syntax            : Scroll_Display_Right()
' Notes             : IS = 0, Runs "FunctionSet_Basic()" first
' [10]B             : Set the moving direction of cursor and display
' I/D: Increment/Decrement of DDRAM address (cursor or blink)
'   When I/D = 1, cursor/blink moves to right and DDRAM address is increased by 1.
'   When I/D = 0, cursor/blink moves to left and DDRAM address is decreased by 1.
'       CGRAM operates the same as DDRAM, when read from or write to CGRAM.
' S: Shift entire display
'   When DDRAM read (CGRAM read/write) operation or S=0, shift of entire display is not
'   performed.
' If S=1 and DDRAM write operation, shift entire display is performed according to I/D value
' (I/D=1: shift left, I/D=0: shift right).
'
$define Scroll_Display_Right() mScrollDisplayRight

mScrollDisplayRight Macro
    GoSub mScrollDisplayRight_Sub
Endm

#ifdef mScrollDisplayRight#REQ
mScrollDisplayRight_Sub:
    If  LCD_FunctionSet_IS = 1 Then
        'LCD_InstToWrite = LCD_FunctionSet_2Line_Basic
        'GoSub LCD_WriteInstruction_Sub
        'LCD_FunctionSet_IS = 0
        'DelayUS 30
        GoSub mFunctionSet_2LBasic_Sub
    EndIf
    LCD_InstToWrite = LCD_DisplayCursorSHIFT | LCD_DisplayToRight
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30   
    Return
#endif
'-------------------------------------------------------------------------------
' END: LCD_FunctionSet
'-------------------------------------------------------------------------------
' Function Name     : ' Display_To_Left
' Routine Number    : X074  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : EntryModeSet_Display_To_Left()
' Notes             : Send the 'Display_To_Left' to the I2C Bus
' [3]A                LCD_EntryModeSet
'
$define EntryModeSet_Display_To_Left() mDisplayToLeft

mDisplayToLeft Macro
    GoSub mDisplayToLeft_Sub
Endm

#ifdef mDisplayToLeft#REQ
mDisplayToLeft_Sub:
    LCD_InstToWrite = LCD_ShiftDisplayToLeft
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    Return    
#endif
'-------------------------------------------------------------------------------
' Function Name     : Display_To_Right
' Routine Number    : X075  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : EntryModeSet_Display_To_Right()
' Notes             : Send the 'Display_To_Right' to the I2C Bus
' [3]B                LCD_EntryModeSet
'
$define EntryModeSet_Display_To_Right() mDisplayToRight

mDisplayToRight Macro
    GoSub mDisplayToRight_Sub
Endm

#ifdef mDisplayToRight#REQ   
mDisplayToRight_Sub:    
    LCD_InstToWrite = LCD_ShiftDisplayToRight
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    Return    
#endif
'
'-------------------------------------------------------------------------------
' Function Name     : 
' Routine Number    : X076  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : 
' Notes             : Here, your library code
'

' 
'*******************************************************************************
' NO WRITE MACRO LIBRARY BELOW THIS LINE
'*******************************************************************************
'*******************************************************************************
'-------------------------------------------------------------------------------
' Function Name     : CLS_LCD()
' Routine Number    : X100  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : CLS_LCD()
' Notes             : Initialization routine same as X101 routine, but with macros.
'  
$define CLS_LCD() mCLS_Display

mCLS_Display Macro
    GoSub Library_InitLCD_Sub
Endm

#ifdef mCLS_Display#REQ
Library_InitLCD_Sub:
' Initialization of the I2C LCD
    FunctionSet_2Line_Basic()    
    FunctionSet_2Line_Extended()
    Internal_OSC_Frequency()
    Contrast_Low_Nible()    
    Contrast_High_Nible_Icon_Power()
    Follower_Control()    
    FunctionSet_2Line_Basic()
    Display_On()
    EntryModeSet_Display_To_Right()
    Scroll_Cursor_Right()
    Clear_Display()    
    LCD_Initialized = 1
    Return
#endif
'-------------------------------------------------------------------------------
' Function Name     : Init LCD manufacturer method 
' Routine Number    : X101  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : gosub Manufacturer_InitLCD_Sub
' Notes             : Included version for +5V and +3V voltage LCD power input
'  
Manufacturer_InitLCD_Sub:
$ifdef LCDVoltage5V
    LCD_InstToWrite = %00111000     '$38: Function Set: DL=1,N=1; DH=0,IS=0   
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    LCD_InstToWrite = %00111001     '$39: Function Set: DL=1,N=1; DH=0,IS=1; Function Set
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    LCD_InstToWrite = %00010100     '$14: Internal OSC Frequency;BS=0 1/5Bias;F2=0;F1=1;F0=0
    GoSub LCD_WriteInstruction_Sub  'Frequency = 192Hz
    DelayUS 30
    LCD_InstToWrite = %01111001     '$79: Contrast Set:C3=1 ;C2=0 ;C1=0 ;C0=1  
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    LCD_InstToWrite = %01010000     '$50: Power/ICON control/Contrast Set:Ion=0;Bon=0;C5=0;C4=0;
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    LCD_InstToWrite = %01101100     '$6C: Follower Control:Fon=1 On;Rab2=1;Rab1=0;Rab0=0;
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    LCD_InstToWrite = %00001100     '$0C: Display ON/OFF:D=1 entire display ON;C=0 Cursor OFF;B=0 cursor position OFF    
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    LCD_InstToWrite = %00111000     '$38: Function Set: N=2; 0x38:IS=0   
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    'New value
    LCD_InstToWrite = %00000110     '$05: EntryMode Set:I/D=1; S=0 NO Shift Entire Display (nuevo)
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    'New command
    LCD_InstToWrite = %00010100     'Cursor Or Display Shift : Shift Cursor To the right AC = AC+1
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    '    
    LCD_InstToWrite = %00000001     '$01: Clear Display:
    GoSub LCD_WriteInstruction_Sub
    LCD_Initialized = 1
    DelayUS 1200
    Return
$endif
'
$ifdef LCDVoltage3V
    LCD_InstToWrite = %00111000     '$38: Function Set: N=2; 0x38:IS=0   
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    LCD_InstToWrite = %00111001     '$39: Function Set: N=2; 0x39:IS=1;Function Set
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    LCD_InstToWrite = %00010100     '$14: Internal OSC Frequency;BS=0 1/5Bias;F2=0;F1=1;F0=0
    GoSub LCD_WriteInstruction_Sub  'Frequency = 183Hz
    DelayUS 30
    LCD_InstToWrite = %01110100     '$74: Contrast Set:C3=1 ;C2=0 ;C1=0 ;C1=1   
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    LCD_InstToWrite = %01010100     '$54: Power/ICON control/Contrast Set:Ion=0;Bon=0;C5=0;C4=0;
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    LCD_InstToWrite = %01101111     '$6F: Follower Control:Fon=1 On;Rab2=1;Rab1=0;Rab0=0;
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    LCD_InstToWrite = %00001100     '$0C: Display ON/OFF:D=1 entire display ON;C=0 Cursor OFF;B=0 cursor position OFF    
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    LCD_InstToWrite = %00111000     '$38: Function Set: N=2; 0x38:IS=0   
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    LCD_InstToWrite = %00000110     '$05: EntryMode Set: I/D= 1; S= 0 NO Shift Entire Display (nuevo) 
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    'New command
    LCD_InstToWrite = %00010100     'Cursor Or Display Shift : Shift Cursor To the right AC = AC+1
    GoSub LCD_WriteInstruction_Sub
    DelayUS 30
    '    
    LCD_InstToWrite = %00000001     '$01: Clear Display:
    GoSub LCD_WriteInstruction_Sub
    LCD_Initialized = 1
    DelayUS 1200
    Return
$endif
'-------------------------------------------------------------------------------
' PRINT PROTON SUBROUTINES:
'-------------------------------------------------------------------------------
' Function Name     : Compute the Address of DDRAM send by the "Cls" or "Print AT" 
'                   : commands send by the LCD@CRS routine.
' Routine Number    : X102  : Version Number 1.00
' Input Variables   : CRS_Col
' Output Variables  : LCD_InstToWrite
' Syntax            : Gosub mLCDCRS_Sub
' Notes             : 
'
mLCDCRS_Sub:
    CUR_Col = CRS_Col & %01111111                   ' Filter the 128 value
    LCD_InstToWrite = LCD_SetDDRAMADDR | CUR_Col    ' Compute the byte to Send       
    GoTo mLCDCUR2_Sub                               ' to the LCD.
'-------------------------------------------------------------------------------
' Function Name     : Set_Cursor(Row, Col)
' Routine Number    : X103  : Version Number 1.00
' Input Variables   : CRS_Col, CUR_Row
' Output Variables  : 
' Syntax            : GoSub mSetCursor_Sub
' Notes             : Using the Proton Syntax
'
mLCDCUR_Sub:
    GoSub mSetCursor_Sub                    ' Compute Row and Column to DDRAM Address
mLCDCUR2_Sub:
    GoSub LCD_WriteInstruction_Sub          ' Send the byte command to the I2C Bus
    DelayUS 30    
    Return        
'-------------------------------------------------------------------------------
' Function Name     : Compute the compatibility of direct command like "Print LcdCmd,LcdHome"
' Routine Number    : X104  : Version Number 1.00
' Input Variables   : LCD_ByteToWrite
' Output Variables  : 
' Syntax            : Gosub LCD_WriteCmd_Sub
' Notes             : Send a command stream to the I2C bus
'
' Proton Print commands:
' Print LcdCmd,LcdClr
' Print LcdCmd,LcdHome
' Print LcdCmd,LcdBlinkingOn
' Print LcdCmd,Display_On
' Print LcdCmd,Display_Off
' Ans so on...
LCD_WriteCmd_Sub:
    Select LCD_ByteToWrite
        Case LcdClr
            Clear_Display()
        Case LcdHome
            Return_Home()
        Case LcdUnderlineOff
            Cursor_Off()
        Case LcdUnderlineOn
            Cursor_On()
        Case LcdBlinkingOff
            NO_Blink()
        Case LcdBlinkingOn
            Cursor_Blinking_On()
        Case LcdCurMoveLeft
            Scroll_Display_Left()
        Case LcdCurMoveRight
            Scroll_Display_Right()
        Case LcdHome2
            CUR_Col = 1         ' Define the Column
            CUR_Row = 2         ' Define the Row
            GoSub mLCDCUR_Sub
        Case LcdScrollRight
            Scroll_Display_Right()
        Case LcdScrollLeft
            Scroll_Display_Left()
        Case LcdDisplayOff
            Display_Off()    
        Case LcdDisplayOn
            Display_On()
    EndSelect
    Return
'-------------------------------------------------------------------------------
'*******************************************************************************
' PLACE THIS MACRO IN THE INIT FILE OF THE PROYECT:
' Syntax : Init_LCD_ST7032i() or CLs (Proton Command)
'-------------------------------------------------------------------------------
' Function Name     : Init the LCD in main code
' Routine Number    : X105  : Version Number 1.00
' Input Variables   : 
' Output Variables  : 
' Syntax            : Init_LCD_ST7032i() or Cls
' Notes             : 
'  
$define Init_LCD_ST7032i() GoSub mInitLCD_Sub

mInitLCD_Sub:
    LCD_Flags = 0
    Contrast = 0            ' In case of error
'    
'Sample for ICON ON : (LCD_ICON_ON << 4)
'Sample for LCD_Booster ON : (LCD_Booster_ON << 4)
'
$ifdef LCDVoltage5V
' Manufacturer setting
' Low nibble: Contrast set Low byte
' High nibble: Contrast set High byte
' Initialization Values from Manucfaturer
'
    Contrast = $09 & LCD_ICON_OFF & LCD_Booster_OFF          'NO ICON , NO Booster
$endif
$ifdef LCDVoltage3V
' Manufacturer setting
' Low nibble: Contrast set Low byte
' High nibble: Contrast set High byte
' Initialization Values
'
    Contrast = $04 | LCD_ICON_OFF | (LCD_Booster_ON << 4)    'NO ICON, Booster Yes
$endif
'
' Initialization of the basic parameters first
'    _displaycontrol = 0
    _displaycontrol = LCD_Display_ON ' [4] Setting.
'
' Initialization of the I2C LCD
    GoTo Manufacturer_InitLCD_Sub
'    goto Library_InitLCD_Sub            ' Same with macros
'    GoSub Manufacturer_InitLCD_Sub
'    Return
'
'*******************************************************************************
'*******************************************************************************
' Define Library _ST7032i_
$endif 
    
OverDriverST7032:

