;*******************************************************************************
;PROGRAM			:	V/F control of 3-Phase Induction Motor
;MICROCONTROLLER	:	PIC18Fxx31	
;*******************************************************************************
;*******************************************************************************
;AUTHOR		:	Jon Burroughs, 
;			:	based on V/f code by Padmaraja Yedamale
;			:	Microchip Technology Inc
;DATE		:	2/11/04
;Version	:	V1.0
;*******************************************************************************
;Description:
;
;This code implements V/f control of a three-phase induction motor using
;the PIC18F4431 Microcontroller.  This code accompanies the application note
;AN890 "3-Phase AC-Induction Motor Control Using the PIC18F4431"
;
;By varying the frequency of supply to the Induction motor we can control the 
;speed of the motor. But by varying only frequency, changes the flux
;also which results in changes in torque. So to keep the flux, hence
;torque, the magnitude of voltage applied also needs to be changed. Or in other
;words, the ratio of V/F needs to be kept constant throughout the
;operating range.
;
;*******************************************************************************
;                    Software License Agreement                       
;                                                                     
; The software supplied herewith by Microchip Technology              
; Incorporated (the "Company") for its PICmicro Microcontroller is   
; intended and supplied to you, the Companys customer, for use       
; solely and exclusively on Microchip PICmicro Microcontroller        
; products. The software is owned by the Company and/or its           
; supplier, and is protected under applicable copyright laws. All     
; rights are reserved. Any use in violation of the foregoing          
; restrictions may subject the user to criminal sanctions under       
; applicable laws, as well as to civil liability for the breach of    
; the terms and conditions of this license.                           
;					                              
; THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES,   
; WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED   
; TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 	      
; PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,   
; IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 	      
; CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.	              
;                                                                     
;*******************************************************************************
	include		<p18f4431.inc>
	include		<3im_vf.inc>
;*******************************************************************************

	__CONFIG _CONFIG1H, 0x02 ;_OSC_HS_1H &_FCMEN_OFF_1H&_IESO_OFF_1H
	__CONFIG _CONFIG2L, 0x0C ;_PWRTEN_ON_2L & _BOREN_ON_2L & _BORV_20_2L  
	__CONFIG _CONFIG2H, 0x3E ;_WDTEN_OFF_2H
	__CONFIG _CONFIG3L, 0x3C ;0x24 ;_PWMPIN_OFF_3L & _LPOL_LOW_3L & _HPOL_LOW_3L & _GPTREN_ON_3L
	__CONFIG _CONFIG3H, 0x9D ;_FLTAMX_RC1_3H & _PWM4MX_RB5_3H
	__CONFIG _CONFIG4L, 0x7b 
	__CONFIG _CONFIG5L, 0x0F 
	__CONFIG _CONFIG5H, 0xC0  
	__CONFIG _CONFIG6L, 0x0F 
	__CONFIG _CONFIG6H, 0xE0 
	__CONFIG _CONFIG7L, 0x0F 
	__CONFIG _CONFIG7H, 0x40  


;FLAGS bits
#define TIMER0_OV_FLAG	0
#define	FLAG_FAULT		1
#define	PARAM_DISPLAY	2
#define	POSITION_BIT 	3
#define OFFSET1_FLAG	4
#define OFFSET2_FLAG	5
#define OFFSET3_FLAG	6
#define	MOTOR_DIRECTION	7

;FLAGS1 bits
#define	DEBOUNCE		0
#define	KEY_RS			1
#define	KEY_FR			2
#define	KEY_PRESSED 	3
#define	RUN_STOP		4
#define	FWD_REV			5
#define	FREQ_UPDATE		6
#define	FEEDBACK_UPDATE	7

;Keys parameters
#define KEY_PORT 		PORTD
#define RUN_STOP_KEY	6
#define FWD_REV_KEY		7
#define DEBOUNCE_COUNT	0xFF

;Delay parameters
#define	DELAY_COUNT1	0xFF
#define	DELAY_COUNT2	0xFF

;LED parameters
#define LED_PORT 		PORTD
#define RUN_STOP_LED	1
#define FWD_REV_LED		2

;Duty cycle limit definition, for 20KHz @20MHz, 2uS deadtime
#define	MINH_DUTY_CYCLE	0x00	;minimum duty cycle corresponds to 3 x deadtime = 3 x 2uS = 6uS  PDC = 6uS/(4/Fosc)
#define	MINL_DUTY_CYCLE	0x3C	;	try doubling from 0x1E	;
#define	MAXL_DUTY_CYCLE	0xE0	;maximum duty cycle is 4 x PTPER
#define	MAXH_DUTY_CYCLE	0x03

;Duty cycle limit definition, for 16KHz @20MHz, 2uS deadtime
;#define	MINH_DUTY_CYCLE	0x00	;minimum duty cycle corresponds to 3 x deadtime = 3 x 2uS = 6uS  PDC = 6uS/(4/Fosc)
;#define	MINL_DUTY_CYCLE	0x1E	;
;#define	MAXL_DUTY_CYCLE	0xDC	;maximum duty cycle is 4 x PTPER
;#define	MAXH_DUTY_CYCLE	0x04


;*******************************************************************************
;RAM locations in Access bank, uninitialized
;*******************************************************************************
	UDATA_ACS 	
TABLE_OFFSET1		res	1		;Phase1 offset to the Sine table(0)
TABLE_OFFSET2		res	1		;Phase2 offset to the Sine table(120)
TABLE_OFFSET3		res	1		;Phase3 offset to the Sine table(240)
COUNTER				res	1		;General counters
COUNTER1			res	1
COUNTER_SP			res	1
COUNTER_SP1			res	1
FLAGS				res	1		;Flags registers used to indicate different status
FLAGS1				res	1
FREQ_REF_H			res	1		;Reference Frequency input in counts
FREQ_REF_L			res	1
FREQUENCY			res	1
TEMP				res	1
TEMP1				res	1
TEMP_LOCATION		res	2
DEBOUNCE_COUNTER	res	1
PDC0L_TEMP			res	1
PDC0H_TEMP			res	1
PDC1L_TEMP			res	1
PDC1H_TEMP			res	1
PDC2L_TEMP			res	1
PDC2H_TEMP			res	1
CURRENT				res 1
TEMPERATURE			res 1
VELOCITY_HIGH		res 1
VELOCITY_LOW		res 1

SINE_TABLE			res	0x14	;Sine table

temp				res 1
temp1				res 1


;*******************************************************************************
;*******************************************************************************
;						RESET AND INTERRUPT VECTORS
;*******************************************************************************
;*******************************************************************************
STARTUP	code 0x00
	goto	Start				;Reset Vector address 
	
	CODE	0x08
	goto	ISR_HIGH			;High priority ISR at 0x0008

PROG_LOW	CODE	0x018
	nop
	goto	ISR_LOW				;Low priority ISR at 0x0018



;*******************************************************************************
;*******************************************************************************
;							INITIALIZATION
;*******************************************************************************
;*******************************************************************************
PROG1	code
Start

	clrf	FREQUENCY
	clrf	FLAGS
	clrf	FLAGS1

;	clrf	TRISD
;	clrf	PORTD	
;	bsf		PORTD, 0
;	bsf		PORTD, 1
;	goto	$-2

	call	INIT_HSADC
	call	INIT_PCPWM
	call	INIT_QEI
	call	INIT_TMR0
	call	INIT_PORTC
	call	INIT_PORTD
	call	COPY_TABLE_TO_RAM
	call	INIT_MOTOR_START
	
WAIT_HERE
	call	KEY_CHECK				;wait until key is pressed before initializing motor
	btfss	FLAGS1,KEY_PRESSED
	bra		WAIT_HERE
	
	bcf		PORTC,0
	bcf		PORTD,0
	bcf		PORTD,1
	bcf		PORTD,2
;	clrf	FLAGS
	clrf	FLAGS1
	

	call	INIT_INTERRUPTS





;*******************************************************************************	
;*******************************************************************************
; 								MAIN LOOP
;*******************************************************************************
;*******************************************************************************
MAIN_LOOP
	btfss	FLAGS,TIMER0_OV_FLAG	;back from Timer0 overflow?
	bra		bypass					;No
	call	UPDATE_PWM_DUTYCYCLES	;Yes, update the PWM duty cycle with new value
	call	UPDATE_TABLE_OFFSET		;Update 3 offsets
	bcf		FLAGS,TIMER0_OV_FLAG	;Clear the flag
bypass
	btfsc	FLAGS1,FREQ_UPDATE		;back from A/D overflow?
	call	CALCULATE_TIMER0_RELOAD ; only calculate new Timer0 reload if frequency has been updated

	btfss	ADCON0, GO				;If AD Conversion is complete
	bsf		ADCON0, GO				;	then start a new conversion

	btfss   FLAGS1, FEEDBACK_UPDATE
	call	CALCULATE_ACTUAL_SPEED

	call	KEY_CHECK				;Check keys change
	call	PROCESS_KEY_PRESSED

	btfsc	FLTCONFIG,FLTAS
	call	FAULTA_PROCESS

;	btfsc	FLAGS,FLAG_FAULT
;	call	TOGGLE_LEDS
	bra		MAIN_LOOP





;*******************************************************************************
;*******************************************************************************
; 							INTERRUPT SERVICE ROUTINES
;*******************************************************************************
;*******************************************************************************
	
;*******************************************************************************
;High priority interrupt service routine
;Timer0 overflow are checked
;*******************************************************************************
ISR_HIGH
	btfsc	INTCON,TMR0IF			;Timer0 overflow Interrupt?	
	bra		TIMER0_OVERFLOW			;Yes
	RETFIE	FAST

TIMER0_OVERFLOW						;TMR0 overflow ISR
	movff	FREQ_REF_H,TMR0H		;Load the Higher byte of SpeedCommand to TMR0H
	movff	FREQ_REF_L,TMR0L		;Load the Lower byte of SpeedCommand to TMR0L
	
	bsf		FLAGS,TIMER0_OV_FLAG
	bcf		INTCON,TMR0IF			;Clear TMR0IF
	RETFIE	FAST	

;*******************************************************************************
;Low priority interrupt service routine
;*******************************************************************************
ISR_LOW

	btfsc	PIR3,IC1IF				;IC1 (QEI Velocity Mode) Interrupt?
	bra		READ_SPEED_FEEDBACK		;Yes

	btfsc	PIR1,ADIF				;HSADC Interrupt?
	bra		READ_ADC_RESULTS		;Yes

	RETFIE	FAST

READ_SPEED_FEEDBACK	
	movff	VELRH, VELOCITY_HIGH	;copy VELRH:VELRL value for feedback velocity calculation
	movff	VELRL, VELOCITY_LOW
	bsf		FLAGS1, FEEDBACK_UPDATE		;set flag to indicate velocity feedback has been updated
	nop
	bcf		PIR3,IC1IF				;IC1IF flag is cleared for next interrupt
	RETFIE	FAST

READ_ADC_RESULTS													
	movff	ADRESH,FREQUENCY 		;first value is group A, assigned to AN8, current measurement
	movlw	0x30					;Minimum Frequency set to 5Hz (scaling factor X4) 
	cpfsgt	FREQUENCY				; if frequency is less than or equal to 5Hz..
	movwf	FREQUENCY				;	set it to 5Hz
	movlw	0xF0					;Limiting V/F to F= 60Hz (scaling factor X4) 
	cpfslt	FREQUENCY				; if frequency is greater than or equal to 60Hz..
	movwf	FREQUENCY					;	set it to 60Hz
		
;	movff	ADRESH,CURRENT 		;second value is group B, assigned to AN1, speed reference potentiometer


;	movlw	0x3C			;FOR  DEBUGGING = 15Hz
;	movlw	0xDD			;FOR  DEBUGGING
;	movlw	0xF0			;FOR  DEBUGGING = 60Hz
;	movwf	FREQUENCY		;FOR  DEBUGGING



	bsf		FLAGS1, FREQ_UPDATE		;set flag to indicate FREQUENCY has been updated
	bcf		PIR1,ADIF				;ADIF flag is cleared for next interrupt
	RETFIE	FAST		


;*******************************************************************************
;*******************************************************************************
;							MOTOR DRIVE SUBROUTINES
;*******************************************************************************
;*******************************************************************************

;*******************************************************************************
;UPDATE_PWM_DUTYCYCLES
;
;This routine will update the PWM duty cycle on CCPx according to the 
;offset to the table with 0-120-240 degrees.
;This routine scales the PWM value from the table based on the frequency to keep V/F
;constant.
;*******************************************************************************
UPDATE_PWM_DUTYCYCLES

	movf	TABLE_OFFSET1,W			;Load the table offset for Phase 1
	movf	PLUSW0,W				;Use offset to access value in sine table via indirect addressing
	mulwf	FREQUENCY, W			;Table_value X Frequency
	movff	PRODH,PDC0H_TEMP		;Copy high product into temporary variable for PDC0H
	movff	PRODL,PDC0L_TEMP		;Copy low product into temporary variable for PDC0L

;	movlw	0x1E					;Add 78h (1Eh shifted left two bits), in order to prevent 0 duty cycle
;	addwf	PDC0L_TEMP, F
;	btfsc	STATUS, C
;	incf	PDC0H_TEMP, F

UPDATE_PWM2
	movf	TABLE_OFFSET2,W			;Load the table offset for Phase 2
	movf	PLUSW0,W				;Use offset to access value in sine table via indirect addressing
	mulwf	FREQUENCY, W			;Table_value X Frequency
	movff	PRODH,PDC1H_TEMP		;Copy high product into temporary variable for PDC1H
	movff	PRODL,PDC1L_TEMP		;Copy low product into temporary variable for PDC1L
	
;	movlw	0x1E					;Add 78h (1Eh shifted left two bits), in order to prevent 0 duty cycle
;	addwf	PDC0L_TEMP, F
;	btfsc	STATUS, C
;	incf	PDC0H_TEMP, F
	
UPDATE_PWM3
	movf	TABLE_OFFSET3,W			;Load the table offset for Phase 3
	movf	PLUSW0,W				;Use offset to access value in sine table via indirect addressing
	mulwf	FREQUENCY, W			;Table_value X Frequency
	movff	PRODH,PDC2H_TEMP		;Copy high product into temporary variable for PDC2H
	movff	PRODL,PDC2L_TEMP		;Copy low product into temporary variable for PDC2L
	
;	movlw	0x1E					;Add 78h (1Eh shifted left two bits), in order to prevent 0 duty cycle
;	addwf	PDC0L_TEMP, F
;	btfsc	STATUS, C
;	incf	PDC0H_TEMP, F
	
TRUNCATE_PWM123						;Truncate results of multiply to 10 uppermost bits
	bcf		STATUS,C				;discarding lower two bits and right justifying 
	rlcf	PDC0L_TEMP,F
	rlcf	PDC0H_TEMP,F
	rlcf	PDC0L_TEMP,F
	rlcf	PDC0H_TEMP,F
	rlcf	PDC0L_TEMP,W
	andlw	0x3
	movff	PDC0H_TEMP,PDC0L_TEMP
	movwf	PDC0H_TEMP
	
	bcf		STATUS,C
	rlcf	PDC1L_TEMP,F
	rlcf	PDC1H_TEMP,F
	rlcf	PDC1L_TEMP,F
	rlcf	PDC1H_TEMP,F
	rlcf	PDC1L_TEMP,W
	andlw	0x3
	movff	PDC1H_TEMP,PDC1L_TEMP
	movwf	PDC1H_TEMP

	bcf		STATUS,C
	rlcf	PDC2L_TEMP,F
	rlcf	PDC2H_TEMP,F
	rlcf	PDC2L_TEMP,F
	rlcf	PDC2H_TEMP,F
	rlcf	PDC2L_TEMP,W
	andlw	0x3
	movff	PDC2H_TEMP,PDC2L_TEMP
	movwf	PDC2H_TEMP

;testofPWM
;	infsnz	temp, f
;	incf	temp1, f
;	btfss	temp1, 2			;if this bit is set temp1=4
;	goto	donewiththis
;	clrf	temp
;	clrf	temp1
;donewiththis
;	movff	temp, PDC0L_TEMP
;	movff	temp1, PDC0H_TEMP	
				
;	movlw	0x01
;	movwf	PDC0L_TEMP
;	movlw	0x00
;	movwf	PDC0H_TEMP
;	movlw	0xe8
;	movwf	PDC1L_TEMP
;	movlw	0x03
;	movwf	PDC1H_TEMP

	call	CHECK_LIMITS

	bsf		PWMCON1, UDIS		;Disable updates to duty cycle and period
	movff	PDC0L_TEMP,PDC0L	;Transfer temporary values into duty cycle registers
	movff	PDC0H_TEMP,PDC0H
	movff	PDC1L_TEMP,PDC1L
	movff	PDC1H_TEMP,PDC1H
	movff	PDC2L_TEMP,PDC2L
	movff	PDC2H_TEMP,PDC2H
	
	
;	movlw	0xFA
;	movwf	PDC0L
;	movlw	0x00
;	movwf	PDC0H
;	movlw	0xF4
;	movwf	PDC1L
;	movlw	0x01
;	movwf	PDC1H
;	movlw	0xEE
;	movwf	PDC2L
;	movlw	0x02
;	movwf	PDC2H
	
	bcf		PWMCON1, UDIS		;Enable updates to duty cycle and period to update simultaneously.

	
	return	

;*******************************************************************************
;UPDATE_TABLE_OFFSET
;
;This routine Updates the offset pointers to the table after every access
;*******************************************************************************
UPDATE_TABLE_OFFSET
	btfss	FLAGS,OFFSET1_FLAG			;If set incr. on table
	bra		DECREMENT_OFFSET1
	movlw	(SINE_TABLE_ENTRIES-1)		;Check for the last value on the table
	cpfslt	TABLE_OFFSET1
	bra		CLEAR_OFFSET1_FLAG
	incf	TABLE_OFFSET1,F				;Increment offset1
	bra		UPDATE_OFFSET2

CLEAR_OFFSET1_FLAG
	bcf		FLAGS,OFFSET1_FLAG

DECREMENT_OFFSET1
	dcfsnz	TABLE_OFFSET1,F				;Decrement offset1
	bsf		FLAGS,OFFSET1_FLAG

UPDATE_OFFSET2
	btfss	FLAGS,OFFSET2_FLAG			;If set incr. on table
	bra		DECREMENT_OFFSET2
	movlw	(SINE_TABLE_ENTRIES-1)		;Check for the last value on the table
	cpfslt	TABLE_OFFSET2
	bra		CLEAR_OFFSET2_FLAG
	incf	TABLE_OFFSET2,F				;Increment offset2
	bra		UPDATE_OFFSET3

CLEAR_OFFSET2_FLAG
	bcf		FLAGS,OFFSET2_FLAG

DECREMENT_OFFSET2
	dcfsnz	TABLE_OFFSET2,F				;Decrement offset2
	bsf		FLAGS,OFFSET2_FLAG

UPDATE_OFFSET3
	btfss	FLAGS,OFFSET3_FLAG			;If set incr. on table
	bra		DECREMENT_OFFSET3
	movlw	(SINE_TABLE_ENTRIES-1)		;Check for the last value on the table
	cpfslt	TABLE_OFFSET3
	bra		CLEAR_OFFSET3_FLAG
	incf	TABLE_OFFSET3,F				;Increment offset3
	return	

CLEAR_OFFSET3_FLAG
	bcf		FLAGS,OFFSET3_FLAG

DECREMENT_OFFSET3
	dcfsnz	TABLE_OFFSET3,F				;Decrement offset3
	bsf		FLAGS,OFFSET3_FLAG
	return	

;*******************************************************************************
; CALCULATE_TIMER0_RELOAD
;
;This routine calculates the Timer0 reload value based on ADC read value and the 
;scaling factor calculated based on the main clock and number of Sine table entries.
;Timer0 value = FFFF - (FREQUENCY_SCALE/Frequency)	Frequ = (adc result)
;*******************************************************************************
CALCULATE_TIMER0_RELOAD

	bsf		PORTD,0
	bcf		FLAGS1,FREQ_UPDATE
	clrf	TEMP
	clrf	TEMP1
	movlw	HIGH(FREQUENCY_SCALE)		;FREQUENCY_SCALE/Frequency
	movwf	TEMP_LOCATION				;16 bit by 8 bit division	     	
	movlw	LOW(FREQUENCY_SCALE)		;
	movwf	TEMP_LOCATION+1

continue_subtraction

	movf	FREQUENCY,W
	btfsc	STATUS,Z
	return
	bsf		STATUS,C
	movf	FREQUENCY,W
	subwfb	TEMP_LOCATION+1,F
	clrf	WREG
	subwfb	TEMP_LOCATION,F
	btfss	STATUS,C
	goto	keep_result_in_rpm	
	incf	TEMP,F
	btfsc	STATUS,C					;Result of the division is stored in TEMP&TEMP1
	incf	TEMP1,F
	goto	continue_subtraction	

keep_result_in_rpm	
;Timer0 value = FFFF-Timer0 
	bsf		STATUS,C
	movlw	0xFF
	subfwb	TEMP,F
	subfwb	TEMP1,F						;The Timer0 reload value stored in 
	movff	TEMP1,FREQ_REF_H			;FREQ_REF_H & FREQ_REF_L 
	movff	TEMP,FREQ_REF_L				;These values will be loaded to

	bcf		PORTD,0
	return								;Timer0 in Timer0 overflow interrupt
	
	
	
;*******************************************************************************
;CHECK_LIMIT ROUTINE
;   for frequency < 60Hz, duty cycle will be less than MAX_DUTY_CYCLE (4 x PTPER)due to 
;		the selection of sine table values
;	it is still necessary to ensure that PDC is greater or equal to MINL_DUTY_CYCLE (3 x deadtime)
;
;*******************************************************************************
CHECK_LIMITS

CHK_PWM0_MIN					;Test to see if PDC0H:PDC0L < 0:MINL_DUTY_CYCLE
	movf	PDC0H_TEMP, F		;First, is PDC0H = 0?
	bnz		CHK_PWM1_MIN		;	if no, then PDC cannot be less than minimum, check next value
	movlw	MINL_DUTY_CYCLE		;Second, is PDC0L > MINL_DUTY_CYCLE?
	cpfsgt	PDC0L_TEMP			;		if yes, then PDC is not less than minimum, check next value
	movwf	PDC0L_TEMP			;		if no, make it the minimum value

CHK_PWM1_MIN					;Test to see if PDC1H:PDC1L < 0:MINL_DUTY_CYCLE
	movf	PDC1H_TEMP, F		;First, is PDC1H = 0?
	bnz		CHK_PWM2_MIN		;	if no, then PDC cannot be less than minimum, check next value
	movlw	MINL_DUTY_CYCLE		;Second, is PDC1L > MINL_DUTY_CYCLE?
	cpfsgt	PDC1L_TEMP			;		if yes, then PDC is not less than minimum, check next value
	movwf	PDC1L_TEMP			;		if no, make it the minimum value
	
CHK_PWM2_MIN					;Test to see if PDC2H:PDC2L < 0:MINL_DUTY_CYCLE
	movf	PDC2H_TEMP, F		;First, is PDC2H = 0?
	bnz		DONE_CHECK_LIMITS	;	if no, then PDC cannot be less than minimum, check next value
	movlw	MINL_DUTY_CYCLE		;Second, is PDC2L > MINL_DUTY_CYCLE?
	cpfsgt	PDC2L_TEMP			;		if yes, then PDC is not less than minimum, check next value
	movwf	PDC2L_TEMP			;		if no, make it the minimum value

DONE_CHECK_LIMITS
	return


;*******************************************************************************
;This routine stops the motor by driving the PWMs to 0% duty cycle.
;*******************************************************************************
STOP_MOTOR
	bcf		PIE1,ADIE
	bcf		INTCON,TMR0IE
	clrf	OVDCOND			;enable output overrides before setting duty cycle to zero (due to an errata)
;	clrf	PDC0H
;	clrf	PDC1H
;	clrf	PDC2H	
;	clrf	PDC3H	
;	clrf	PDC0L	
;	clrf	PDC1L
;	clrf	PDC2L
;	clrf	PDC3L
	clrf	TABLE_OFFSET1
	clrf	TABLE_OFFSET2
	clrf	TABLE_OFFSET3
	bcf		FLAGS, TIMER0_OV_FLAG
;	goto	$-2
	
	return
	
;*******************************************************************************
;This routine starts motor from previous stop with motor parameters initialized
;*******************************************************************************
RUN_MOTOR_AGAIN
	bsf		FLAGS1,RUN_STOP	
	bcf		FLAGS,FLAG_FAULT
	bsf		PIE1,ADIE
	call	INIT_MOTOR_START
	call	UPDATE_PWM_DUTYCYCLES	;Yes, update the PWM duty cycle with new value
	call	UPDATE_TABLE_OFFSET		;Update 3 offsets
	bsf		INTCON,TMR0IE		
	movlw	b'11111111'				;after duty cycles contain non-zero values, 
	movwf	OVDCOND					;OVDCOND is configured such that there is no output override
	
	return

;*******************************************************************************
;*******************************************************************************
;							FEEDBACK CONTROL SUBROUTINES
;*******************************************************************************
;*******************************************************************************

;*******************************************************************************
; CALCULATE_ACTUAL_SPEED
;
;This routine calculates the actual motor speed based on captured VELRH:VELRL
;
; Factual = (((Fosc x T5PS)/4) x QE edges per revolution))/(VELRH:VELRL)
; 
;
;*******************************************************************************
CALCULATE_ACTUAL_SPEED
	bcf		FLAGS1,FEEDBACK_UPDATE
	return



;*******************************************************************************
;*******************************************************************************
;							KEY SWITCH SUBROUTINES
;*******************************************************************************
;*******************************************************************************

;*******************************************************************************
;This routine checks for the keys status. 2 keys are checked, Run/Stop and 
;Forward(FWD)/Reverse(REV)  
;*******************************************************************************
KEY_CHECK
	btfsc	KEY_PORT,RUN_STOP_KEY		;Is key pressed "RUN/STOP"?
	goto	CHECK_FWD_REV_KEY
	btfsc	FLAGS1,DEBOUNCE
	return
	call	KEY_DEBOUNCE
	btfss	FLAGS1,DEBOUNCE
	return
	bsf		FLAGS1,KEY_RS
	return
	
CHECK_FWD_REV_KEY
 	btfsc	KEY_PORT,FWD_REV_KEY		;Is key pressed "FWD/REV"?
	goto	SET_KEYS
	btfsc	FLAGS1,DEBOUNCE
	return
	call	KEY_DEBOUNCE
	btfss	FLAGS1,DEBOUNCE
	return
	bsf		FLAGS1,KEY_FR
	return

SET_KEYS
	btfss	FLAGS1,DEBOUNCE
	return
	bcf		FLAGS1,DEBOUNCE
	bsf		FLAGS1,KEY_PRESSED	
	btfss	FLAGS1,KEY_RS
	bra		ITS_FWD_REV	
	btg		FLAGS1,RUN_STOP
	return
ITS_FWD_REV	
	btg		FLAGS1,FWD_REV
	return

;*******************************************************************************
KEY_DEBOUNCE
	decfsz	DEBOUNCE_COUNTER,F
	return
	bsf		FLAGS1,DEBOUNCE
	movlw	DEBOUNCE_COUNT
	movwf	DEBOUNCE_COUNTER
	return
;*******************************************************************************
PROCESS_KEY_PRESSED
	btfss	FLAGS1,KEY_PRESSED
	return
	btfss	FLAGS1,KEY_RS
	goto	CHECK_FWD_REV
	btfss	FLAGS1,RUN_STOP	
	goto	STOP_MOTOR_NOW
	call	RUN_MOTOR_AGAIN
	bcf		FLAGS1,KEY_PRESSED
	bcf		FLAGS1,KEY_RS

	bsf		LED_PORT,RUN_STOP_LED	
	return

STOP_MOTOR_NOW
	call	STOP_MOTOR					;STOP motor
	bcf		FLAGS1,KEY_PRESSED
	bcf		FLAGS1,KEY_RS
	bcf		LED_PORT,RUN_STOP_LED	
	return

CHECK_FWD_REV
	btfss	FLAGS1,KEY_FR
	return

	btg		LED_PORT,FWD_REV_LED	
	bcf		LED_PORT,RUN_STOP_LED	
	call	STOP_MOTOR

	call	DELAY
	call	DELAY
	call	DELAY
	call	DELAY

	btg		FLAGS,MOTOR_DIRECTION

	call	RUN_MOTOR_AGAIN
	bcf		FLAGS1,KEY_PRESSED
	bcf		FLAGS1,KEY_FR
	bsf		LED_PORT,RUN_STOP_LED	
	return


;*******************************************************************************
; FAULTA_PROCESS
;*******************************************************************************
FAULTA_PROCESS
	bsf		PORTC,0
	bsf		PORTD,0
	bsf		PORTD,1
	bsf		PORTD,2

	bsf		FLAGS,FLAG_FAULT
	call	STOP_MOTOR
	bcf		FLTCONFIG,FLTAS
	bcf		FLAGS1,KEY_PRESSED
	bcf		FLAGS1,KEY_RS
	bcf		FLAGS1,RUN_STOP	
	return	

;*******************************************************************************
; TOGGLE_LEDS
;*******************************************************************************
TOGGLE_LEDS

	incfsz	COUNTER_SP,F
	return
	incfsz	COUNTER_SP1,F
	return
	btg		PORTC,0
	btg		PORTD,0
	btg		PORTD,1
	btg		PORTD,2
	return

;*******************************************************************************
; Delay routine.
;*******************************************************************************
DELAY
	movlw	DELAY_COUNT1
	movwf	COUNTER
dec_count	
	movlw	DELAY_COUNT2
	movwf	COUNTER1
dec_count1
	decfsz	COUNTER1,F
	bra		dec_count1
	decfsz	COUNTER,F
	bra		dec_count
	clrf	COUNTER
	clrf	COUNTER1
	return		






;*******************************************************************************
;*******************************************************************************
; 						INITIALIZATION SUBROUTINES
;*******************************************************************************
;*******************************************************************************

;*******************************************************************************
;Initialize High-Speed ADC
;*******************************************************************************
INIT_HSADC
	movlw	b'00000000'			; ADCON1 is configured such that:
	movwf	ADCON1				; a) Vref+ and Vref- are Avdd and Avss, respectively.
								; b) The FIFO buffer is disabled

	movlw	b'00110010'			; ADCON2 is configured such that:
	movwf	ADCON2				; a) The A/D result is left justified (so remember to read ADRESL before reading ADRESH)
								; b) The A/D acquisition time is set to 12Tad, as required for sequential conversion.
								; c) The A/D conversion clock is set to Fosc/32.

	movlw	b'01000000'			; ADCON3 is configured such that:
	movwf	ADCON3				; a) An interrupt is generated on every 2nd and 4th write to the FIFO buffer.
								; b) No external ADC triggers are enabled.
	
	movlw	b'00000100'			; ADCHS is configured such that:
	movwf	ADCHS				; a) Group A signal is AN0, current reference  
								; b) Group B signal is AN1, speed reference
								; c) Group C signal is AN6, 

	movlw	b'01000011'			; ANSEL0 is configured such that:
	movwf	ANSEL0				; a) AN0 and AN1 and AN6 are analog input pins.
	movlw	b'00000011'			; b) Corresponding bits of TRISA are also set to inputs.
	movwf	TRISA
	bsf		TRISE, 0			; c) Corresponding bit of TRISE is also set to input.

	movlw	b'00000001'			; ANSEL1 is configured such that:
	movwf	ANSEL1				; a) AN8 is an analog input pin.
	bsf		TRISE, 2			; b) Corresponding bit of TRISE is also set to input.
	
	movlw	b'00000101'			; ADCON0 is configured such that:
	movwf	ADCON0				; a) Single shot mode is enabled
								; b) Single-channel mode is enabled
								; c) Group B signal is sampled (speed ref on development board)
								; d) The ADC is turned on.
	return
	
	
	
;*******************************************************************************
;Initialize PCPWM
;	NOTES:
; 	1)	PTPER has 12-bit resolution, 4 LSBs of PTPERH and 8 bits of PTPERL
; 	2)	In edge aligned mode, PTMR reset to zero on match with PTPER
;	3)	PDC has 14-bit resolution, 6 LSBs of PDCxH and 8 bits of PDCxL
;	4)	Lower 2 bits of PDC compared to Q clocks
;
;	5)	Resolution(of duty cycle)= log((Fosc/4)/Fpwm)/log(2) = log(5Mhz/20kHz)/log(2) = 8 bits
;		so 6 LSBs of PDCxH and 2 MSBs of PDCxL will be used.
;	   (for 16kHz, resolution = log(5Mhz/16kHz)/log(2) = 8 bits, also.)
;
;*******************************************************************************
INIT_PCPWM

	movlw	b'00000000'			; PTCON0 is configured such that:
	movwf	PTCON0			; a) Postscale value is 1:1
								; b) PWM time base input is Fosc/4
								; c) PWM time base mode is free-running for edge-aligned operation
	
	movlw	0xF9				; PTPERL and PTPERH are set up for a 20KHz PWM frequency.		
	movwf	PTPERL			; 	PTPERH:PTPERL = ((Fosc/4)/(PWMfreq x PTMRps))-1
	movlw	0x00				; 	PTPERH:PTPERL = ((20MHz/4)/(20KHz x 1))-1 = 249d = F9h
	movwf	PTPERH			

;	movlw	0x37				; PTPERL and PTPERH are set up for a 16KHz PWM frequency.		
;	movwf	PTPERL			; 	PTPERH:PTPERL = ((Fosc/4)/(PWMfreq x PTMRps))-1
;	movlw	0x01				; 	PTPERH:PTPERL = ((20MHz/4)/(16KHz x 1))-1 = 311d = 137h
;	movwf	PTPERH	
	
	movlw	b'01000000'			; PWMCON0 is configured such that:
	movwf	PWMCON0			; a) PWM0, PWM1, PWM2, PWM3, PWM4, and PWM5 are enabled for output.
								; b) All PWM I/O pairs are set to complimentary mode
										
	movlw	b'00000001'			;PWMCON1 is configured such that:
	movwf	PWMCON1				; a) Special event trigger post-scaler is set to 1:1
								; b) Special event trigger occurs when time-base is counting upwards
								; c) Updates from duty cycle and period buffer registers are enabled.
								; d) Output overrides via OVDCON are synchronous to the PWM timebase.			


	movlw	b'00001010'			;1us deadtime instead of 2us		
;	movlw	b'00010100'			;DTCON is configured such that:
	movwf	DTCON				; a) Clock source for dead-time is Fosc/2.
								; b) Dead time = Dead time value / (Fosc/2) = 2uS.							
;	clrf	DTCON				;DTCON for zero dead-time (relying on IRAMS cross-conduction prevention logic)							

;	movlw	b'00000000'
	movlw	b'11111111'			;OVDCOND is configured such that there is no output override.
	movwf	OVDCOND
	
	movlw	b'00000000'			;OVDCONS is configured such that all PWM outputs are 0 upon power-up.
;	movlw	b'11111111'			;OVDCONS is configured such that all PWM outputs are 1 (inactive) upon power-up.
	movwf	OVDCONS
	
	
	;FLTA is overcurrent fault, muxed via configuration bit to RC1
	;FLTB is overvoltage fault on RC2
;	bsf		TRISC, 1			;Ensure that RC1 is an input
;	bsf		TRISC, 2			;Ensure that RC2 is an input
;	movlw	b'10000000'			;Fault A and FaultB are disabled.
;	movlw	b'10000011'			;Fault B is disabled
	movlw	b'10010001'			;Fault A and FaultB are enabled in catastrophic mode.
;	movlw	b'10110011'			;FLTCONFIG is configured such that: 
	movwf	FLTCONFIG			; a) Enable fault condition on break-point for use with ICD2
								; b) Enable FaultA in cycle-by-cycle mode
								; c) Enable FaultB in cycle-by-cycle mode
								; d) Fault A and Fault B disable PWM channels 0 to 5
	
	movlw	0x00				;SEVTCMPL and SEVTCMPH are clear.
	movwf	SEVTCMPL
	movlw	0x00
	movwf	SEVTCMPH

;	clrf	PDC0L				;PDC0L, PDC1L, PDC2L, PDC3L, PDC0H, PDC1H, PDC2H, PDC3H 
;	clrf	PDC1L				; 	are clear so that duty cycles are initially 0.
;	clrf	PDC2L	
;	clrf	PDC3L	
;	clrf	PDC0H	
;	clrf	PDC1H	
;	clrf	PDC2H	
;	clrf	PDC3H	


	bsf		PTCON1, PTEN		;PTEN bit in the PTCON1 is set to enable the PWM time base.
	
;----------------------
;infinite_loop
;	goto	infinite_loop
;----------------------
	
	
	return
	
	
	
;*******************************************************************************
;Initialize QEI
;*******************************************************************************
INIT_QEI
	movlw	b'00001000'			; QEICON is configured such that:
	movwf	QEICON				; a) Velocity mode is enabled
								; b) QEI mode is set to x2 velocity input mode.
								; c) Velocity pulse reduction ratio is set to 1:1 

	movlw	b'00011100'			; TRISA is configured such that:
	iorwf	TRISA, F			; a) pins corresponding to INDX, QEA, and QEB are made inputs
								; (inclusive or is used to preserve settings of other TRISA pins)

	movlw	b'00000001'			; T5CON is configured such that:
	movwf	T5CON				; a) Special event reset is disabled				
								; b) Continuous count mode is enabled
								; c) Timer5 input clock prescaler is 1:1
								; e) Timer5 is enabled 

	movlw	b'00111010'			; Digital filter is configured
	movwf	DFLTCON				; a) QEA, QEB, and INDX filters enabled
								; b) noise filter clock divider = 1:4
	return
	
;*******************************************************************************
;Initialize IC1
;	- an alternative mode of speed measurement using input capture of index 
;	pulse from quadrature encoder.  Simulates a basic tachometer.  Uses Input Capture 1.
;*******************************************************************************
INIT_IC1
	clrf	QEICON				; Clear QEICON to make sure that QEI mode is disabled.
	
	movlw	b'01000111'			;Configure CAP1CON such that:
	movwf	CAP1CON				; a)Timer5 resets on capture event
								; b)Pulse width measurement mode every rising to falling edge
	
	bsf		TRISA, 2			; Make TRISA pin corresponding to IC1 (INDX signal from encoder)an input

	movlw	b'00000001'			; T5CON is configured such that:
	movwf	T5CON				; a) Special event reset is disabled				
								; b) Continuous count mode is enabled
								; c) Timer5 input clock prescaler is 1:1
								; e) Timer5 is enabled 

	movlw	b'00001010'			; Digital filter is configured
	movwf	DFLTCON				; a) INDX filter enabled
								; b) noise filter clock divider = 1:4
	return

;*******************************************************************************
;Initialize PORTC
;*******************************************************************************
INIT_PORTC
	movlw	b'10111110'
	movwf	TRISC
	bsf		PORTC,0
	return

;*******************************************************************************
;Initialize PORTD
;*******************************************************************************
INIT_PORTD
	movlw	b'11010000'			;RD4 is an input (Fault A)
	movwf	TRISD
	bsf		PORTD,0				;*************this sequence could cause RMW issues*************
	bsf		PORTD,1				;*************this sequence could cause RMW issues*************
	bsf		PORTD,2				;*************this sequence could cause RMW issues*************
	return

;*******************************************************************************
;Initialize Timer0
;*******************************************************************************
INIT_TMR0
	movlw	b'10000100'					;T0CON is configured such that:
	movwf	T0CON						;a) TMR0 ON, 
										;b) 16-bit operation, 
										;c) clock source is instruction cycle clock, 
										;d) prescalar is 1:16

	movlw	0xF8						;Timer0 Initialisation 
	movwf	TMR0H
	movlw	0x5E						;
	movwf	TMR0L	
	return

;*******************************************************************************
;Initialize interrupts
;*******************************************************************************
INIT_INTERRUPTS

	bsf		INTCON,TMR0IE				;Timer0 overflow Interrupt enable
	bsf		INTCON2,TMR0IP				;Timer0 overflow Interrupt high priority

	bsf		PIE1,ADIE					;AD Converter Interrupt enable (will trigger on 2nd and 4th writes to FIFO)
	bcf		IPR1,ADIP					;make sure it's low priority
;	bsf		ADCON0, GO					;set GO bit to start conversions in continuous loop (see initHSADC)

	bsf		PIE3, IC1IE					;Enable IC1 interrupt (for QEI velocity measurement mode)
	bcf		IPR3, IC1IP					;make sure it's low priority	

	movlw	b'10010011'					;Power ON reset status bit/Brownout reset status bit
	movwf	RCON						;and Instruction flag bits are set
										;Enable Priority levels on Interrupts
	
	bsf		INTCON,GIEL					;Enable low priority interrupts
	bsf		INTCON,GIEH					;Enable high priority interrupts
		


; 3) PIE3 and IPR3 are configured such that every time a value is captured from Timer5, a low-priority interrupt is generated
	return


;*******************************************************************************
;Initialize Motor
;*******************************************************************************
INIT_MOTOR_START
	movlw	0x09						;Initialize the table offset to 3 registers
	movwf	TABLE_OFFSET1				;to form 0-120-240 degrees
	bsf		FLAGS,OFFSET1_FLAG			;Offset flags initialization
	btfss	FLAGS,MOTOR_DIRECTION
	bra		INIT_MOTOR_START_REV
	movlw	0x03
	movwf	TABLE_OFFSET2
	bcf		FLAGS,OFFSET2_FLAG
	movlw	0x0F
	movwf	TABLE_OFFSET3
	bcf		FLAGS,OFFSET3_FLAG
	bsf		PORTC,0
	bra		CONT_INIT_MOT
INIT_MOTOR_START_REV
	movlw	0x0F
	movwf	TABLE_OFFSET2
	bcf		FLAGS,OFFSET2_FLAG
	movlw	0x03
	movwf	TABLE_OFFSET3
	bcf		FLAGS,OFFSET3_FLAG

	bcf		PORTC,0

CONT_INIT_MOT
	movlw	0x30						;Initialize frequency to 12Hz
	movwf	FREQUENCY
	movlw	0xFD						;Timer0 Initialisation 
	movwf	FREQ_REF_H
	movwf	TMR0H
	movlw	0x2C						;
	movwf	TMR0L	
	movwf	FREQ_REF_L
	bsf		FLAGS,TIMER0_OV_FLAG
	return

;*******************************************************************************
;Upon initialization the Sine table contents are copied to the RAM from 
;Program memory 
;*******************************************************************************
COPY_TABLE_TO_RAM
	movlw	UPPER sine_table			;Initialize Table pointer to the first  
	movwf	TBLPTRU						;location of the table
	movlw	HIGH sine_table
	movwf	TBLPTRH
	movlw	LOW sine_table
	movwf	TBLPTRL
	movlw	LOW(SINE_TABLE)
	movwf	FSR0L
	movlw	HIGH(SINE_TABLE)
	movwf	FSR0H
	movlw	0x14
	movwf	TEMP
COPY_AGAIN
	TBLRD*+
	movff	TABLAT,POSTINC0
	decfsz	TEMP,F
	bra		COPY_AGAIN

	movlw	LOW(SINE_TABLE)				;FSR0 points to the starting of the table
	movwf	FSR0L
	movlw	HIGH(SINE_TABLE)
	movwf	FSR0H
	return		




;*******************************************************************************
;*******************************************************************************
;								SINE TABLE
;*******************************************************************************
;*******************************************************************************

TABLE	code 0x0600		;was 0x0100
;below table is from 270 degrees to 90 degrees @ 10 degree resolution; for 20MHz, PR2 = F9, Timer2 1:1 prescale
;sine_table db	0x00,0x02,0x08,0x11,0x1E,0x2E,0x40,0x54,0x69,0x80,0x96,0xAB,0xBF,0xD1,0xE1,0xEE,0xF7,0xFD,0xFF

;below table is from 270 degrees to 90 degrees @ 10 degree resolution; for 16MHz, PR2 = 137, Timer2 1:1 prescale
sine_table db	0x00,0x02,0x08,0x11,0x1E,0x2E,0x40,0x54,0x69,0x80,0x96,0xAB,0xBF,0xD1,0xE1,0xEE,0xF7,0xFD,0xFF

;*******************************************************************************

	END

