;*******************************************************************************
;PROGRAM			:	Closed Loop V/F control of 3-Phase Induction Motor
;MICROCONTROLLER	:	PIC18Fxx31	
;*******************************************************************************
;*******************************************************************************
;AUTHOR		:	Jon Burroughs, 
;			:	based on V/f code by Padmaraja Yedamale
;			:	Microchip Technology Inc
;DATE		:	2/11/04
;Version	:	V1.0
;*******************************************************************************
;Description:
;
;This code implements closed loop V/f control of a three-phase induction motor using
;the PIC18F4431 Microcontroller.  This code accompanies the application note
;AN890 "3-Phase AC-Induction Motor Control Using the PIC18F4431"
;
;Velocity feedback is added to standard V/f control to control speed more accurately
;with varying loads.
;
;Potentiometer connected to AN1 is used to set target velocity.  Target velocity is compared 
;to actual velocity to generate velocity error.  Velocity error used by PID algorithm to determine 
;the drive frequency.  The drive frequency is then input to V/f algorithm.
;
;Feedback from QEI in x2 velocity mode is a 16-bit value corresponding to the period of time 
;between edges of QEA signal.  To simplify error calculation, the ADC result is used to set a
;target period value, rather than velocity.  The error = target period - actual period.  The PID output
;is drive period, which is then converted to drive frequency for use with V/f algorithm.
;
;*******************************************************************************
;                    Software License Agreement                       
;                                                                     
; The software supplied herewith by Microchip Technology              
; Incorporated (the "Company") for its PICmicro Microcontroller is   
; intended and supplied to you, the Companys customer, for use       
; solely and exclusively on Microchip PICmicro Microcontroller        
; products. The software is owned by the Company and/or its           
; supplier, and is protected under applicable copyright laws. All     
; rights are reserved. Any use in violation of the foregoing          
; restrictions may subject the user to criminal sanctions under       
; applicable laws, as well as to civil liability for the breach of    
; the terms and conditions of this license.                           
;					                              
; THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES,   
; WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED   
; TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 	      
; PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,   
; IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 	      
; CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.	              
;                                                                     
;*******************************************************************************
	LIST p=18f4431,f=INHX32

	include		<p18f4431.inc>
	include		<3im_vf.inc>
;*******************************************************************************

	__CONFIG _CONFIG1H, 0x02 ;_OSC_HS_1H &_FCMEN_OFF_1H&_IESO_OFF_1H
	__CONFIG _CONFIG2L, 0x0C ;_PWRTEN_ON_2L & _BOREN_ON_2L & _BORV_20_2L  
	__CONFIG _CONFIG2H, 0x3E ;_WDTEN_OFF_2H
	__CONFIG _CONFIG3L, 0x3C ;0x24 ;_PWMPIN_OFF_3L & _LPOL_LOW_3L & _HPOL_LOW_3L & _GPTREN_ON_3L
	__CONFIG _CONFIG3H, 0x9D ;_FLTAMX_RC1_3H & _PWM4MX_RB5_3H
	__CONFIG _CONFIG4L, 0x7b 
	__CONFIG _CONFIG5L, 0x0F 
	__CONFIG _CONFIG5H, 0xC0  
	__CONFIG _CONFIG6L, 0x0F 
	__CONFIG _CONFIG6H, 0xE0 
	__CONFIG _CONFIG7L, 0x0F 
	__CONFIG _CONFIG7H, 0x40  

	extern	FXM1616U, FXD1616U, FXD2416U,_24_bit_sub, _24_BitAdd
	extern	PidInitialize, PidMain, PidInterrupt
	extern	percent_err, percent_out, pidStat1

	extern	AARGB0,AARGB1,AARGB2,AARGB3		
	extern	BARGB0,BARGB1,BARGB2,BARGB3
	

#define	SEND_TO_HYPERTERMINAL
;******************************************************************
;Routines used for displaying few key parameters 
;on the screen using Hyper terminal(serial port), refer to the motor_com1.asm
	extern		INITIALIZE_SERIAL_PORT
	extern		WELCOME_MESSAGE
	extern		DISPLAY_PARAMETERS
	extern		DISPLAY_DIGITS
	extern		SEND_BYTE_FROM_WREG	
			
;FLAGS bits
#define TIMER0_OV_FLAG	0
#define	FLAG_FAULT		1
#define	PARAM_DISPLAY	2
#define	POSITION_BIT 	3
#define OFFSET1_FLAG	4
#define OFFSET2_FLAG	5
#define OFFSET3_FLAG	6
#define	MOTOR_DIRECTION	7

;FLAGS1 bits
#define	DEBOUNCE		0
#define	KEY_RS			1
#define	KEY_FR			2
#define	KEY_PRESSED 	3
#define	RUN_STOP		4
#define	FWD_REV			5
#define	DRIVE_FREQ_UPDATE		6
#define	FEEDBACK_UPDATE	7

;PID_FLAGS bits
#define NEGATIVE_SLIP	0		;This is set if the slip frequency is negative
								;	(i.e. the rotor frequency is faster than the drive frequency)
#define VELOCITY_READY	1
#define SLIP_NEGATIVE	2
#define CALC_PWM		3
#define	DIRECTION_CMD	4
#define	PID_INTERRUPT	5
#define	DO_PID_MAIN		6
#define PID_DONE		7

;Keys parameters
#define KEY_PORT 		PORTD
#define RUN_STOP_KEY	6
#define FWD_REV_KEY		7
#define DEBOUNCE_COUNT	0xFF

;Delay parameters
#define	DELAY_COUNT1	0xFF
#define	DELAY_COUNT2	0xFF

;LED parameters
#define LED_PORT 		PORTD
#define RUN_STOP_LED	1
#define FWD_REV_LED		2

;Duty cycle limit definition, for 20KHz @20MHz, 2uS deadtime
#define	MINH_DUTY_CYCLE	0x00	;minimum duty cycle corresponds to 3 x deadtime = 3 x 2uS = 6uS  PDC = 6uS/(4/Fosc)
#define	MINL_DUTY_CYCLE	0x3C	;	try doubling from 0x1E	;
#define	MAXL_DUTY_CYCLE	0xE0	;maximum duty cycle is 4 x PTPER
#define	MAXH_DUTY_CYCLE	0x03

;Duty cycle limit definition, for 16KHz @20MHz, 2uS deadtime
;#define	MINH_DUTY_CYCLE	0x00	;minimum duty cycle corresponds to 3 x deadtime = 3 x 2uS = 6uS  PDC = 6uS/(4/Fosc)
;#define	MINL_DUTY_CYCLE	0x1E	;
;#define	MAXL_DUTY_CYCLE	0xDC	;maximum duty cycle is 4 x PTPER
;#define	MAXH_DUTY_CYCLE	0x04

;---------------------------------------------------
;Variables defined in PID_main routine

	extern	error0, error1, pidStat1, percent_err, percent_out, kp, ki, kd

;----------------------------------------------------------------
;Variables used for displaying the parameters on Hyper terminal
	extern	DISPLAY_TARGET_FREQ
	extern	DISPLAY_ROTOR_FREQ_U
	extern	DISPLAY_ROTOR_FREQ_L
	extern	DISPLAY_SLIP_FREQ_U
	extern	DISPLAY_SLIP_FREQ_L	
	extern	DISPLAY_SLIP_PERCENTAGE
	extern	DISPLAY_PID_PERCENTAGE
	extern	DISPLAY_DRIVE_FREQ
	extern	DISPLAY_PID_FLAGS
	extern	DISPLAY_TEMP1
	
;******************************************************************


;*******************************************************************************
;RAM locations in Access bank, uninitialized
;*******************************************************************************
	UDATA_ACS 	
TABLE_OFFSET1		res	1		;Phase1 offset to the Sine table(0)
TABLE_OFFSET2		res	1		;Phase2 offset to the Sine table(120)
TABLE_OFFSET3		res	1		;Phase3 offset to the Sine table(240)
COUNTER				res	1		;General counters
COUNTER1			res	1
COUNTER_SP			res	1
COUNTER_SP1			res	1
FLAGS				res	1		;Flags registers used to indicate different status
FLAGS1				res	1
FREQ_REF_H			res	1		;Reference Frequency input in counts
FREQ_REF_L			res	1
FREQUENCY			res	1
TEMP				res	1
TEMP1				res	1
TEMP_LOCATION		res	2
DEBOUNCE_COUNTER	res	1
PDC0L_TEMP			res	1
PDC0H_TEMP			res	1
PDC1L_TEMP			res	1
PDC1H_TEMP			res	1
PDC2L_TEMP			res	1
PDC2H_TEMP			res	1
CURRENT				res 1
TEMPERATURE			res 1
VELOCITY_HIGH		res 1
VELOCITY_LOW		res 1
VELOCITY_COUNT		res 1
PID_INTERRUPT_COUNT		res 1

COMMAND_FREQUENCY			res 1
RAMP_COUNTER				res 1

TARGET_FREQUENCY			res 1
ROTOR_FREQUENCY_UPPER		res 1
ROTOR_FREQUENCY_LOWER		res 1
SLIP_FREQUENCY_UPPER		res 1
SLIP_FREQUENCY_LOWER		res 1
SLIP_PERCENTAGE				res 1
PID_PERCENTAGE				res 1
DRIVE_FREQUENCY				res 1
PID_FLAGS					res 1

QUICK_DISPLAY_TEMP1			res 1
QUICK_DISPLAY_TEMP2			res 1
DISPLAY_FLAGS				res 1
DISPLAY_STEP				res 1
#define QUICK_DISPLAY_GO		0
TEMP2				res 1

SINE_TABLE			res	0x14	;Sine table

temp				res 1
temp1				res 1


;*******************************************************************************
;*******************************************************************************
;						RESET AND INTERRUPT VECTORS
;*******************************************************************************
;*******************************************************************************
STARTUP	code 0x00
	goto	Start				;Reset Vector address 
	
	CODE	0x08
	goto	ISR_HIGH			;High priority ISR at 0x0008

PROG_LOW	CODE	0x018
	nop
	goto	ISR_LOW				;Low priority ISR at 0x0018



;*******************************************************************************
;*******************************************************************************
;							INITIALIZATION
;*******************************************************************************
;*******************************************************************************
PROG1	code
Start
	clrf	DRIVE_FREQUENCY
	clrf	VELOCITY_LOW
	clrf	VELOCITY_HIGH
	clrf	VELOCITY_COUNT
	clrf	PID_INTERRUPT_COUNT
		
	clrf	FLAGS
	clrf	FLAGS1
	clrf	DISPLAY_FLAGS

	call	INIT_HSADC
	call	INIT_PCPWM
	call	INIT_QEI
	call	INIT_TMR0
	call	INIT_PORTC
	call	INIT_PORTD
	call	COPY_TABLE_TO_RAM
	call	INIT_MOTOR_START
	
#ifdef	SEND_TO_HYPERTERMINAL
	call	INITIALIZE_SERIAL_PORT	;If serial communication is enabled, the serial port is initialized to communicate with hyper terminal
	call	WELCOME_MESSAGE			;display "ACIM motor control"
#endif
	

	bsf		PORTD,0
	bcf		PORTD,1
	bsf		PORTD,2
	bcf		PORTC,0

	
	
WAIT_HERE
	call	KEY_CHECK				;wait until key is pressed before initializing motor
	btfss	FLAGS1,KEY_PRESSED
	bra		WAIT_HERE
	
	bsf		PORTD,0
	bcf		PORTD,1
	bcf		PORTD,2
	bsf		PORTC,0

	clrf	FLAGS
	clrf	FLAGS1
	
	call	PidInitialize
	call	INIT_INTERRUPTS



;*******************************************************************************	
;*******************************************************************************
; 								MAIN LOOP
;*******************************************************************************
;*******************************************************************************
MAIN_LOOP

;If a PID interrupt has happened, call PidInterrupt
	btfss	PID_FLAGS, PID_INTERRUPT
	bra		bypass0
	call	RAMP_TARGET_FREQUENCY	
	call	PidInterrupt
	bcf		PID_FLAGS, PID_INTERRUPT
;****************************************
	btfsc	DISPLAY_FLAGS, QUICK_DISPLAY_GO
	bra		QUICK_DISPLAY_BUSY
	movff	TARGET_FREQUENCY, QUICK_DISPLAY_TEMP1
	movff	DRIVE_FREQUENCY, QUICK_DISPLAY_TEMP2
	clrf	DISPLAY_STEP
	bsf		DISPLAY_STEP, 0
	bsf		DISPLAY_FLAGS, QUICK_DISPLAY_GO
QUICK_DISPLAY_BUSY
;****************************************
	goto	MAIN_LOOP
bypass0

;If a Timer0 overflow has occurred, then update the PWM duty-cycles.
	btfss	FLAGS,TIMER0_OV_FLAG	;back from Timer0 overflow?
	bra		bypass					;No
	call	UPDATE_PWM_DUTYCYCLES	;Yes, update the PWM duty cycle with new value
	call	UPDATE_TABLE_OFFSET		;Update 3 offsets
	bcf		FLAGS,TIMER0_OV_FLAG	;Clear the flag
	goto	MAIN_LOOP
bypass

;If the drive frequency value has been updated, then calculate a new Timer0 reload value.
	btfss	FLAGS1,DRIVE_FREQ_UPDATE		;has DRIVE_FREQUENCY been updated?
	bra		bypass2	
	call	CALCULATE_TIMER0_RELOAD ; only calculate new Timer0 reload if DRIVE_FREQUENCY has been updated
	bcf		FLAGS1, DRIVE_FREQ_UPDATE
	goto	MAIN_LOOP
bypass2

;If the A/D conversion is complete, start a new conversion
	btfss	ADCON0, GO				;If AD Conversion is complete
	bsf		ADCON0, GO				;	then start a new conversion

;If the velocity feedback has been updated, calculate a new SLIP_PERCENTAGE, 
	btfss   FLAGS1, FEEDBACK_UPDATE	
	bra		bypass3
	
	call	CALCULATE_ROTOR_FREQUENCY
	call	CALCULATE_SLIP_FREQUENCY
	call	CALCULATE_SLIP_PERCENTAGE

	movff	SLIP_PERCENTAGE, percent_err	
	bsf		pidStat1, 2					;set err_sign to indicate positive
	btfsc	PID_FLAGS, NEGATIVE_SLIP	;if slip is negative,
	bcf		pidStat1, 2					;  then clear err_sign to indicate negative
	
	bcf		FLAGS1, FEEDBACK_UPDATE
	goto	MAIN_LOOP
bypass3	

;At a regular interval, set by the Timer1 overflow rate, execute the PID routine 
;  and update the drive frequency
	btfss	PID_FLAGS, DO_PID_MAIN
	bra		bypass4
;	bsf		PORTD, 1		;***LATENCY MEASUREMENT***
	call	PidMain	
;	movlw	0x00					;FOR DEBUGGING, MANUALLY SET PWM OUTPUT
;	movwf	PID_PERCENTAGE
	movff	percent_out, PID_PERCENTAGE
	bcf		PID_FLAGS, DO_PID_MAIN
	bsf		PID_FLAGS, PID_DONE
;	bcf		PORTD, 1		;***LATENCY MEASUREMENT***
;****************************************
;	btfsc	DISPLAY_FLAGS, QUICK_DISPLAY_GO
;	bra		QUICK_DISPLAY_BUSY
;	movff	TARGET_FREQUENCY, QUICK_DISPLAY_TEMP1
;	movff	DRIVE_FREQUENCY, QUICK_DISPLAY_TEMP2
;	clrf	DISPLAY_STEP
;	bsf		DISPLAY_STEP, 0
;	bsf		DISPLAY_FLAGS, QUICK_DISPLAY_GO
;QUICK_DISPLAY_BUSY
;****************************************
	goto	MAIN_LOOP
			
bypass4
;If PID output has been updated, calculate a new drive frequency
	btfss	PID_FLAGS, PID_DONE
	bra		bypass5
	call	CALCULATE_DRIVE_FREQUENCY
	bcf		PID_FLAGS,PID_DONE
	goto	MAIN_LOOP
	
	
bypass5

;#ifdef	SEND_TO_HYPERTERMINAL
;	btfss	FLAGS, PARAM_DISPLAY
;	goto	nodisplay
;	movff	TARGET_FREQUENCY, DISPLAY_TARGET_FREQ
;	movff	ROTOR_FREQUENCY_UPPER, DISPLAY_ROTOR_FREQ_U
;	movff	ROTOR_FREQUENCY_LOWER, DISPLAY_ROTOR_FREQ_L
;	movff	SLIP_FREQUENCY_UPPER, DISPLAY_SLIP_FREQ_U
;	movff	SLIP_FREQUENCY_LOWER, DISPLAY_SLIP_FREQ_L		
;	movff	SLIP_PERCENTAGE, DISPLAY_SLIP_PERCENTAGE
;	movff	PID_PERCENTAGE, DISPLAY_PID_PERCENTAGE
;	movff	DRIVE_FREQUENCY, DISPLAY_DRIVE_FREQ
	
;	bsf		PID_FLAGS, DIRECTION_CMD
;	btfss	FLAGS, MOTOR_DIRECTION
;	bcf		PID_FLAGS, DIRECTION_CMD
	
;	movff	PID_FLAGS, DISPLAY_PID_FLAGS		
;	call	DISPLAY_PARAMETERS
;	bcf		FLAGS, PARAM_DISPLAY
;nodisplay
;#endif

	
;Check for key presses	
	call	KEY_CHECK				;Check keys change
	call	PROCESS_KEY_PRESSED

;Check for faults
	btfsc	FLTCONFIG,FLTAS
	call	FAULTA_PROCESS

	btfsc	FLAGS,FLAG_FAULT
	call	TOGGLE_LEDS

;parameter transmission for debugging purposes
QUICK_DISPLAY
	btfss	DISPLAY_FLAGS, QUICK_DISPLAY_GO
	bra		MAIN_LOOP
	btfss	PIR1,TXIF
	bra		MAIN_LOOP
	btfsc	DISPLAY_STEP, 0
	bra		DISPLAY_FIRST_UPPER_NIBBLE
	btfsc	DISPLAY_STEP, 1
	bra		DISPLAY_FIRST_LOWER_NIBBLE
	btfsc	DISPLAY_STEP, 2
	bra		DISPLAY_SECOND_UPPER_NIBBLE
	btfsc	DISPLAY_STEP, 3
	bra		DISPLAY_SECOND_LOWER_NIBBLE
	btfsc	DISPLAY_STEP, 4
	bra		SEND_CR
	btfsc	DISPLAY_STEP, 5
	bra		SEND_LF
	clrf	DISPLAY_STEP
	bsf		DISPLAY_STEP, 0
	bcf		DISPLAY_FLAGS, QUICK_DISPLAY_GO	
	bra		MAIN_LOOP
	
;display first digit
DISPLAY_FIRST_UPPER_NIBBLE
	movf	QUICK_DISPLAY_TEMP1,W
	andlw	0xF0
	swapf	WREG,W
	addlw	0x30	
	call	CHECK_39_2
	movwf	TXREG
	clrf	DISPLAY_STEP
	bsf		DISPLAY_STEP, 1
	bra		MAIN_LOOP
	
DISPLAY_FIRST_LOWER_NIBBLE
	movf	QUICK_DISPLAY_TEMP1,W
	andlw	0x0F
	addlw	0x30	
	call	CHECK_39_2	
	movwf	TXREG
	clrf	DISPLAY_STEP
	bsf		DISPLAY_STEP, 2
	bra		MAIN_LOOP	
			
;display second digit
DISPLAY_SECOND_UPPER_NIBBLE
	movf	QUICK_DISPLAY_TEMP2,W
	andlw	0xF0
	swapf	WREG,W
	addlw	0x30	
	call	CHECK_39_2
	movwf	TXREG
	clrf	DISPLAY_STEP
	bsf		DISPLAY_STEP, 3
	bra		MAIN_LOOP
	
DISPLAY_SECOND_LOWER_NIBBLE
	movf	QUICK_DISPLAY_TEMP2,W
	andlw	0x0F
	addlw	0x30	
	call	CHECK_39_2	
	movwf	TXREG
	clrf	DISPLAY_STEP
	bsf		DISPLAY_STEP, 4
	bra		MAIN_LOOP
	
;send carriage return
SEND_CR
	movlw	0x0D						;carriage return
	movwf	TXREG
	clrf	DISPLAY_STEP
	bsf		DISPLAY_STEP, 5
	bra		MAIN_LOOP
	
;send line feed
SEND_LF
	movlw	0x0A						;line feed
	movwf	TXREG
	clrf	DISPLAY_STEP
	bsf		DISPLAY_STEP, 0
	bcf		DISPLAY_FLAGS, QUICK_DISPLAY_GO			
	bra		MAIN_LOOP

CHECK_39_2
	movwf	TEMP2
	movlw	0x39
	cpfsgt	TEMP2
	bra		LESS_39_2
	movf	TEMP2,W
	addlw	0x7
	return
LESS_39_2
	movf	TEMP2,W
	return



;*******************************************************************************
;*******************************************************************************
; 							INTERRUPT SERVICE ROUTINES
;*******************************************************************************
;*******************************************************************************
	
;*******************************************************************************
;High priority interrupt service routine
;Timer0 overflow are checked
;*******************************************************************************
ISR_HIGH
	btfsc	INTCON,TMR0IF			;Timer0 overflow Interrupt?	
	bra		TIMER0_OVERFLOW			;Yes
	
	btfsc	PIR1,TMR1IF				;Timer1 overflow Interrupt?
	bra		TIMER1_OVERFLOW
	
	btfsc	PIR3,IC1IF				;IC1 (QEI Velocity Mode) Interrupt?
	bra		READ_SPEED_FEEDBACK		;Yes
	
	btfsc	PIR1,ADIF				;HSADC Interrupt? 		
	bra		READ_ADC_RESULTS		;Yes
	
	RETFIE	FAST

TIMER0_OVERFLOW						;TMR0 overflow ISR

	movff	FREQ_REF_H,TMR0H		;Load the Higher byte of SpeedCommand to TMR0H
	movff	FREQ_REF_L,TMR0L		;Load the Lower byte of SpeedCommand to TMR0L	
	bsf		FLAGS,TIMER0_OV_FLAG
	bcf		INTCON,TMR0IF			;Clear TMR0IF
	
	RETFIE	FAST

TIMER1_OVERFLOW
	bsf		PORTD, 1		;***LATENCY MEASUREMENT***
	bsf		PID_FLAGS, PID_INTERRUPT
	incf	PID_INTERRUPT_COUNT
	btfss	PID_INTERRUPT_COUNT, 4
	bra		T1_OVERFLOW_DONE
	bsf		PID_FLAGS, DO_PID_MAIN
	clrf	PID_INTERRUPT_COUNT
T1_OVERFLOW_DONE
	bcf		PIR1, TMR1IF
	bcf		PORTD, 1		;***LATENCY MEASUREMENT***
	RETFIE	FAST
	
READ_ADC_RESULTS													
	movff	ADRESH,COMMAND_FREQUENCY 		;first value is group A, assigned to AN8, current measurement
	movlw	0x30							;Minimum Frequency set to 12Hz (scaling factor X4) 
	cpfsgt	COMMAND_FREQUENCY				; if frequency is less than or equal to 12Hz..
	movwf	COMMAND_FREQUENCY				;	set it to 12Hz
	movlw	0xF0							;Limiting V/F to F= 60Hz (scaling factor X4) 
	cpfslt	COMMAND_FREQUENCY				; if frequency is greater than or equal to 60Hz..
	movwf	COMMAND_FREQUENCY				;	set it to 60Hz
	bcf		PIR1,ADIF				;ADIF flag is cleared for next interrupt
	RETFIE	FAST		

READ_SPEED_FEEDBACK	

	movf	VELRL, W				;Maintain a running average by adding VELRH:VELRL to
	addwf	VELOCITY_LOW, F			; VELOCITY_HIGH:VELOCITY_LOW and divide by 2 by rotating
	movf	VELRH, W				; once to the right.
	addwfc	VELOCITY_HIGH, F
	bcf		STATUS, C
	rrcf	VELOCITY_HIGH, F
	rrcf	VELOCITY_LOW, F	
			
	incf	VELOCITY_COUNT			;Average 16 times, before setting the feedback update flag
	btfss	VELOCITY_COUNT, 4
	bra		velocity_done
	clrf 	VELOCITY_COUNT
	bsf		FLAGS1, FEEDBACK_UPDATE
	bcf		PIE3,IC1IE				;disable interrupt, re-enable after FEEDBACK_UPDATE flag is cleared
		
velocity_done

	bcf		PIR3,IC1IF				;IC1IF flag is cleared for next interrupt

	RETFIE	FAST
	
;*******************************************************************************
;Low priority interrupt service routine
;*******************************************************************************
ISR_LOW

	RETFIE	FAST


;*******************************************************************************
;*******************************************************************************
;							MOTOR DRIVE SUBROUTINES
;*******************************************************************************
;*******************************************************************************

;*******************************************************************************
;UPDATE_PWM_DUTYCYCLES
;
;This routine will update the PWM duty cycle on CCPx according to the 
;offset to the table with 0-120-240 degrees.
;This routine scales the PWM value from the table based on the frequency to keep V/F
;constant.
;*******************************************************************************
UPDATE_PWM_DUTYCYCLES

	movf	TABLE_OFFSET1,W			;Load the table offset for Phase 1
	movf	PLUSW0,W				;Use offset to access value in sine table via indirect addressing
	mulwf	DRIVE_FREQUENCY, W		;Table value X DRIVE_FREQUENCY
	movff	PRODH,PDC0H_TEMP		;Copy high product into temporary variable for PDC0H
	movff	PRODL,PDC0L_TEMP		;Copy low product into temporary variable for PDC0L

UPDATE_PWM2
	movf	TABLE_OFFSET2,W			;Load the table offset for Phase 2
	movf	PLUSW0,W				;Use offset to access value in sine table via indirect addressing
	mulwf	DRIVE_FREQUENCY, W		;Table_value X DRIVE_FREQUENCY
	movff	PRODH,PDC1H_TEMP		;Copy high product into temporary variable for PDC1H
	movff	PRODL,PDC1L_TEMP		;Copy low product into temporary variable for PDC1L
	
UPDATE_PWM3
	movf	TABLE_OFFSET3,W			;Load the table offset for Phase 3
	movf	PLUSW0,W				;Use offset to access value in sine table via indirect addressing
	mulwf	DRIVE_FREQUENCY, W		;Table_value X DRIVE_FREQUENCY
	movff	PRODH,PDC2H_TEMP		;Copy high product into temporary variable for PDC2H
	movff	PRODL,PDC2L_TEMP		;Copy low product into temporary variable for PDC2L
	
TRUNCATE_PWM123						;Truncate results of multiply to 10 uppermost bits
	bcf		STATUS,C				;discarding lower two bits and right justifying 
	rlcf	PDC0L_TEMP,F
	rlcf	PDC0H_TEMP,F
	rlcf	PDC0L_TEMP,F
	rlcf	PDC0H_TEMP,F
	rlcf	PDC0L_TEMP,W
	andlw	0x3
	movff	PDC0H_TEMP,PDC0L_TEMP
	movwf	PDC0H_TEMP
	
	bcf		STATUS,C
	rlcf	PDC1L_TEMP,F
	rlcf	PDC1H_TEMP,F
	rlcf	PDC1L_TEMP,F
	rlcf	PDC1H_TEMP,F
	rlcf	PDC1L_TEMP,W
	andlw	0x3
	movff	PDC1H_TEMP,PDC1L_TEMP
	movwf	PDC1H_TEMP

	bcf		STATUS,C
	rlcf	PDC2L_TEMP,F
	rlcf	PDC2H_TEMP,F
	rlcf	PDC2L_TEMP,F
	rlcf	PDC2H_TEMP,F
	rlcf	PDC2L_TEMP,W
	andlw	0x3
	movff	PDC2H_TEMP,PDC2L_TEMP
	movwf	PDC2H_TEMP

	call	CHECK_LIMITS

	bsf		PWMCON1, UDIS		;Disable updates to duty cycle and period
	movff	PDC0L_TEMP,PDC0L	;Transfer temporary values into duty cycle registers
	movff	PDC0H_TEMP,PDC0H
	movff	PDC1L_TEMP,PDC1L
	movff	PDC1H_TEMP,PDC1H
	movff	PDC2L_TEMP,PDC2L
	movff	PDC2H_TEMP,PDC2H
	
	bcf		PWMCON1, UDIS		;Enable updates to duty cycle and period to update simultaneously.

	
	return	

;*******************************************************************************
;UPDATE_TABLE_OFFSET
;
;This routine Updates the offset pointers to the table after every access
;*******************************************************************************
UPDATE_TABLE_OFFSET
	btfss	FLAGS,OFFSET1_FLAG			;If set incr. on table
	bra		DECREMENT_OFFSET1
	movlw	(SINE_TABLE_ENTRIES-1)		;Check for the last value on the table
	cpfslt	TABLE_OFFSET1
	bra		CLEAR_OFFSET1_FLAG
	incf	TABLE_OFFSET1,F				;Increment offset1
	bra		UPDATE_OFFSET2

CLEAR_OFFSET1_FLAG
	bcf		FLAGS,OFFSET1_FLAG

DECREMENT_OFFSET1
	dcfsnz	TABLE_OFFSET1,F				;Decrement offset1
	bsf		FLAGS,OFFSET1_FLAG

UPDATE_OFFSET2
	btfss	FLAGS,OFFSET2_FLAG			;If set incr. on table
	bra		DECREMENT_OFFSET2
	movlw	(SINE_TABLE_ENTRIES-1)		;Check for the last value on the table
	cpfslt	TABLE_OFFSET2
	bra		CLEAR_OFFSET2_FLAG
	incf	TABLE_OFFSET2,F				;Increment offset2
	bra		UPDATE_OFFSET3

CLEAR_OFFSET2_FLAG
	bcf		FLAGS,OFFSET2_FLAG

DECREMENT_OFFSET2
	dcfsnz	TABLE_OFFSET2,F				;Decrement offset2
	bsf		FLAGS,OFFSET2_FLAG

UPDATE_OFFSET3
	btfss	FLAGS,OFFSET3_FLAG			;If set incr. on table
	bra		DECREMENT_OFFSET3
	movlw	(SINE_TABLE_ENTRIES-1)		;Check for the last value on the table
	cpfslt	TABLE_OFFSET3
	bra		CLEAR_OFFSET3_FLAG
	incf	TABLE_OFFSET3,F				;Increment offset3
	return	

CLEAR_OFFSET3_FLAG
	bcf		FLAGS,OFFSET3_FLAG

DECREMENT_OFFSET3
	dcfsnz	TABLE_OFFSET3,F				;Decrement offset3
	bsf		FLAGS,OFFSET3_FLAG
	return	

;*******************************************************************************
; CALCULATE_TIMER0_RELOAD
;
;This routine calculates the Timer0 reload value based on ADC read value and the 
;scaling factor calculated based on the main clock and number of Sine table entries.
;Timer0 value = FFFF - (FREQUENCY_SCALE/Frequency)	Frequ = (adc result)
;*******************************************************************************
CALCULATE_TIMER0_RELOAD

	clrf	TEMP
	clrf	TEMP1
	movlw	HIGH(FREQUENCY_SCALE)		;FREQUENCY_SCALE/Frequency
	movwf	TEMP_LOCATION				;16 bit by 8 bit division	     	
	movlw	LOW(FREQUENCY_SCALE)		;
	movwf	TEMP_LOCATION+1

continue_subtraction

	movf	DRIVE_FREQUENCY,W
	btfsc	STATUS,Z
	return
	bsf		STATUS,C
	movf	DRIVE_FREQUENCY,W
	subwfb	TEMP_LOCATION+1,F
	clrf	WREG
	subwfb	TEMP_LOCATION,F
	btfss	STATUS,C
	goto	keep_result_in_rpm	
	incf	TEMP,F
	btfsc	STATUS,C					;Result of the division is stored in TEMP&TEMP1
	incf	TEMP1,F
	goto	continue_subtraction	

keep_result_in_rpm	
;Timer0 value = FFFF-Timer0 
	bsf		STATUS,C
	movlw	0xFF
	subfwb	TEMP,F
	subfwb	TEMP1,F						;The Timer0 reload value stored in 
	movff	TEMP1,FREQ_REF_H			;FREQ_REF_H & FREQ_REF_L 
	movff	TEMP,FREQ_REF_L				;These values will be loaded into Timer0 to set PWM update period

	return
	
	
	
;*******************************************************************************
;CHECK_LIMIT ROUTINE
;   for frequency < 60Hz, duty cycle will be less than MAX_DUTY_CYCLE (4 x PTPER)due to 
;		the selection of sine table values
;	it is still necessary to ensure that PDC is greater or equal to MINL_DUTY_CYCLE (3 x deadtime)
;
;*******************************************************************************
CHECK_LIMITS

CHK_PWM0_MIN					;Test to see if PDC0H:PDC0L < 0:MINL_DUTY_CYCLE
	movf	PDC0H_TEMP, F		;First, is PDC0H = 0?
	bnz		CHK_PWM1_MIN		;	if no, then PDC cannot be less than minimum, check next value
	movlw	MINL_DUTY_CYCLE		;Second, is PDC0L > MINL_DUTY_CYCLE?
	cpfsgt	PDC0L_TEMP			;		if yes, then PDC is not less than minimum, check next value
	movwf	PDC0L_TEMP			;		if no, make it the minimum value

CHK_PWM1_MIN					;Test to see if PDC1H:PDC1L < 0:MINL_DUTY_CYCLE
	movf	PDC1H_TEMP, F		;First, is PDC1H = 0?
	bnz		CHK_PWM2_MIN		;	if no, then PDC cannot be less than minimum, check next value
	movlw	MINL_DUTY_CYCLE		;Second, is PDC1L > MINL_DUTY_CYCLE?
	cpfsgt	PDC1L_TEMP			;		if yes, then PDC is not less than minimum, check next value
	movwf	PDC1L_TEMP			;		if no, make it the minimum value
	
CHK_PWM2_MIN					;Test to see if PDC2H:PDC2L < 0:MINL_DUTY_CYCLE
	movf	PDC2H_TEMP, F		;First, is PDC2H = 0?
	bnz		DONE_CHECK_LIMITS	;	if no, then PDC cannot be less than minimum, check next value
	movlw	MINL_DUTY_CYCLE		;Second, is PDC2L > MINL_DUTY_CYCLE?
	cpfsgt	PDC2L_TEMP			;		if yes, then PDC is not less than minimum, check next value
	movwf	PDC2L_TEMP			;		if no, make it the minimum value

DONE_CHECK_LIMITS
	return


;*******************************************************************************
;This routine stops the motor by driving the PWMs to 0% duty cycle.
;*******************************************************************************
STOP_MOTOR
	bcf		PIE1,ADIE
	bcf		INTCON,TMR0IE
	clrf	OVDCOND			;enable output overrides before setting duty cycle to zero (due to an errata)
	clrf	TABLE_OFFSET1
	clrf	TABLE_OFFSET2
	clrf	TABLE_OFFSET3
	bcf		FLAGS, TIMER0_OV_FLAG
	
	return
	
;*******************************************************************************
;This routine starts motor from previous stop with motor parameters initialized
;*******************************************************************************
RUN_MOTOR_AGAIN
	bsf		FLAGS1,RUN_STOP	
	bcf		FLAGS,FLAG_FAULT
	bsf		PIE1,ADIE
	call	INIT_MOTOR_START
	call	UPDATE_PWM_DUTYCYCLES	;Yes, update the PWM duty cycle with new value
	call	UPDATE_TABLE_OFFSET		;Update 3 offsets
	call	PidInitialize
	clrf	RAMP_COUNTER
	clrf	TARGET_FREQUENCY
	clrf	PID_PERCENTAGE
	clrf	percent_out
	bsf		INTCON,TMR0IE		
	movlw	b'11111111'				;after duty cycles contain non-zero values, 
	movwf	OVDCOND					;OVDCOND is configured such that there is no output override
	
	return

;*******************************************************************************
;*******************************************************************************
;							FEEDBACK CONTROL SUBROUTINES
;*******************************************************************************
;*******************************************************************************

;*******************************************************************************
; RAMP_TARGET_FREQUENCY
;
;*******************************************************************************
RAMP_TARGET_FREQUENCY

	;if ramp counter = ramp constant then
	incf	RAMP_COUNTER, F
;	movlw	0x28			;RAMP CONSTANT for 5Hz/second acceleration limit
	movlw	0x3				;RAMP CONSTANT for 20Hz/second acceleration limit
	cpfseq	RAMP_COUNTER
	bra	  	rampcount_not_equal
	clrf	RAMP_COUNTER
	;compare command frequency to target freguency
	;if command frequency is greater, increment target frequency
	;if target frequency is greater, decrement target frequency
	
	movf	COMMAND_FREQUENCY, W
	cpfseq	TARGET_FREQUENCY
	bra		target_not_equal	;TARGET_FREQUENCY NOT EQUAL TO COMMAND_FREQUENCY
target_equal	
	return						; do nothing
target_not_equal
	movf	COMMAND_FREQUENCY, W
	cpfsgt	TARGET_FREQUENCY
	bra		target_less			;TARGET_FREQUENCY LESS THAN COMMAND_FREQUENCY
target_greater
	decf	TARGET_FREQUENCY 	;TARGET_FREQUENCY IS GREATER THAN COMMAND_FREQUENCY
	bra		limit_target_frequency	
target_less	
	incf	TARGET_FREQUENCY	;TARGET_FREQUENCY IS LESS THAN COMMAND_FREQUENCY		 
limit_target_frequency
	movlw	0x30							;Minimum Frequency set to 12Hz (scaling factor X4) 
	cpfsgt	TARGET_FREQUENCY				; if frequency is less than or equal to 12Hz..
	movwf	TARGET_FREQUENCY				;	set it to 12Hz
	movlw	0xF0							;Limiting V/F to F= 60Hz (scaling factor X4) 
	cpfslt	TARGET_FREQUENCY				; if frequency is greater than or equal to 60Hz..
	movwf	TARGET_FREQUENCY				;	set it to 60Hz
rampcount_not_equal	
	return
	


;*******************************************************************************
; CALCULATE_ROTOR_FREQUENCY
;
;This routine calculates the (electrical) rotor frequency based on captured VELRH:VELRL
;VELRH:VELRL corresponds to time between edges of QEA pulse.
;
;Based on Timer5 period of .2usec and (2048 QEA edge transitions)/revolution   
;
;		ROTOR_FREQUENCY = 20400d/QEAperiod = 4FB0h/(VELRH:VELRL)
;
;	a 16 bit by 16 bit unsigned divide routine is used from math library 
;*******************************************************************************
CALCULATE_ROTOR_FREQUENCY

	movlw	0x12			;constant for 1:64 velocity postscaler is 126CCCh
	movwf	AARGB0
	movlw	0x6C
	movwf	AARGB1
	movwf	0xCC
	movwf	AARGB2
	
	movff	VELOCITY_HIGH, BARGB0	;Load divisor with averaged QEAperiod, 
	movff	VELOCITY_LOW, BARGB1	;  VELOCITY_HIGH:VELOCITY_LOW->BARGB0:BARGB1
	call	FXD2416U		;call 24/16 unsigned fixed point divide routine

	movff	AARGB1, ROTOR_FREQUENCY_UPPER
	movff	AARGB2, ROTOR_FREQUENCY_LOWER

	return
	
;*******************************************************************************
; CALCULATE_SLIP_FREQUENCY
;
;		SLIP_FREQUENCY = |TARGET_FREQUENCY - ROTOR_FREQUENCY|
;		
;		If SLIP_FREQUENCY is negative, set SLIP_NEGATIVE flag.
;		
;*******************************************************************************
CALCULATE_SLIP_FREQUENCY
	bsf		STATUS, C
	movf	TARGET_FREQUENCY, W
	subfwb	ROTOR_FREQUENCY_LOWER, W
	movwf	SLIP_FREQUENCY_LOWER
	movlw	0x00
	subfwb	ROTOR_FREQUENCY_UPPER, W
	movwf	SLIP_FREQUENCY_UPPER			
	bcf		PID_FLAGS, NEGATIVE_SLIP
	bsf		pidStat1, 2, 1
	btfss	SLIP_FREQUENCY_UPPER, 7
	bra		POSITIVE_ERROR
	comf	SLIP_FREQUENCY_UPPER, F
	comf	SLIP_FREQUENCY_LOWER, F
	bsf		PID_FLAGS, NEGATIVE_SLIP
	bcf		pidStat1, 2, 1
POSITIVE_ERROR		
	return	
	
;*******************************************************************************
; CALCULATE_SLIP_PERCENTAGE
;		This 8-bit value between 0 and 100 will be used by PID algorithm
;		
;		SLIP_PERCENTAGE = (100 x SLIP_FREQUENCY)/ TARGET_FREQUENCY
;
;		0 =< SLIP_PERCENTAGE =< 100d
;		0 =< SLIP_FREQUENCY =< 240d  (240d = 60Hz x 4)
;		48d =< TARGET_FREQUENCY =< 240d  (48d = 12Hz x 4)
;
;*******************************************************************************
CALCULATE_SLIP_PERCENTAGE

	; first, 16 x 16 (or 8 x 16) multiply of 100 x SLIP_FREQUENCY
	
	clrf	AARGB0
	movlw	0x64		;load 100 into AARGB1
	movwf	AARGB1		
	
	movff	SLIP_FREQUENCY_UPPER, BARGB0
	movff	SLIP_FREQUENCY_LOWER, BARGB1
	
	call	FXM1616U	;perform multiply
	
	;product is now in AARGB0:AARGB3
	
	;with valid input parameters, result will only be 16-bits
	;add check to make sure that AARGB0 and AARGB1 are zero
	
	;next, 16 x 16 divide of above result by the TARGET_FREQUENCY
	
	movff	AARGB2, AARGB0	;move results to appropriate arguments
	movff	AARGB3, AARGB1
	
	clrf	BARGB0
	movff	TARGET_FREQUENCY, BARGB1
	
	call	FXD1616U		;call 16/16 unsigned fixed point divide routine
	
	;with valid input parameters, result is only eight bits
	;add check to make sure that AARGB0 are zero
	movff	AARGB1, SLIP_PERCENTAGE
	
	return	
	
;*******************************************************************************
; CALCULATE_DRIVE_FREQUENCY
;		Use PID output to modify Target Frequency to generate Drive Frequency
;
;		DRIVE_FREQUENCY = ((100 + PID_PERCENTAGE)* TARGET_FREQUENCY) / 100
;
;*******************************************************************************
CALCULATE_DRIVE_FREQUENCY
	;First, add 100 to PID_PERCENTAGE
	clrf	AARGB0
	clrf	AARGB1
	movlw	0x64		;load 100d into AARGB2
	movwf	AARGB2

	clrf	BARGB0
	clrf	BARGB1
	movff	PID_PERCENTAGE, BARGB2
	
	call	_24_BitAdd
	;result is in AARGB2, (only an eight bit value)
	;Next, multiply this times the TARGET_FREQUENCY	
	clrf	AARGB0
	movff	AARGB2, AARGB1	;move result into AARGB1 for multiply

	clrf	BARGB0	
	movff	TARGET_FREQUENCY, BARGB1
	
	call	FXM1616U
	;result is in AARGB2:AARGB3
	;Next, divide this result by 100
	
	movff	AARGB2, AARGB0	;move results to appropriate arguments
	movff	AARGB3, AARGB1
	
	clrf	BARGB0	
	movlw	0x64		;load 100d into BARGB1
	movwf	BARGB1
	
	call	FXD1616U
	;result is in AARGB0:AARGB1
	;if AARGB0 is not zero, then limit DRIVE_FREQUENCY to max (FF)
	;otherwise, load AARGB1 into DRIVE_FREQUENCY
	
	movff	AARGB1, DRIVE_FREQUENCY
	movlw	0x14							;Minimum Frequency set to 12Hz (scaling factor X4) 
	cpfsgt	DRIVE_FREQUENCY				; if frequency is less than or equal to 8Hz..
	movwf	DRIVE_FREQUENCY				;	set it to 8Hz
	movlw	0xFE							;Limiting V/F to F= 63Hz (scaling factor X4) 
	cpfslt	DRIVE_FREQUENCY				; if frequency is greater than or equal to 63Hz..
	movwf	DRIVE_FREQUENCY				;	set it to 63Hz
	
	
	bsf		FLAGS1, DRIVE_FREQ_UPDATE		;set flag to indicate DRIVE_FREQUENCY has been updated
	return		
	

;*******************************************************************************
;*******************************************************************************
;							KEY SWITCH SUBROUTINES
;*******************************************************************************
;*******************************************************************************

;*******************************************************************************
;This routine checks for the keys status. 2 keys are checked, Run/Stop and 
;Forward(FWD)/Reverse(REV)  
;*******************************************************************************
KEY_CHECK
	btfsc	KEY_PORT,RUN_STOP_KEY		;Is key pressed "RUN/STOP"?
	goto	CHECK_FWD_REV_KEY
	btfsc	FLAGS1,DEBOUNCE
	return
	call	KEY_DEBOUNCE
	btfss	FLAGS1,DEBOUNCE
	return
	bsf		FLAGS1,KEY_RS
	return
	
CHECK_FWD_REV_KEY
 	btfsc	KEY_PORT,FWD_REV_KEY		;Is key pressed "FWD/REV"?
	goto	SET_KEYS
	btfsc	FLAGS1,DEBOUNCE
	return
	call	KEY_DEBOUNCE
	btfss	FLAGS1,DEBOUNCE
	return
	bsf		FLAGS1,KEY_FR
	return

SET_KEYS
	btfss	FLAGS1,DEBOUNCE
	return
	bcf		FLAGS1,DEBOUNCE
	bsf		FLAGS1,KEY_PRESSED	
	btfss	FLAGS1,KEY_RS
	bra		ITS_FWD_REV	
	btg		FLAGS1,RUN_STOP
	return
ITS_FWD_REV	
	btg		FLAGS1,FWD_REV
	return

;*******************************************************************************
KEY_DEBOUNCE
	decfsz	DEBOUNCE_COUNTER,F
	return
	bsf		FLAGS1,DEBOUNCE
	movlw	DEBOUNCE_COUNT
	movwf	DEBOUNCE_COUNTER
	return
;*******************************************************************************
PROCESS_KEY_PRESSED
	btfss	FLAGS1,KEY_PRESSED
	return
	btfss	FLAGS1,KEY_RS
	goto	CHECK_FWD_REV
	btfss	FLAGS1,RUN_STOP	
	goto	STOP_MOTOR_NOW
	call	RUN_MOTOR_AGAIN
	bcf		FLAGS1,KEY_PRESSED
	bcf		FLAGS1,KEY_RS

	bsf		LED_PORT,RUN_STOP_LED	
	return

STOP_MOTOR_NOW
	call	STOP_MOTOR					;STOP motor
	bcf		FLAGS1,KEY_PRESSED
	bcf		FLAGS1,KEY_RS
	bcf		LED_PORT,RUN_STOP_LED	
	return

CHECK_FWD_REV
	btfss	FLAGS1,KEY_FR
	return

	btg		LED_PORT,FWD_REV_LED	
	bcf		LED_PORT,RUN_STOP_LED	
	call	STOP_MOTOR

	call	DELAY
	call	DELAY
	call	DELAY
	call	DELAY
	call	DELAY
	call	DELAY
	call	DELAY
	call	DELAY
	call	DELAY
	call	DELAY
	call	DELAY
	call	DELAY

	btg		FLAGS,MOTOR_DIRECTION

	call	RUN_MOTOR_AGAIN
	bcf		FLAGS1,KEY_PRESSED
	bcf		FLAGS1,KEY_FR
	bsf		LED_PORT,RUN_STOP_LED	
	return


;*******************************************************************************
; FAULTA_PROCESS
;*******************************************************************************
FAULTA_PROCESS
	bsf		PORTD,0
;	bsf		PORTD,1
	bsf		PORTD,2
	bsf		PORTC,0

	bsf		FLAGS,FLAG_FAULT
	call	STOP_MOTOR
	bcf		FLTCONFIG,FLTAS
	bcf		FLAGS1,KEY_PRESSED
	bcf		FLAGS1,KEY_RS
	bcf		FLAGS1,RUN_STOP	
	
;	goto	$-2				;trap here
	
	return	

;*******************************************************************************
; TOGGLE_LEDS
;*******************************************************************************
TOGGLE_LEDS

	incfsz	COUNTER_SP,F
	return
	incfsz	COUNTER_SP1,F
	return
	btg		PORTC,0
	btg		PORTD,0
	btg		PORTD,1
	btg		PORTD,2
	return

;*******************************************************************************
; Delay routine.
;*******************************************************************************
DELAY
	movlw	DELAY_COUNT1
	movwf	COUNTER
dec_count	
	movlw	DELAY_COUNT2
	movwf	COUNTER1
dec_count1
	decfsz	COUNTER1,F
	bra		dec_count1
	decfsz	COUNTER,F
	bra		dec_count
	clrf	COUNTER
	clrf	COUNTER1
	return		






;*******************************************************************************
;*******************************************************************************
; 						INITIALIZATION SUBROUTINES
;*******************************************************************************
;*******************************************************************************

;*******************************************************************************
;Initialize High-Speed ADC
;*******************************************************************************
INIT_HSADC
	movlw	b'00000000'			; ADCON1 is configured such that:
	movwf	ADCON1				; a) Vref+ and Vref- are Avdd and Avss, respectively.
								; b) The FIFO buffer is disabled

	movlw	b'00110010'			; ADCON2 is configured such that:
	movwf	ADCON2				; a) The A/D result is left justified (so remember to read ADRESL before reading ADRESH)
								; b) The A/D acquisition time is set to 12Tad, as required for sequential conversion.
								; c) The A/D conversion clock is set to Fosc/32.

	movlw	b'01000000'			; ADCON3 is configured such that:
	movwf	ADCON3				; a) An interrupt is generated on every 2nd and 4th write to the FIFO buffer.
								; b) No external ADC triggers are enabled.
	
	movlw	b'00000100'			; ADCHS is configured such that:
	movwf	ADCHS				; a) Group A signal is AN0, current reference  
								; b) Group B signal is AN1, speed reference
								; c) Group C signal is AN6, 

	movlw	b'01000011'			; ANSEL0 is configured such that:
	movwf	ANSEL0				; a) AN0 and AN1 and AN6 are analog input pins.
	movlw	b'00000011'			; b) Corresponding bits of TRISA are also set to inputs.
	movwf	TRISA
	bsf		TRISE, 0			; c) Corresponding bit of TRISE is also set to input.

	movlw	b'00000001'			; ANSEL1 is configured such that:
	movwf	ANSEL1				; a) AN8 is an analog input pin.
	bsf		TRISE, 2			; b) Corresponding bit of TRISE is also set to input.
	
	movlw	b'00000101'			; ADCON0 is configured such that:
	movwf	ADCON0				; a) Single shot mode is enabled
								; b) Single-channel mode is enabled
								; c) Group B signal is sampled (speed ref on development board)
								; d) The ADC is turned on.
	return
	
	
	
;*******************************************************************************
;Initialize PCPWM
;	NOTES:
; 	1)	PTPER has 12-bit resolution, 4 LSBs of PTPERH and 8 bits of PTPERL
; 	2)	In edge aligned mode, PTMR reset to zero on match with PTPER
;	3)	PDC has 14-bit resolution, 6 LSBs of PDCxH and 8 bits of PDCxL
;	4)	Lower 2 bits of PDC compared to Q clocks
;
;	5)	Resolution(of duty cycle)= log((Fosc/4)/Fpwm)/log(2) = log(5Mhz/20kHz)/log(2) = 8 bits
;		so 6 LSBs of PDCxH and 2 MSBs of PDCxL will be used.
;	   (for 16kHz, resolution = log(5Mhz/16kHz)/log(2) = 8 bits, also.)
;
;*******************************************************************************
INIT_PCPWM

	movlw	b'00000000'			; PTCON0 is configured such that:
	movwf	PTCON0			; a) Postscale value is 1:1
								; b) PWM time base input is Fosc/4
								; c) PWM time base mode is free-running for edge-aligned operation
	
	movlw	0xF9				; PTPERL and PTPERH are set up for a 20KHz PWM frequency.		
	movwf	PTPERL			; 	PTPERH:PTPERL = ((Fosc/4)/(PWMfreq x PTMRps))-1
	movlw	0x00				; 	PTPERH:PTPERL = ((20MHz/4)/(20KHz x 1))-1 = 249d = F9h
	movwf	PTPERH			
	
	movlw	b'01000000'			; PWMCON0 is configured such that:
	movwf	PWMCON0			; a) PWM0, PWM1, PWM2, PWM3, PWM4, and PWM5 are enabled for output.
								; b) All PWM I/O pairs are set to complimentary mode
										
	movlw	b'00000001'			;PWMCON1 is configured such that:
	movwf	PWMCON1				; a) Special event trigger post-scaler is set to 1:1
								; b) Special event trigger occurs when time-base is counting upwards
								; c) Updates from duty cycle and period buffer registers are enabled.
								; d) Output overrides via OVDCON are synchronous to the PWM timebase.			


	movlw	b'00001010'			;1us deadtime instead of 2us		
;	movlw	b'00010100'			;DTCON is configured such that:
	movwf	DTCON				; a) Clock source for dead-time is Fosc/2.
								; b) Dead time = Dead time value / (Fosc/2) = 2uS.								

	movlw	b'11111111'			;OVDCOND is configured such that there is no output override.
	movwf	OVDCOND
	
	movlw	b'00000000'			;OVDCONS is configured such that all PWM outputs are 0 upon power-up.
	movwf	OVDCONS
	
	
	;FLTA is overcurrent fault, muxed via configuration bit to RC1
	;FLTB is overvoltage fault on RC2
	movlw	b'10010001'			;Fault A and FaultB are enabled in catastrophic mode.
;	movlw	b'10110011'			;FLTCONFIG is configured such that: 
	movwf	FLTCONFIG			; a) Enable fault condition on break-point for use with ICD2
								; b) Enable FaultA in cycle-by-cycle mode
								; c) Enable FaultB in cycle-by-cycle mode
								; d) Fault A and Fault B disable PWM channels 0 to 5
	
	movlw	0x00				;SEVTCMPL and SEVTCMPH are clear.
	movwf	SEVTCMPL
	movlw	0x00
	movwf	SEVTCMPH

;	clrf	PDC0L				;PDC0L, PDC1L, PDC2L, PDC3L, PDC0H, PDC1H, PDC2H, PDC3H 
;	clrf	PDC1L				; 	are clear so that duty cycles are initially 0.
;	clrf	PDC2L	
;	clrf	PDC3L	
;	clrf	PDC0H	
;	clrf	PDC1H	
;	clrf	PDC2H	
;	clrf	PDC3H	


	bsf		PTCON1, PTEN		;PTEN bit in the PTCON1 is set to enable the PWM time base.
	return
	
	
	
;*******************************************************************************
;Initialize QEI
;*******************************************************************************
INIT_QEI
	bsf		CAP1CON, CAP1REN	;set this to clear TMR5 time base on each velocity event

	movlw	b'00001011'			; QEICON is configured such that:
	movwf	QEICON				; a) Velocity mode is enabled
								; b) QEI mode is set to x2 velocity input mode.
								; c) Velocity pulse reduction ratio is set to 1:64 

	movlw	b'00011100'			; TRISA is configured such that:
	iorwf	TRISA, F			; a) pins corresponding to INDX, QEA, and QEB are made inputs
								; (inclusive or is used to preserve settings of other TRISA pins)

	movlw	b'00000001'			; T5CON is configured such that:
	movwf	T5CON				; a) Special event reset is disabled				
								; b) Continuous count mode is enabled
								; c) Timer5 input clock prescaler is 1:1
								; e) Timer5 is enabled 

	movlw	b'00111010'			; Digital filter is configured
	movwf	DFLTCON				; a) QEA, QEB, and INDX filters enabled
								; b) noise filter clock divider = 1:4

	return
	
;*******************************************************************************
;Initialize IC1
;	- an alternative mode of speed measurement using input capture of index 
;	pulse from quadrature encoder.  Simulates a basic tachometer.  Uses Input Capture 1.
;*******************************************************************************
INIT_IC1
	clrf	QEICON				; Clear QEICON to make sure that QEI mode is disabled.
	
	movlw	b'01000111'			;Configure CAP1CON such that:
	movwf	CAP1CON				; a)Timer5 resets on capture event
								; b)Pulse width measurement mode every rising to falling edge
	
	bsf		TRISA, 2			; Make TRISA pin corresponding to IC1 (INDX signal from encoder)an input

	movlw	b'00000001'			; T5CON is configured such that:
	movwf	T5CON				; a) Special event reset is disabled				
								; b) Continuous count mode is enabled
								; c) Timer5 input clock prescaler is 1:1
								; e) Timer5 is enabled 

	movlw	b'00001010'			; Digital filter is configured
	movwf	DFLTCON				; a) INDX filter enabled
								; b) noise filter clock divider = 1:4
	return

;*******************************************************************************
;Initialize PORTC
;*******************************************************************************
INIT_PORTC
	movlw	b'10111110'
	movwf	TRISC
	bsf		PORTC,0
	return

;*******************************************************************************
;Initialize PORTD
;*******************************************************************************
INIT_PORTD
	movlw	b'11010000'			;RD4 is an input (Fault A)
	movwf	TRISD
	clrf	PORTD
	return

;*******************************************************************************
;Initialize Timer0
;*******************************************************************************
INIT_TMR0
	movlw	b'10000100'					;T0CON is configured such that:
	movwf	T0CON						;a) TMR0 ON, 
										;b) 16-bit operation, 
										;c) clock source is instruction cycle clock, 
										;d) prescalar is 1:16

	movlw	0xF8						;Timer0 Initialisation 
	movwf	TMR0H
	movlw	0x5E						;
	movwf	TMR0L	
	return

;*******************************************************************************
;Initialize interrupts
;*******************************************************************************
INIT_INTERRUPTS

	bsf		INTCON,TMR0IE				;Timer0 overflow Interrupt enable
	bsf		INTCON2,TMR0IP				;Timer0 overflow Interrupt high priority

	bsf		PIE1,ADIE					;AD Converter Interrupt enable (will trigger on 2nd and 4th writes to FIFO)
	bsf		IPR1,ADIP					;make it high priority
;	bsf		ADCON0, GO					;set GO bit to start conversions in continuous loop (see initHSADC)

	bsf		PIE3, IC1IE					;Enable IC1 interrupt (for QEI velocity measurement mode)
;	bcf		IPR3, IC1IP					;make sure it's low priority	
	bsf		IPR3, IC1IP					;make it high priority

	movlw	b'10010011'					;Power ON reset status bit/Brownout reset status bit
	movwf	RCON						;and Instruction flag bits are set
										;Enable Priority levels on Interrupts
	
	bsf		INTCON,GIEL					;Enable low priority interrupts
	bsf		INTCON,GIEH					;Enable high priority interrupts
		


; 3) PIE3 and IPR3 are configured such that every time a value is captured from Timer5, a low-priority interrupt is generated
	return


;*******************************************************************************
;Initialize Motor
;*******************************************************************************
INIT_MOTOR_START
	movlw	0x09						;Initialize the table offset to 3 registers
	movwf	TABLE_OFFSET1				;to form 0-120-240 degrees
	bsf		FLAGS,OFFSET1_FLAG			;Offset flags initialization
	btfss	FLAGS,MOTOR_DIRECTION
	bra		INIT_MOTOR_START_REV
	movlw	0x03
	movwf	TABLE_OFFSET2
	bcf		FLAGS,OFFSET2_FLAG
	movlw	0x0F
	movwf	TABLE_OFFSET3
	bcf		FLAGS,OFFSET3_FLAG
	bsf		PORTC,0
	bra		CONT_INIT_MOT
INIT_MOTOR_START_REV
	movlw	0x0F
	movwf	TABLE_OFFSET2
	bcf		FLAGS,OFFSET2_FLAG
	movlw	0x03
	movwf	TABLE_OFFSET3
	bcf		FLAGS,OFFSET3_FLAG

	bcf		PORTC,0

CONT_INIT_MOT
	movlw	0x30						;Initialize frequency to 12Hz
	movwf	DRIVE_FREQUENCY
	movlw	0xFD						;Timer0 Initialisation 
	movwf	FREQ_REF_H
	movwf	TMR0H
	movlw	0x2C						;
	movwf	TMR0L	
	movwf	FREQ_REF_L
	bsf		FLAGS,TIMER0_OV_FLAG
	return

;*******************************************************************************
;Upon initialization the Sine table contents are copied to the RAM from 
;Program memory 
;*******************************************************************************
COPY_TABLE_TO_RAM
	movlw	UPPER sine_table			;Initialize Table pointer to the first  
	movwf	TBLPTRU						;location of the table
	movlw	HIGH sine_table
	movwf	TBLPTRH
	movlw	LOW sine_table
	movwf	TBLPTRL
	movlw	LOW(SINE_TABLE)
	movwf	FSR0L
	movlw	HIGH(SINE_TABLE)
	movwf	FSR0H
	movlw	0x14
	movwf	TEMP
COPY_AGAIN
	TBLRD*+
	movff	TABLAT,POSTINC0
	decfsz	TEMP,F
	bra		COPY_AGAIN

	movlw	LOW(SINE_TABLE)				;FSR0 points to the starting of the table
	movwf	FSR0L
	movlw	HIGH(SINE_TABLE)
	movwf	FSR0H
	return		




;*******************************************************************************
;*******************************************************************************
;								SINE TABLE
;*******************************************************************************
;*******************************************************************************

TABLE	code 0x0600		;was 0x0100
;below table is from 270 degrees to 90 degrees @ 10 degree resolution; for 20MHz, PR2 = F9, Timer2 1:1 prescale
;sine_table db	0x00,0x02,0x08,0x11,0x1E,0x2E,0x40,0x54,0x69,0x80,0x96,0xAB,0xBF,0xD1,0xE1,0xEE,0xF7,0xFD,0xFF

;below table is from 270 degrees to 90 degrees @ 10 degree resolution; for 16MHz, PR2 = 137, Timer2 1:1 prescale
sine_table db	0x00,0x02,0x08,0x11,0x1E,0x2E,0x40,0x54,0x69,0x80,0x96,0xAB,0xBF,0xD1,0xE1,0xEE,0xF7,0xFD,0xFF

;*******************************************************************************

	END

