'
' Read and play WAV files from a Multi Media Card (MMC) formatted as FAT16
' Reads all valid files on the card sequentially
' Auto detects the WAV file samples per second. i.e. 8KHz, 11.025KHz, 16KHz, and 22.05KHz
'
' The PICmicro's Hardware PWM is used as an 8-bit DAC for the audio output
'
' For 16-bit (18F) devices only using version 3.1.7 onwards of the PROTON+ compiler
'
    REMINDERS = OFF
	OPTIMISER_LEVEL = 6
	Device = 18F252
    XTAL = 8											' Set the initial frequency to 8MHz
    PLL_REQ = TRUE										' Multiply it by 4 to make 32MHz operating frequency
	
    HSERIAL_BAUD = 19200								' Set baud rate for USART serial coms      
	HSERIAL_RCSTA = %10010000       					' Enable serial port and continuous receive
	HSERIAL_TXSTA = %00100100       					' Enable transmit and asynchronous mode
    
    ON_HARDWARE_INTERRUPT Goto DOUBLE_BUFFER_INTERRUPT	' Point the hardware interrupt vector to "DOUBLE_BUFFER_INTERRUPT"
    
    Symbol TIMER1 = TMR1L.Word							' Create a WORD variable from the TMR1L\H registers			
	Symbol TMR1IF = PIR1.0								' Timer1 interrupt overflow flag
	Symbol GIE = INTCON.7								' Global interrupts Enable\Disable
    
    Dim BUFFER_POSITION as Word SYSTEM					' Position within the buffer being played
    Dim INTERRUPT_SAMPLE_RATE as Word SYSTEM			' The rate at which the interrupt triggers
    Dim INTERRUPT_CONTROL_FLAG as Byte SYSTEM			' Used as flags and switches for interrupt control
    Dim BUSY_PLAYING_BUFFER as INTERRUPT_CONTROL_FLAG.0	' Whether a buffer is being output to the PWM
    Dim USE_BUFFER2 as INTERRUPT_CONTROL_FLAG.1			' Play from buffer 1 or buffer 2
    Dim LOAD_BUFFER1 as INTERRUPT_CONTROL_FLAG.2		' Load the sector into buffer 1
    Dim LOAD_BUFFER2 as INTERRUPT_CONTROL_FLAG.3		' Load the sector into buffer 2
    Dim PLAY_BUFFER as INTERRUPT_CONTROL_FLAG.4  		' Interrupt control flag 
       
    Dim AMOUNT_OF_FILES as Word
    Dim FILE_SELECT as Word
    Dim FILE_NOW_PLAYING as Word
    
    Dim FSR0 as FSR0L.Word								' Create a word variable from registers FSR0L\H
    Dim FSR2 as FSR2L.Word								' Create a word variable from registers FSR2L\H
    
    Symbol BARGRAPH_LEDS = PORTB						' The port for the bargraph LEDs   
    
    Symbol TRUE = 1
    Symbol FALSE = 0
    
'----------------------------------------------------------------------------------------------------------   
    Delayms 200											' Wait for things to stabilise
    ALL_DIGITAL = TRUE									' Set PORTA and PORTE to all digital

    Include "WAV_READ_MMC.INC"							' Load the MMC FAT16 WAV file reading subroutines into memory

REMINDERS = OFF   
    Goto MAIN_PROGRAM									' Jump over the interrupt subroutine
    
'----------------------------------------------------------------------------------------------------------
' Timer1 hardware interrupt handler to play from two buffers and output via the hardware PWM port
'
' Input		: PLAY_BUFFER = TRUE (1) if the interrupt is to start reading and outputting the buffer
' Output	: None
' Notes		: Uses registers FSR2L\H as a buffer pointer for fast access
'
DOUBLE_BUFFER_INTERRUPT:          
    TIMER1 = INTERRUPT_SAMPLE_RATE 						' Load Timer1 with the appropriate value for the sample rate of the WAV file
    If PLAY_BUFFER = TRUE Then							' Is the file to be played ?    
    	If USE_BUFFER2 = TRUE Then						' Yes. So are we reading from buffer 2 ?
        	FSR2 = Varptr SECTOR_BUFFER2				' Yes. So point FSR2L\H to SECTOR_BUFFER2 
    	Else											' Otherwise.. Play from buffer 1
        	FSR2 = Varptr SECTOR_BUFFER1				' And point FSR2L\H to SECTOR_BUFFER1
    	Endif  
    	FSR2 = FSR2 + BUFFER_POSITION					' Add the buffer position to FSR2L\H
        CCP1CON = CCP1CON & %11001111					' \
        WREG = INDF2 << 4								'  \  
		WREG = WREG & %00110000							'  / Load bits 4 and 5 of CCP1CON with bits 0 and 1 of INDF2
		CCP1CON = CCP1CON | WREG						' /
        CCPR1L = INDF2 >> 2								' Load CCPR1L with the remaining 6 Most Significant bits shifted into place
        Inc BUFFER_POSITION								' Move up the buffer
    	If BUFFER_POSITION.9 = 1 Then 					' End of buffer reached ? (i.e. 512)
        	BUFFER_POSITION.HighByte = 0				' Yes. So clear BUFFER_POSITION. The low byte is already clear i.e. 512 = $0100
        	Btg USE_BUFFER2 							' and use the other buffer 
        	BUSY_PLAYING_BUFFER = FALSE					' Indicate that playing has stopped and a buffer needs loading
    	Else
        	Bra $ + 2									' \ Waste 3 cycles to balance the timing of the interrupt
            Nop											' / 
        Endif  
    Endif
    Clear	TMR1IF										' Clear the Timer1 interrupt flag
    Retfie 	FAST										' Exit the interrupt, restoring WREG, STATUS and BSR   
'----------------------------------------------------------------------------------------------------------
' The main program loop starts here
MAIN_PROGRAM:
       
    Input PORTC.2											' Disable the HPWM output while things are being setup
    Clear													' Clear all RAM before starting
    Low BARGRAPH_LEDS										' Extinguish all the LEDs
'
' Initialise the MMC and the FAT16 system
'    
    Repeat
        FAT_INIT											' Read the FAT root and extract the info from it
    Until FAT_RESPONSE = TRUE								' Keep reading the root until the MMC card initialises
'
' Count the amount of valid files in the directory
'    
    AMOUNT_OF_FILES = 0
    Repeat
    	FAT_LIST_DIR										' Read a valid file
        If LIST_RESPONSE = 1 Then Inc AMOUNT_OF_FILES		' Increment the Amount of Files if a valid file found
    Until LIST_RESPONSE = 0									' Until the end of the directory is found 
'
' Setup a Timer1 overflow hardware interrupt
'      
	T1CON = %00000001                         				' Turn on Timer1, 16-bit, prescaler = 1:1
    PIR1 = %00000000										' Clear the Timer1 interrupt flag
    PIE1 = %00000001										' Enable Timer1 as a peripheral interrupt source
    TIMER1 = 0                                				' Clear Timer1    
    INTCON = %11000000                       				' Enable Global and Peripheral interrupts
'
' Setup the Hardware PWM generator for 64KHz or 80KHz frequency at approx 8-bits resolution using a 32MHz oscillator
'
   	T2CON = %00000100 										' Turn on Timer2 with a Prescaler value of 1:1
    CCPR1L = 0												' Reset the CCPR1L register
    CCP1CON = %00001100										' Turn on PWM Module 1 by setting bits 2 and 3 of CCP1CON   
    
	FILE_NOW_PLAYING = 0
	While 1 = 1      										' Create an infinite loop      
        FILE_SELECT = 0										' Reset the File selection variable
        Repeat												' Form a loop to read the directory
    		FAT_LIST_DIR									' Read a single valid file
        	If LIST_RESPONSE = 1 Then						' Have we reached a valid file ?
                If FILE_SELECT = FILE_NOW_PLAYING Then 		' Yes. So is it the next file in the directory ?
                	STARTED_LISTING_DIR = FALSE				' Yes. So reset the Directory Listing subroutine
                    Break									' And exit the loop
                Endif
                Inc FILE_SELECT								' Increment the selected file variable
        	Endif
    	Until LIST_RESPONSE = 0								' Until the end of the directory is found 
		FAT_FILENAME = FAT_INTERNAL_FILENAME
        FAT_OPEN_FILE [FAT_FILENAME] 						' Open a file for reading
		If FAT_RESPONSE = TRUE Then							' Only proceed if the file was found      	
        	Gosub FAT_READ_SECTOR1							' Discard the first sector containing the WAV file's header info
            'Gosub DISPLAY_WAV_FORMAT					' << Uncomment to serially display the WAV file format
'
' Auto select the appropriate timing for different WAV file sample rates
'
            PR2 = 89										' Set PWM frequency to 88KHz, 8.5 bits of resolution (Better for 22KHz samples)
            INTERRUPT_SAMPLE_RATE = 65186					' Default to 22.05KHz samples
            Select WAV_SAMPLE_RATE							' What sample rate is the file ?
            	Case 8000									' 8KHz ?
                	INTERRUPT_SAMPLE_RATE = 64546			' Set Timer1 interrupt rate accordingly
                    PR2 = 124								' Set PWM frequency to 64KHz, 8.9 bits of resolution (Better for 8KHz samples)
            	Case 11025									' 11.025KHz ?
                    INTERRUPT_SAMPLE_RATE = 64818			' Set Timer1 interrupt rate accordingly
                    PR2 = 119								' Set PWM frequency to 66.15KHz, 8.9 bits of resolution (Better for 11.025KHz samples)           	
                Case 16000 									' 16KHz ?
                	INTERRUPT_SAMPLE_RATE = 65026			' Set Timer1 interrupt rate accordingly
                    PR2 = 99								' Set PWM frequency to 80KHz with 8.6 bits of resolution (Better for 16KHz samples)
            End Select
        	Gosub FAT_READ_SECTOR1							' Pre-Read a sector into the first buffer
            Gosub FAT_READ_SECTOR2							' Pre-Read a sector into the second buffer
            BUFFER_POSITION = 0								' Make sure the buffer position is 0
        	BUSY_PLAYING_BUFFER = TRUE						' Force a "no sector read" situation to start with
        	USE_BUFFER2 = FALSE								' Start with playing from SECTOR_BUFFER1         
        	Output PORTC.2 									' Enable PORTC.2 (CCP1) as output for PWM
            PLAY_BUFFER = TRUE								' Enable the buffer playing interrupt          
        	While 1 = 1										' Create a loop to read the file into the buffers
				If BUSY_PLAYING_BUFFER = FALSE Then 		' Does a buffer require loading from the MMC ?
                    If USE_BUFFER2 = FALSE Then				' Yes. So is it buffer 2 being played ?
                		Gosub FAT_READ_SECTOR2				' No. So load buffer 2
            		Else									' Otherwise...
                		Gosub FAT_READ_SECTOR1				' Load buffer 1
                	Endif
                    BUSY_PLAYING_BUFFER = TRUE				' Indicate to the interrupt that the buffer has been loaded
                    If FAT_RESPONSE = FALSE Then Break		' Exit the play loop if the End Of File is reached
                    Gosub ILLUMINATE_BARGRAPH				' Display the bargraph on the LEDs based upon the contents of BARGRAPH_BYTE
            	Endif
            Wend
    		Input PORTC.2									' Disable the HPWM output while another file is being setup      
   			PLAY_BUFFER = FALSE								' Disable the buffer playing interrupt
        Endif                           
        Inc FILE_NOW_PLAYING								' Increment the current file being played variable
        If FILE_NOW_PLAYING = AMOUNT_OF_FILES Then 			' Have we reached the last file in the directory ?
        	FILE_NOW_PLAYING = 0							' Yes. So start again at the beginning
        Endif
SKIP_AUTO_INCREMENT:
		Clear BARGRAPH_LEDS
	Wend
'-------------------------------------------------------------------------------
'
' Illuminate the bar graph LEDs in sympathy with the WAV file being played
'
' Input		: PRODL holds the last byte of the sector buffer
' Output	: None
' Notes		: None
'
ILLUMINATE_BARGRAPH:
	PRODL = ABS PRODL										' Get the absolute value of the variable
    Select PRODL 
    	Case 0 to 100
      		BARGRAPH_LEDS = %11111111
		Case 101 to 105
      		BARGRAPH_LEDS = %11111110
       	Case 106 to 108 
          	BARGRAPH_LEDS = %11111100
       	Case 109 to 112
            BARGRAPH_LEDS = %11111000
      	Case 113 to 116  
         	BARGRAPH_LEDS = %11110000
       	Case 117 to 119 
           	BARGRAPH_LEDS = %11100000
       	Case 120 to 122 
           	BARGRAPH_LEDS = %11000000
       	Case 123 to 125
          	BARGRAPH_LEDS = %10000000
       	Case 126 to 127
          	BARGRAPH_LEDS = %00000000
   	End Select
	Return
'-------------------------------------------------------------------------------
' Serially Transmit the WAV file format
'
' Uncomment to use
'
'DISPLAY_WAV_FORMAT:
	  
    'Hrsout "PLAYING FILE : ",FAT_FILENAME,13
    'Hrsout "-------------------\r" 
    'Hrsout "SIZE OF WAV : ", Dec WAV_SIZE,13
    'Hrsout "NO# CHANNELS : ", Dec WAV_NUMBER_OF_CHANNELS,13
    'Hrsout "SAMPLE RATE : ", Dec WAV_SAMPLE_RATE, " Hz\r"
    'Hrsout "BITS PER SAMPLE : ", Dec WAV_BITS_PER_SAMPLE,13
    'Return
                  
