;**********************************************************************
;                                                                     *
;                        Software License Agreement                   *
;                                                                     *
;    The software supplied herewith by Microchip Technology           *
;    Incorporated (the "Company") for its dsPIC controller            *
;    is intended and supplied to you, the Company's customer,         *
;    for use solely and exclusively on Microchip dsPIC                *
;    products. The software is owned by the Company and/or its        *
;    supplier, and is protected under applicable copyright laws. All  *
;    rights are reserved. Any use in violation of the foregoing       *
;    restrictions may subject the user to criminal sanctions under    *
;    applicable laws, as well as to civil liability for the breach of *
;    the terms and conditions of this license.                        *
;                                                                     *
;    THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION.  NO           *
;    WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING,    *
;    BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND    *
;    FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE     *
;    COMPANY SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL,  *
;    INCIDENTAL OR CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.  *
;                                                                     *
;*********************************************************************/

;**********************************************************************
;                                                                     *
;    Filename:	    AN1047.asm				                          *
;    Date:          5/12/2006                                         *
;    File Version:  1.0                                               *
;                                                                     *
;    Author:        Steve Bowling                                     *
;    Company:       Microchip Technology                              *
;                                                                     * 
;                                                                     *
;**********************************************************************
;                                                                     *
;    Other Files required:                                            *
;                    16F785.lkr                                       *
;                 or 16F785i.lkr (if using ICD2)                      *
;                                                                     *
;**********************************************************************
;                                                                     *
;    Notes:                                                           *
;                                                                     *
;    This application implements a constant current LED driver using  *
;    the PIC16F785 device.  The supply current is measured instead of *
;    LED current to simplify the sensing requirements.                *
;                                                                     *
;    Configuration:                                                   *
;                                                                     *
;    This code is setup to use OpAmp #2 and Comparator #1.  You may   *
;    wish to change those assignments based on your application       *
;    hardware.  The PWM is setup for 250KHz switching frequency.      *
;    Modify the SetupPWM function as required to suit your component  *
;    values.                                                          *
;**********************************************************************

	list      p=16F785           ; list directive to define processor
	#include <p16F785.inc>        ; processor specific variable definitions

	errorlevel  -302              ; suppress message 302 from list file

	__CONFIG _CP_OFF & _WDT_OFF & _MCLRE_ON & _FCMEN_ON & _IESO_ON & _BOD_OFF & _BOR_OFF & _PWRTE_ON & _INTRC_OSC_NOCLKOUT & _CPD_OFF
; '__CONFIG' directive is used to embed configuration word within .asm file.
; The lables following the directive are located in the respective .inc file.
; See data sheet for additional information on configuration word settings.



;***** VARIABLE DEFINITIONS


	udata_shr   
w_temp		res     1		; variable used for context saving 
status_temp	res     1		; variable used for context saving
pclath_temp	res		1		; variable used for context saving

BtnFlags	res		1		; flag register for button debounce
Vbus		res		1		; Holds the measured bus voltage
Duty		res		1		; Holds the duty cycle for LED brightness
PerCount	res		1		; Holds the count reference for LED brightness


; bit definitions for 'BtnFlags' register
Btn1Event	equ		0
Btn2Event	equ		1
Btn1Press	equ		2
Btn2Press	equ		3

; Button pins on PORTB
Button1		equ		7
Button2		equ		6


;**********************************************************************
STARTUP 	code	0x0				; processor reset vector
		goto    Main              	; go to beginning of program
		nop
		nop
		retfie						; Interrupt vector not used


PROG1		code

; This is a data table that adjusts the current compare reference
; voltage as a function of the measured input voltage. The CCP1 
; module is used in PWM mode and the output is filtered with a RC
; circuit to provide the reference voltage.  This table returns a
; duty cycle for the CCP1 module.
; The purpose of this table is to maintain a constant power input 
; to the LED. The 16F785 is regulating the supply current, not the
; LED current.  For a given input voltage, this table provides a
; supply current reference level that will provide the necessary
; supply power to get the desired LED power level.  The LED drive
; power level is reduced at lower input voltages to avoid over-
; stressing the drive electronics.
; This table assumes that a DC offset of 1.8V is present at the
; output of the current sensing op-amp.
CalibrationTable:
		addwf	PCL
		retlw	.0		; 0
		retlw	.0		; 1
		retlw	.0		; 2
		retlw	.0		; 3
		retlw	.0		; 4
		retlw	.0		; 5
		retlw	.0		; 6
		retlw	.0		; 7
		retlw	.0		; 8
		retlw	.0		; 9
		retlw	.0		; 10
		retlw	.0		; 11
		retlw	.0		; 12
		retlw	.0		; 13
		retlw	.0		; 14
		retlw	.0		; 15
		retlw	.0		; 16
		retlw	.0		; 17
		retlw	.0		; 18
		retlw	.0		; 19
		retlw	.0		; 20
		retlw	.0		; 21
		retlw	.0		; 22
		retlw	.0		; 23
		retlw	.0		; 24
		retlw	.0		; 25
		retlw	.132	; 26  (Vin = 3.0V)
		retlw	.133	; 27
		retlw	.134	; 28
		retlw	.135	; 29
		retlw	.136	; 30
		retlw	.136	; 31
		retlw	.137	; 32
		retlw	.138	; 33
		retlw	.138	; 34	(Vin = 4.0V)
		retlw	.139	; 35
		retlw	.139	; 36
		retlw	.140	; 37
		retlw	.140	; 38
		retlw	.141	; 39
		retlw	.141	; 40
		retlw	.142	; 41
		retlw	.142	; 42
		retlw	.143	; 43	(Vin = 5.0V)
		retlw	.143	; 44
		retlw	.143	; 45
		retlw	.144	; 46
		retlw	.144	; 47
		retlw	.144	; 48
		retlw	.145	; 49
		retlw	.145	; 50
		retlw	.145	; 51	(Vin = 6.0V)
		retlw	.145	; 52
		retlw	.146	; 53
		retlw	.146	; 54
		retlw	.146	; 55
		retlw	.146	; 56
		retlw	.147	; 57
		retlw	.147	; 58
		retlw	.147	; 59
		retlw	.147	; 60	(Vin = 7.0V)
		retlw	.147	; 61
		retlw	.148	; 62
		retlw	.148	; 63
		retlw	.148	; 64
		retlw	.148	; 65
		retlw	.148	; 66
		retlw	.148	; 67
		retlw	.149	; 68	(Vin = 8.0V)
		retlw	.148	; 69
		retlw	.148	; 70
		retlw	.147	; 71
		retlw	.147	; 72
		retlw	.146	; 73
		retlw	.146	; 74
		retlw	.145	; 75
		retlw	.145	; 76
		retlw	.145	; 77	(Vin = 9.0V)
		retlw	.144	; 78
		retlw	.144	; 79
		retlw	.143	; 80
		retlw	.143	; 81
		retlw	.143	; 82
		retlw	.142	; 83
		retlw	.142	; 84
		retlw	.141	; 85	(Vin = 10.0V)
		retlw	.141	; 86
		retlw	.141	; 87
		retlw	.140	; 88
		retlw	.140	; 89
		retlw	.140	; 90
		retlw	.139	; 91
		retlw	.139	; 92
		retlw	.139	; 93
		retlw	.139	; 94	(Vin = 11.0V)
		retlw	.138	; 95
		retlw	.138	; 96
		retlw	.138	; 97
		retlw	.138	; 98
		retlw	.137	; 99
		retlw	.137	; 100
		retlw	.137	; 101
		retlw	.137	; 102	(Vin = 12.0V)
		retlw	.136	; 103	
		retlw	.136	; 104
		retlw	.136	; 105
		retlw	.136	; 106
		retlw	.136	; 107
		retlw	.135	; 108
		retlw	.135	; 109
		retlw	.135	; 110
		retlw	.135	; 111	(Vin = 13.0V)
		retlw	.134	; 112
		retlw	.134	; 113
		retlw	.134	; 114
		retlw	.133	; 115
		retlw	.134	; 116
		retlw	.133	; 117
		retlw	.133	; 118
		retlw	.133	; 119	(Vin = 14.0V)
		retlw	.133	; 120
		retlw	.133	; 121
		retlw	.133	; 122
		retlw	.132	; 123
		retlw	.132	; 124
		retlw	.132	; 125
		retlw	.132	; 126
		retlw	.132	; 127
		retlw	.132	; 128


; these first 4 instructions are not required if the internal oscillator is not used
Main
		;Setup internal oscillator speed
		banksel	OSCCON
		bsf		OSCCON,IRCF0
		bsf		OSCCON,IRCF1
		bsf		OSCCON,IRCF2	; Set RC oscillator to 8 MHz
				
		clrf	Vbus			; Variable for DC bus voltage
		clrf	PerCount		; Variable for software PWM timebase
		
		movlw	.5
		movwf	Duty			; Variable for software PWM duty cycle
		
		banksel	PORTA
		clrf	PORTA
		clrf	PORTB
		clrf	PORTC
				
		; Setup outputs
		banksel	TRISA
		bcf		TRISA,5			; status LED
		bcf		TRISC,1			; Ph 1 PWM output
		
		; Setup CCP for PWM mode to make adj. voltage reference.	
		call	SetupCCP1
		; Opamp #2 is used to amplify the current sensing resistor
		call	SetupOpAmp
		; Comparator #1 is used for PWM shutdown.
		call	SetupComparator
		; Setup ADC to sample bus voltage
		call	SetupADC
		; Enable the PWM module
		call	SetupPWM
		
	
;----------------------------------------------------------------------
; This is the main software loop that does the following tasks:
;	1.  Generate a 100Hz software PWM using Timer2 overflows
;       that modulates the LED to control brightness.
;	2.  Check buttons to raise and lower the brightness level.
;	3.  Measure bus voltage and LED voltage.
;	4.  Sets a comparator reference voltage that will deliver
;       the appropriate amount of current to the LED.
;----------------------------------------------------------------------
		
MainLoop:
		; Wait for Timer2 to expire.  This occurs at a 7.8KHz rate. A 
		; software count, PerCount, is incremented each time Timer 2
		; expires.  The PerCount variable and another variable, Duty,
		; are used to generate a low frequency PWM control for the LED
		; brightness.  The frequency of the PWM signal is approx. 100 Hz.
		banksel	PIR1
WaitTimer2:
		btfss	PIR1,T2IF
		goto	WaitTimer2
		
		; If Timer2 has expired, run the following code:
		bcf		PIR1,T2IF		; Clear the interrupt flag.
		incf	PerCount		; Increment the period count register.
		
		; Check to see if there is a period match (78 TMR2 cycles)
		movlw	.78
		subwf	PerCount,W
		btfss	STATUS,Z
		goto	DutyCheck		; no period match, go check duty
				
		; When the period match occurs, clear the PerCount variable,
		; Enable the PWM to drive the LED, start a conversion, and
		; check the buttons.
PeriodMatch:
		clrf	PerCount		; reset the period count

TurnOnLED:
		banksel	PWMCON0
		bsf		PWMCON0,PH1EN
		goto	StartConversion
				
DutyCheck:
		; Check to see if PerCount = Duty
		; Turn off PWM output when match occurs.
		movf	Duty,W
		subwf	PerCount,W
		btfss	STATUS,Z
		goto 	MainLoop

TurnOffLED:
		banksel PWMCON0
		bcf		PWMCON0,PH1EN
		goto 	MainLoop


StartConversion:		
		banksel	ADCON0
		bsf		ADCON0,GO		
				
		; Check button 1 for a press
CheckBtn1:
		call	Btn1Handler
		btfss	BtnFlags,Btn1Event
		goto	CheckBtn2
		; Increment duty cycle by 5 when button is pressed, but
		; limit max duty cycle to 75.
		movlw	.75
		subwf	Duty,W
		btfsc	STATUS,Z
		goto	MaxDuty
		movlw	.5
		addwf	Duty
MaxDuty:		
		bcf		BtnFlags,Btn1Event
		
		
		; Check button 2 for a press		
CheckBtn2:
		call	Btn2Handler
		btfss	BtnFlags,Btn2Event
		goto	CheckADC
		; Decrement duty cycle by 5 when button is pressed, but
		; limit min duty cycle to 5.
		movlw	.5
		subwf	Duty,W
		btfsc	STATUS,Z
		goto	MinDuty
		movlw	.5
		subwf	Duty
MinDuty:				
		bcf		BtnFlags,Btn2Event

CheckADC:		
		banksel	ADCON0
WaitConversion:		
		btfsc	ADCON0,GO
		goto	WaitConversion
		
		banksel	ADRESH
		movf	ADRESH,W
		; Save ADC result in Vbus variable for use in other parts
		; of software.
		movwf	Vbus
		
		; Use the Vbus value to set the current reference value.
SetCurrentRef:
		call	CalibrationTable
		banksel	CCPR1L
		movwf	CCPR1L
		goto	MainLoop
		
		
;----------------------------------------------------------------------
; The button handler functions set a flag when a button press is 
; detected.  The function then waits for the button to be released
; before an action is taken.
;----------------------------------------------------------------------

Btn1Handler:
		banksel	PORTB
		btfsc	PORTB,Button1		; 
		goto	Btn1Release
		bsf		BtnFlags,Btn1Press
		return
Btn1Release:
		btfss	BtnFlags,Btn1Press
		return
		bcf		BtnFlags,Btn1Press
		bsf		BtnFlags,Btn1Event
		return

;----------------------------------------------------------------------

Btn2Handler:
		banksel	PORTB
		btfsc	PORTB,Button2		; 
		goto	Btn2Release
		bsf		BtnFlags,Btn2Press
		return
Btn2Release:
		btfss	BtnFlags,Btn2Press
		return
		bcf		BtnFlags,Btn2Press
		bsf		BtnFlags,Btn2Event
		return

;----------------------------------------------------------------------

SetupCCP1:
		banksel TRISC
		bsf		TRISC,5
		movlw	0xFF
		movwf	PR2
		banksel	CCP1CON
		movlw	0x0C
		movwf	CCP1CON
		movlw	.0
		movwf	CCPR1L
		bcf		PIR1,TMR2IF
		clrf	TMR2
		bsf		T2CON,TMR2ON
CheckT2Overflow:
		btfss	PIR1,TMR2IF
		goto	CheckT2Overflow
		banksel	TRISC
		bcf		TRISC,5
		
		return

;----------------------------------------------------------------------

SetupPWM
		banksel PWMPH1
		movlw	0x20
		movwf	PWMPH1
		movlw	0x1F		;value for 250 Khz
		movwf	PWMCLK
		movlw	0x81
		movwf	PWMCON0
	
		return

;----------------------------------------------------------------------

SetupADC:
		movlw	0x07
		banksel ANSEL0
		movwf	ANSEL0
		movlw	0x50
		banksel ADCON1
		movwf	ADCON1
		movlw	0x09
		banksel ADCON0
		movwf	ADCON0
		bsf		ADCON0,GO
		return
		
;----------------------------------------------------------------------

		; Comparator 1 on
		; Comparator output pin disabled.
		; Comparator output polarity inverted
		; High speed mode
		; Reference connected to external reference
		; (comparator reference generated by CCP module)
		; OA2 is input		
SetupComparator:
		movlw	0x9A			
		banksel CM1CON0
		movwf	CM1CON0
		return		
		
;----------------------------------------------------------------------

SetupOpAmp:
		; Turn on opamp	
		banksel	OPA1CON
		bsf		OPA2CON,OPAON
		return
	
	end                       ; directive 'end of program'

