'*****************************************************************************
'*  Name    : TMR_TX-18.bas                                                  *
'*  Author  : Darrel Taylor                                                  *
'*  Date    : 4/6/2008                                                       *
'*  Version : 1.0                                                            *
'*  Notes   :                                                                *
'*          :                                                                *
'*****************************************************************************
' Available Commands:                                                        *
'  TX?C       Const            - send a Char (constant)                      *
'  TX?B       Value            - Send a byte from a BYTE var                 *
'  TX?W       Value            - Send a binary Word var                      *
'  TX_DEC?B   Value            - Ascii Decimal Byte, no leading 0's          *
'  TX_DEC?W   Value            - Ascii Decimal Word, no leading 0's          *
'  TX_DEC?CB  Len, Value       - Ascii Decimal WORD, Len is a constant       *
'  TX_DEC?CW  Len, Value       - Ascii Decimal WORD, Len is a constant       *
'  TX_STR?C   Str, Len         - Send an Array, Len is a constant            *
'  TX_STR?B   Str, BLen        - Send an Array, Len is BYTE var              *
'  LoadStr    Array, Str, Len  - Load an Array from flash, Returns           *
'                                 the length of the string in BYTE var       *
'  TX_FSTR   Fstr              - Send Fixed String from Flash                *
'  TX_DSTR   Dstr              - Send PreDefined String from Flash           *
'*****************************************************************************
disable debug
                              
TXbyte      VAR BYTE                     ; Current Byte being sent (destroyed)
TXcount     VAR BYTE                     ; bit counter
TXflags     VAR BYTE                     ; TX bit flags
  TXstart   VAR TXflags.0                ;   Start bit in progress
  TXstop    VAR TXflags.1                ;   Stop bit in progress
  ArrayOP   VAR TXflags.2                ;   Flash read goes to an Array
ReloadTemp  VAR BYTE                     ; temp var for reloading timer

STRaddress  var WORD                     ; used for Array copies
STRlength   VAR BYTE
  
High TX_Pin                              ; Start at Idle level
TXflags = 0                              ; reset bit flags

GOTO OverTMR_TX                          ; Jump over the Module

ASM
;----[Make sure User specified a valid Timer]---------------------------------
  if (TX_TIMER != 0) && (TX_TIMER != 1) && (TX_TIMER != 3)
      ERROR "Invalid TX_TIMER"(TX_TIMER) specified
      messg "TX_TIMER must be 0, 1 or 3"
  endif

;----[Reset a Timer] ---(set it to 16-bit, 1:1, OFF and zero)-----------------
ClearTimer  macro Tmr
    MOVE?CB  0, T#v(Tmr)CON              ; clear the T?CON register
    if (Tmr == 0)
        MOVE?CT  1, T0CON,PSA            ; prescaler NOT assigned (1:1)
        #define TMRIF  INTCON,TMR0IF
    endif
    if (Tmr == 1)
        MOVE?CT  1, T1CON,RD16           ; R/W timer as 16-bit
        #define TMRIF  PIR1,TMR1IF
    endif
    if (Tmr == 3)
        MOVE?CT  1, T3CON,RD16           ; R/W timer as 16-bit
        #define TMRIF  PIR2,TMR3IF
    endif
    MOVE?CB  0, TMR#v(Tmr)H              ; Clear Timer Value
    MOVE?CB  0, TMR#v(Tmr)L
  endm
ENDASM


;---[reload a Timer using RD16]-----------------------------------------------
ASM
ReloadTmrRD16?C  macro  Tmr, TimerConst
    movf    TMR#v(Tmr)L,W            ;   get the RD16 timer value
    movwf   _ReloadTemp              ; 1 save LowByte
    movlw   high(TimerConst)         ; 2
    addwf   TMR#v(Tmr)H,F            ; 3  add HighBytes
    movlw   low(TimerConst)          ; 4  add LowBytes    
    addwf   _ReloadTemp,W            ; 5  but don't put back in timer    
    btfsc   STATUS,C                 ; 6/7    
    incf    TMR#v(Tmr)H,F            ; 7 inc HighByte if carry    
    movwf   TMR#v(Tmr)L              ; 8 then put lowbyte in timer
                                     ;   also loads HB with RD16  endm
  endm
ReloadCount = 8
ENDASM

;---[ Find TX timer Load @ Baud rate ] ---------------------------------------
ASM
MINcount = 50
Tcount = (((OSC * 10000000) / 4) / TX_BAUD + 5) / 10  ; Timer counts per bit
ReLoad = 65536 - Tcount + ReloadCount                 ; value to reload
    if (Tcount < MINcount)
        ERROR  TX baud rate(TX_BAUD) is too high for OSC mhz crystal
    endif
    if (ReLoad < 0)
        ERROR  TX baud rate(TX_BAUD) is too Low for OSC mhz crystal
        variable ReLoad = ReLoad & 0xFFFF          ; stop some of the warnings
    endif
ENDASM

;---[Send serial data, using Timer No INT]------------------------------------ 
TX_Jump:                               ; Jumpman entry
@   MOVE?BB  FSR0L, _TXbyte            ;  byte to send was saved in FSR0L

TX_Send:                               ; Data is in TXbyte
@   ClearTimer  TX_TIMER
  TX_Loop:
    IF TXstart THEN                    ; if Start bit has been sent
        IF TXcount > 0 THEN            ;   and there's more bits to send
            TX_Pin = TXbyte.0          ;     copy LSB to Pin
            TXbyte = TXbyte >> 1       ;     Shift data right 1
            TXcount = TXcount - 1      ;     decrement bit count
        ELSE                           ;   No more bits left
            HIGH TX_Pin                ;     set Pin to Idle state
            TXstop = 1                 ;     starting Stop bit
        ENDIF
    else                               ;  start bit hasn't been sent yet
        LOW TX_Pin                     ;     set Pin low
        TXstart = 1                    ;     starting the Start bit
        TXcount = 8                    ;     8 bits to shift out
    endif
ASM
    ReloadTmrRD16?C  TX_TIMER, ReLoad  ; reload the period count
    MOVE?CT   1, T#v(TX_TIMER)CON,TMR#v(TX_TIMER)ON  ; start the Timer
    MOVE?CT  0, TMRIF                  ; clear the flag
WaitForTimer                           
    clrwdt                             ; kick the dog
    btfss  TMRIF
    goto WaitForTimer                  ; wait for timer to ovrflow
ENDASM

    IF !TXstop then TX_Loop            ; if not the Stop bit, do it again
    TXflags = 0                        ; Reset TX flags
                                                    ; We're done, 
@   MOVE?CT  0, T#v(TX_TIMER)CON, TMR#v(TX_TIMER)ON ; Turn off the timer
return

ASM
  ifdef SEROUT2DEC_USED
TX_DEC?B  macro Value                  ; Ascii Decimal BYTE, no leading 0's
    MOVE?CB  0, R4 + 1
    MOVE?BW  Value, R2
    MOVE?CW _TX_Jump, R8
    L?CALL  SEROUT2DEC
  endm

TX_DEC?CB  macro Len, Value            ; Ascii Decimal BYTE, Len is a constant
    MOVE?CB  Len, R4 + 1
    MOVE?BW  Value, R2
    MOVE?CW _TX_Jump, R8
    L?CALL  SEROUT2DEC
  endm

TX_DEC?W  macro Value                  ; Ascii Decimal Word, no leading 0's
    MOVE?CB  0, R4 + 1
    MOVE?WW  Value, R2
    MOVE?CW _TX_Jump, R8
    L?CALL  SEROUT2DEC
  endm

TX_DEC?CW  macro Len, Value            ; Ascii Decimal WORD, Len is a constant
    MOVE?CB  Len, R4 + 1
    MOVE?WW  Value, R2
    MOVE?CW _TX_Jump, R8
    L?CALL  SEROUT2DEC
  endm
  endif
  
TX?B  macro Value                      ; Send a byte from a BYTE var
    MOVE?BB  Value, _TXbyte
    L?CALL   _TX_Send
  endm
  
TX?W  macro Value                      ; Send a binary Word var
    MOVE?BB  Value, _TXbyte            ;   low byte first
    L?CALL   _TX_Send
    MOVE?BB  Value + 1, _TXbyte
    L?CALL   _TX_Send
  endm
  
TX?C  macro Const                      ; send a Char (constant)
    MOVE?CB  Const, _TXbyte
    L?CALL   _TX_Send
  endm
  
  ifdef SEROUT2STRN_USED
TX_STR?C  macro Str, Len               ; Send an array, Len is a constant
    MOVE?CB  Len, R4 + 1
    MOVE?CW  _TX_Jump, R8
    MOVE?CB  high (Str), FSR2H
    MOVE?CA  low (Str)
    L?CALL  SEROUT2STRN
  endm
  
TX_STR?B  macro Str, BLen              ; Send an array, Len is BYTE var
    MOVE?BB  BLen, R4 + 1
    MOVE?CW  _TX_Jump, R8
    MOVE?CB  high (Str), FSR2H
    MOVE?CA  low (Str)
    L?CALL  SEROUT2STRN
  endm
  endif
ENDASM

ASM
Expand_LoadStrCode  macro
    ifndef LoadStrCode_Expanded        ; only expand it once
        variable LoadStrCode_Expanded = 1
    ENDASM
      GOTO OverLoadStr               ; make sure you can't "Fall" into it
        LoadString:
            @ TBLRD*+
            if TABLAT = 0 then LoadStrDone
            IF ArrayOP then
                POKE STRaddress, TABLAT
            else
                TXbyte = TABLAT
                gosub  TX_Send
            endif
            STRaddress = STRaddress + 1
            STRlength = STRlength + 1
            GOTO LoadString
        LoadStrDone:
        return
      OverLoadStr:
    ASM    
    endif 
  endm

LoadStr  macro  Array, Str, Len        ; Load an array from flash, returns 
  local TheStr, OverStr                ; the length of the string in BYTE var
    Expand_LoadStrCode                 ; expand the main code once, if needed
    goto  OverStr
TheStr
    data Str,0                         ; store the string in Flash
OverStr
    MOVE?CW  Array, _STRaddress        ; Set the address of the Array
    MOVE?CB    low(TheStr), TBLPTRL    ; Set the address of the String
    MOVE?CB   high(TheStr), TBLPTRH
    MOVE?CB  Upper(TheStr), TBLPTRU
    MOVE?CB  0, _STRlength             ; zero the Length
    MOVE?CT  1, _ArrayOP               ; indicate String goes into an Array
    L?CALL   _LoadString               ; do it
    MOVE?BB  _STRlength, Len           ; copy Length to users variable
  endm


TX_FSTR  macro Fstr                    ; Send Fixed String from Flash
  local TheStr, OverStr                
    Expand_LoadStrCode                 ; expand the main code, if needed
    goto  OverStr
TheStr
    data Fstr,0                        ; store the string in Flash
OverStr
    MOVE?CB    low(TheStr), TBLPTRL    ; Set the address of the String
    MOVE?CB   high(TheStr), TBLPTRH
    MOVE?CB  Upper(TheStr), TBLPTRU
    MOVE?CB  0, _STRlength             ; begin with zero length
    MOVE?CT  0, _ArrayOP               ; indicate String goes to TX_Pin
    L?CALL   _LoadString               ; do it
  endm


TX_DSTR  macro Dstr                    ; Send PreDefined String from Flash
    Expand_LoadStrCode                 ; expand the main code, if needed
    movlw    low(Dstr)
    movwf    TBLPTRL
    movlw    high(Dstr)
    movwf    TBLPTRH
    movlw    upper(Dstr)
    movwf    TBLPTRU
    MOVE?CB  0, _STRlength
    MOVE?CT  0, _ArrayOP               ; indicate String goes to TX_Pin
    L?CALL   _LoadString               ; do it
  endm

string  macro Str
    data  Str,0
  endm
ENDASM

@ ifdef DoesNotCompile                 ; Force PBP to include certain routines
    DEBUG DEC R0                       ; SEROUT2DEC
    DEBUG STR TXflags\TXflags          ; SEROUT2STRN
;    DEBUG STR TXflags                  ; SEROUT2STR
@ endif
OverTMR_TX:

enable debug
