/*********************************************************************
 *
 *                Microchip Full Speed USB Demo Tool Version 1.0
 *
 *********************************************************************
 * FileName:        codeDeviceManager.cpp
 * Dependencies:    See INCLUDES section below
 * Processor:       PIC18
 * Compiler:        Borland C++ Builder 6.0
 * Company:         Microchip Technology, Inc.
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the Company) for its PICmicro Microcontroller is intended and
 * supplied to you, the Companys customer, for use solely and
 * exclusively on Microchip PICmicro Microcontroller products. The
 * software is owned by the Company and/or its supplier, and is
 * protected under applicable copyright laws. All rights are reserved.
 * Any use in violation of the foregoing restrictions may subject the
 * user to criminal sanctions under applicable laws, as well as to
 * civil liability for the breach of the terms and conditions of this
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN AS IS CONDITION. NO WARRANTIES,
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Ravi Kumar M        11/04/04     Original.
 ********************************************************************/


#pragma hdrstop
/* I N C L U D E S **********************************************************/
//---------------------------------------------------------------------------
#include "codeDeviceManager.h"

/** V A R I A B L E S ********************************************************/
//---------------------------------------------------------------------------
int Device::UsedDevIDs[4] = {0,0,0,0};
HINSTANCE libHandle;

/** D E C L A R A T I O N S **************************************************/
//---------------------------------------------------------------------------
/******************************************************************************
 * Function:        Device::Device(AnsiString deviceName,
 *                  int Mode, int assignedIndex,int communicationChannelType)
 *
 * PreCondition:    None
 *
 * Input:           deviceName - Used to set DeviceName member
 *                  Mode - Used to set CurrentModeOfOperation member
 *                  assignedIndex - Used to set the the AssignedIndex member
 *                  communicationChannelType - Used to set the Channel member to
 *                  to either USB or RS-232 communication object.
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Constructor. Initalizes class members with
 *                  the arguments passed.
 *
 * Note:            None
 *****************************************************************************/
Device::Device(AnsiString deviceName, int Mode, int assignedIndex,int communicationChannelType)
{
    DeviceName = deviceName;
    CurrentModeOfOperation = Mode;
    AssignedIndex = assignedIndex;
    Disabled = false;
    Channel = NULL;
    if (communicationChannelType == 0) // USB
    {
        Channel = new USB(assignedIndex);
        if (Mode == 1 ) // Demo Mode
        {
            // Assign appropriate ID
            // There can be a maximum of 4 devices
            int i;
            for (i = 0; i < 4; i++)
            {
                if (UsedDevIDs[i] == 0) {
                    DemoDevID = i;
                    UsedDevIDs[i] = 0x1;
                    break;
                }
            }
            if (i == 4) {
                // No place exists for this device to make a connection.
                DemoDevID = -1;
                DeviceName = "DISABLED - " + DeviceName ;
                Disabled = true;
            }
        }
    }
    else if (communicationChannelType == 1) // RS-232
    {
        // Transfer the required data
        Channel = new COMPort();
    }
}

/******************************************************************************
 * Function:        Device::~Device()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Destructor. Responsible for deleting the Channel object
 *                  from memory.
 *
 * Note:            None
 *****************************************************************************/

_fastcall Device::~Device()
{
  if (CurrentModeOfOperation == 1) //Demo Mode
        UsedDevIDs[DemoDevID] = 0;
  if (ByteBuffer) delete ByteBuffer;
  if (Channel) delete Channel;
}

/******************************************************************************
 * Function:        int Device::OpenDevice()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          Error Code
 *
 * Side Effects:    None
 *
 * Overview:    Performs operations based on the type of communication channel.
 *              Opens the communication channel using the Open member function.
 *
 * Note:            None
 *****************************************************************************/

int Device::OpenDevice()
{                            
    if (Disabled) return 1;
    // if the device is USB type
    if (Channel->CommunicationChannelType() == 0)
    {

    }

    // if the device is COM Port Type
    if (Channel->CommunicationChannelType() == 1)
    {

    }
    
    return Channel->Open(CurrentModeOfOperation );
}

/******************************************************************************
 * Function:        int Device::OpenDevice()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          Error Code
 *
 * Side Effects:    None
 *
 * Overview:    Closes previously opened device.
 *              
 *
 * Note:            None
 *****************************************************************************/
int Device::CloseDevice()
{
    if (Disabled) return 1;
    // if the device is USB type

    Channel->Close();
    // if the device is COM port type
    return 0;
}
/******************************************************************************
 * Function:        int Device::SendData(void *Bytes, int Length)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          Error Code
 *
 * Side Effects:    None
 *
 * Overview:       Sends data using the Channel Object
 *
 *
 * Note:            Not Implemented
 *****************************************************************************/
int Device::SendData(void *Bytes, int Length)
{
    if (Disabled) return 1;
}
/******************************************************************************
 * Function:        int Device::ReceiveData(void *Bytes, int &Length)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          Error Code
 *
 * Side Effects:    None
 *
 * Overview:       Receivess data using the Channel Object
 *
 *
 * Note:            Not Implemented
 *****************************************************************************/
int Device::ReceiveData(void *Bytes, int &Length)
{
    if (Disabled) return 1;
}
/******************************************************************************
 * Function:        int Device::GetModeOfOperation()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          Current Mode of operation of the selected
 *                  device.
 * Side Effects:    None
 *
 * Overview:       Returns current mode of operation of this device.
 *
 *
 * Note:            None
 *****************************************************************************/

int Device::GetModeOfOperation()
{
    return  CurrentModeOfOperation;
}
/******************************************************************************
 * Function:        int Device::SendReceivePacket(BYTE *SendData,
 *                  DWORD SendLength, BYTE *ReceiveData, DWORD *ReceiveLength,
 *                  UINT SendDelay, UINT ReceiveDelay)
 *
 * PreCondition:    Open the device using OpenDevice().
 *
 * Input:           SendData - Array of bytes to be sent to Channel object
 *                  SendLength - Send data Length
 *                  ReceiveData - Array of Bytes to be received.
 *                  ReceiveLength - Pointer intialized to Expected Length.
 *                                  After receiving the data it stores the received
 *                                  data length.
 *                  SendDelay - Delay used during Sending the data
 *                  ReceiveDelay - Delay used during receiving the data
 *
 * Output:          Error code.
 *
 * Side Effects:    None
 *
 * Overview:        This function sends data and receives data inresponse to the
 *                  data sent. It calls the SendReceivePacket member of the Channel
 *                  member object.
 * Note:            None
 *****************************************************************************/
int Device::SendReceivePacket(BYTE *SendData, DWORD SendLength, BYTE *ReceiveData, DWORD *ReceiveLength,UINT SendDelay, UINT ReceiveDelay)
{

    return Channel->SendReceivePacket(SendData,SendLength,ReceiveData,ReceiveLength,SendDelay, ReceiveDelay);
}


/******************************************************************************
 * Function:        DeviceManager::DeviceManager()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Constructor. Initializes the members variables.
 *                  Initializes the CurrentDevice member to NULL.
 *                  Initializes the DeviceList member to new List.
 *
 * Note:            None
 *****************************************************************************/

DeviceManager::DeviceManager():CurrentDevice(NULL)
{
    DeviceList = new TObjectList;
    if (DeviceList) DeviceList->OwnsObjects = true;
    AbortOperation = IsOpen = false;
}
/******************************************************************************
 * Function:        DeviceManager::~DeviceManager()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Destructor.
 *                  Deletes the memory allocated to DeviceList.
 *
 *
 * Note:            None
 *****************************************************************************/
DeviceManager::~DeviceManager()
{
    if (DeviceList) delete DeviceList;
  //  if(libHandle != NULL) FreeLibrary(libHandle);
}
/******************************************************************************
 * Function:        void DeviceManager::StartExecution()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Calls the Refresh() there by retriving the information on
 *                  devices connected to the PC. The DeviceList stores the
 *                  information on the objects.
 *
 * Note:            None
 *****************************************************************************/
void DeviceManager::StartExecution()
{
  Refresh();
}
/******************************************************************************
 * Function:        void DeviceManager::StopExecution()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Clears all details about the devices.
 *
 *
 * Note:            None
 *****************************************************************************/
void DeviceManager::StopExecution()
{
   DeviceList->Clear();
}
/******************************************************************************
 * Function:        void DeviceManager::Refresh()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Whenever this function is invoked, the DeviceList member
 *                  is cleared reinitialized with the details of the USB devices
 *                  connected. It creates the list for both boot mode as well as
 *                  demo mode.
 *
 * Note:            None
 *****************************************************************************/
void DeviceManager::Refresh()
{

  DWORD count;
  AnsiString temp;

  Device *dev;

  // Scan for USB Devices and place them in the
  // Device List
  DeviceList->Clear();

  if (ActiveMode == 0)
  {
    count = MPUSBGetDeviceCount(picdemfs_boot);
    for(UINT i = 0; i < count; i++)
    {
        temp="";
        temp.cat_sprintf("PICDEM FS USB %d (Boot)",i);
        dev = new Device(temp,0/*BOOT*/,i,0/*USB*/);
        DeviceList->Add(dev);
    }
  }//end if

  if (ActiveMode == 1)
  {
    count = MPUSBGetDeviceCount(picdemfs_demo);
    for(UINT i = 0; i < count; i++)
    {
        temp = "";
        temp.cat_sprintf("PICDEM FS USB %d (Demo)",i);
        dev = new Device(temp,1/*DEMO*/,i,0/*USB*/);
        DeviceList->Add(dev);
    }
  }//end if

//  if (ActiveMode == 1) // DEMO
//  {
//     for(UINT i = 0; i < 4; i++)
//     {
//        temp = "";
//        temp.cat_sprintf("COM%d (Demo)",i);
//        dev = new Device(temp,1/*DEMO*/,i,1/*RS-232*/);
//        DeviceList->Add(dev);
//     }
//  }

}

/******************************************************************************
 * Function:        AnsiString DeviceManager::GetDeviceList()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          List of device identifiers(names) delimited by "\r\n"
 *
 * Side Effects:    None
 *
 * Overview:        Returns the a string containing all the device names.
 *                  delimited by "\r\n".
 *
 *
 * Note:            None
 *****************************************************************************/
AnsiString DeviceManager::GetDeviceList()
{
    AnsiString NameList;
    // Return List of all Devices connected

    for (int i = 0; i < DeviceList->Count; i++)
    {

        Device *Obj = (Device *)DeviceList->Items[i];
        NameList += (Obj->DeviceName+"\r\n");
    }
    return NameList;
}
/******************************************************************************
 * Function:        BOOL DeviceManager::SelectDevice(AnsiString deviceName)
 *
 * PreCondition:    None
 *
 * Input:           deviceName - The device name
 *
 * Output:          TRUE - Device selected
 *                  FALSE - No device exists with that name.
 * Side Effects:    None
 *
 * Overview:        Selects a device and assigns it to the variable
 *                  CurrentDevice.
 *
 *
 * Note:            None
 *****************************************************************************/
BOOL DeviceManager::SelectDevice(AnsiString deviceName)
{
 //   if (CurrentDevice != NULL) CurrentDevice->CloseDevice();
    Refresh();
    for (int i = 0; i < DeviceList->Count; i++)
    {
        CurrentDevice = NULL;
        Device *Obj = (Device *)DeviceList->Items[i];
        if (Obj->DeviceName == deviceName && Obj->GetModeOfOperation() == this->ActiveMode )
        {
             CurrentDevice = Obj;
 //            CurrentDevice->OpenDevice();
             return true;
        }
    }
    return false;
}
/******************************************************************************
 * Function:        BOOL DeviceManager::OpenDevice()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          TRUE - Device opened
 *                  FALSE - Device is not opened.
 * Side Effects:    None
 *
 * Overview:        Opens the CurrentDevice.
 *
 *
 * Note:            None
 *****************************************************************************/
BOOL DeviceManager::OpenDevice()
{
    if (CurrentDevice != NULL && IsOpen == false) {
        if(CurrentDevice->OpenDevice() == 0)
        {
            IsOpen = true;
            return true;
        }
        else
            return false;
    }
    return false;
}
/******************************************************************************
 * Function:        BOOL DeviceManager::CloseDevice()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          TRUE - Device closed successfully
 *                  FALSE - Device is not closed.
 * Side Effects:    None
 *
 * Overview:        Closes the CurrentDevice.
 *
 *
 * Note:            None
 *****************************************************************************/
BOOL DeviceManager::CloseDevice()
{
    if (CurrentDevice != NULL && IsOpen) {
        IsOpen = false;
        CurrentDevice->CloseDevice();
        return true;
    }
    Refresh();
    return false;
}
/******************************************************************************
 * Function:        int DeviceManager::ResetBoard()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 * Side Effects:    None
 *
 * Overview:        Sends a command to reset the baord
 *
 *
 * Note:            None
 *****************************************************************************/
int DeviceManager::ResetBoard()
{
  if (CurrentDevice == NULL) return 0;
  BYTE buf[64],buf1[64];
  DWORD RecvLength=0;

  buf[0] = RESET;

  if( CurrentDevice->SendReceivePacket(buf,1,buf1,&RecvLength) == 0) return 1;
}
/******************************************************************************
 * Function:        int DeviceManager::ReadVersion()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          BoardVersion - Returns a string containing error message
 *                  or the correct board version.
 * Side Effects:    None
 *
 * Overview:        Sends a command to read the version.
 *
 *
 * Note:            None
 *****************************************************************************/

AnsiString DeviceManager::ReadVersion()
{
  if (CurrentDevice == NULL)    return AnsiString("Demo board not selected.");
  BYTE buf[64],buf1[64];
  DWORD RecvLength=4;

  buf[0] = READ_VERSION;
  buf[1] = 0x02;

  AnsiString Version;

  Version = "(Failed to obtain Version)";
  if ( CurrentDevice->SendReceivePacket(buf,2,buf1,&RecvLength,1000,1000) == 1)
  {
    if (RecvLength == 4 && buf1[0] == 0x00 && buf1[1] == 0x02)
    {
        Version = "";
        Version.cat_printf("%d.%d",buf1[3],buf1[2]);
    }
  }
  return Version;
}
/******************************************************************************
 * Function:        int DeviceManager::ReadFLASH(UINT Len, UINT Addr, BYTE *Data)
 *
 * PreCondition:    None
 *
 * Input:           Len - Number of bytes to be read
 *                  Addr - Start Address 
 *
 * Output:          Data - Array to store the bytes read.
 *                  Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation    
 * Side Effects:    None
 *
 * Overview:        Sends a command to read FLASH and stores the received data
 *                  in to argument Data
 *
 * Note:            None
 *****************************************************************************/
int DeviceManager::ReadFLASH(UINT Len, UINT Addr, BYTE *Data)
{
  if (CurrentDevice == NULL) return 0;

  BYTE buf[64];
  BYTE buf1[0xFF+5];
  DWORD RecvLength,ExpectedReceiveLength;
  UINT StartAddr, BytesRead = 0;

  AbortOperation = false;
  while(!AbortOperation)
  {
      Application->ProcessMessages(); 
      buf[0] = READ_FLASH;
      if (Len-BytesRead <= 0) break;
      if (Len-BytesRead >= 59) RecvLength = buf[1] = 59;
      else  RecvLength = buf[1] = Len-BytesRead;

      RecvLength = RecvLength + 5;
      ExpectedReceiveLength = RecvLength;

      buf[2] = (Addr + BytesRead) & 0xFF; // Addr Low
      buf[3] = ((Addr + BytesRead) & 0xFF00) >> 0x08; // Addr High
      buf[4] = ((Addr + BytesRead) & 0xFF0000) >> 0x10;// Addr Upper

      int RetVal = CurrentDevice->SendReceivePacket(buf,5,buf1,&RecvLength,1000,3000);
      if ( RetVal == 1 || RetVal == 2)
      {
        if (memcmp(buf1,buf,2) == 0) // Data received properly
        {
          int i;
          i = 0;
          for( i = 0; i < RecvLength-5; i++)
              Data[BytesRead+i] = buf1[5+i];
        }
        else return 2;
      }
      else return 1;
      BytesRead = BytesRead + buf[1];
  }
  if (AbortOperation) return 100;
  return 0;
}
/******************************************************************************
 * Function:        int DeviceManager::ReadEEDATA(UINT Len, UINT Addr, BYTE *Data)
 *
 * PreCondition:    None
 *
 * Input:           Len - Number of bytes to be read
 *                  Addr - Start Address 
 *
 * Output:          Data - Array to store the bytes read.
 *                  Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation    
 * Side Effects:    None
 *
 * Overview:        Sends a command to read EEPROM and stores the received
 *                  data in to argument Data
 *
 * Note:            None
 *****************************************************************************/

int DeviceManager::ReadEEDATA(UINT Len, UINT Addr, UCHAR *Data)
{
  if (CurrentDevice == NULL) return 0;

  BYTE buf[64];
  BYTE buf1[0xFF+5];
  DWORD RecvLength,ExpectedReceiveLength;
  UINT StartAddr, BytesRead = 0;

  AbortOperation = false;
  while(!AbortOperation)
  {
      Application->ProcessMessages();
      //buf[0] = READ_CONFIG;
      buf[0] = READ_EEDATA;
      if (Len-BytesRead <= 0) break;
      if (Len-BytesRead >= 59) RecvLength = buf[1] = 59;
      else  RecvLength = buf[1] = Len-BytesRead;

      RecvLength = RecvLength + 5;
      ExpectedReceiveLength = RecvLength;

      buf[2] = (Addr + BytesRead) & 0xFF; // Addr Low
      buf[3] = ((Addr + BytesRead) & 0xFF00) >> 0x08; // Addr High
      buf[4] = ((Addr + BytesRead) & 0xFF0000) >> 0x10;// Addr Upper

      if (CurrentDevice->SendReceivePacket(buf,5,buf1,&RecvLength,1000,1000) == 1)
      {
        if (memcmp(buf1,buf,2) == 0) // Data received properly
        {
          memcpy(&Data[BytesRead],&buf1[5],RecvLength-5);
        }
        else return 2;
      }
      else return 1;
      BytesRead = BytesRead + buf[1];
  }
  if (AbortOperation) return 100;
  return 0;
}
/******************************************************************************
 * Function:        int DeviceManager::ReadCONFIG(UINT Addr, BYTE *Data)
 *
 * PreCondition:    None
 *
 * Input:           Addr - Start Address 
 *
 * Output:          Data - Array to store the bytes read.
 *                  Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation    
 * Side Effects:    None
 *
 * Overview:        Sends a command to read a CONFIG register and stores the received
 *                  data in to argument Data
 *
 * Note:            None
 *****************************************************************************/

int DeviceManager::ReadCONFIG(UINT Addr, UCHAR *Data)
{
  if (CurrentDevice == NULL) return 0;

  BYTE buf[64];
  BYTE buf1[0xFF+5];
  DWORD RecvLength,ExpectedReceiveLength;
  UINT StartAddr;

  buf[0] = READ_CONFIG;
  buf[1] = 1;
  RecvLength = 1;
  RecvLength = RecvLength + 5;
  ExpectedReceiveLength = RecvLength;

  buf[2] = (Addr ) & 0xFF; // Addr Low
  buf[3] = ((Addr) & 0xFF00) >> 0x08; // Addr High
  buf[4] = ((Addr) & 0xFF0000) >> 0x10;// Addr Upper

  if (CurrentDevice->SendReceivePacket(buf,5,buf1,&RecvLength,1000,1000) == 1)
  {
        if (memcmp(buf1,buf,2) == 0) // Data received properly
        {
          Data[0]=buf1[5];
        }
        else return 2;
  }
  else return 1;
  return 0;
}

/******************************************************************************
 * Function:        int DeviceManager::WriteFLASH(AnsiString& MemoryBuffer)
 *
 * PreCondition:    None
 *
 * Input:           MemoryBuffer - Formatted buffer containing memory data. 
 *
 * Output:          Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation    
 * Side Effects:    None
 *
 * Overview:        Sends the required commands to program and verify FLASH. 
 *
 * Note:            None
 *****************************************************************************/

int DeviceManager::WriteFLASH(AnsiString& MemoryBuffer)
{
  if (CurrentDevice == NULL) return 0;

  BYTE buf[64],buf2[64];
  BYTE buf1[0xFF+5];
  DWORD RecvLength,ExpectedReceiveLength,StartAddr;

  UINT BufferLength = MemoryBuffer.Length();
  UINT Index = 2,i;
  int RetVal;

  char *DataPtr = MemoryBuffer.c_str();

  // Erase the device first
  if ( (RetVal = EraseFLASH( 0x8000-0x800,0x800)) != 0) return RetVal;

  AbortOperation = false;
  while(Index <= BufferLength && !AbortOperation)
  {
      Application->ProcessMessages();
      // Read Start Address
      sscanf(&DataPtr[Index],"%06X",&StartAddr);
      Index = Index + 7;

      if (StartAddr < 0x800) {
            Index += (3 * 16);
            Index += 2;
            continue;
      }

      // Read the remaining Bytes of Data starting from this address
      int DummyLine=0;
      for (i = 0; i < 16 && Index <BufferLength; i++)
      {
            if (DataPtr[Index] != 'X')
                sscanf(&DataPtr[Index],"%02X",&buf[5+i]);
            else {buf[5+i] = 0xFF;DummyLine++;}
            Index+=3;
      }
      Index+=2;

      // Pad with 0xFF to make a 16 byte packet
      if (i < 16)
      {
        for(; i < 16; i++)
             buf[5+i] = 0xFF;
      }

      // The entire line is containing XX charecters.
      if (DummyLine == 16 ) continue;

      // Write the 16 bytes to memory

      buf[0] = WRITE_FLASH;
      buf[1] = i;
      RecvLength = 1;
      ExpectedReceiveLength = RecvLength;

      buf[2] = (StartAddr ) & 0xFF; // Addr Low
      buf[3] = ((StartAddr) & 0xFF00) >> 0x08; // Addr High
      buf[4] = ((StartAddr) & 0xFF0000) >> 0x10;// Addr Upper

      if (CurrentDevice->SendReceivePacket(buf,5+i,buf1,&RecvLength,500,1000) == 1)
      {
        if (buf1[0] != WRITE_FLASH) // Data received properly
        {
          return 2;
        }

      }
      else return 1;
      // Read the 16 bytes to memory

      buf2[0] = READ_FLASH;
      buf2[1] = i;
      RecvLength = 5+i;
      ExpectedReceiveLength = RecvLength;

      buf2[2] = (StartAddr ) & 0xFF; // Addr Low
      buf2[3] = ((StartAddr) & 0xFF00) >> 0x08; // Addr High
      buf2[4] = ((StartAddr) & 0xFF0000) >> 0x10;// Addr Upper

      //if (CurrentDevice->SendReceivePacket(buf2,5+16,buf1,&RecvLength,500,1000) == 1)
      if (CurrentDevice->SendReceivePacket(buf2,5,buf1,&RecvLength,500,1000) == 1)
      {
        if (buf1[0] != READ_FLASH) // Data received properly
        {
          return 2;
        }

      }
      else return 1;

      // Verify if the data written exists in the memory
      if(memcmp(&buf[5],&buf1[5],i) != 0)
        return -1;
  }
  if (AbortOperation) return 100;
  return 0;
}
/******************************************************************************
 * Function:        int DeviceManager::WriteUID(AnsiString& MemoryBuffer)
 *
 * PreCondition:    None
 *
 * Input:           MemoryBuffer - Formatted buffer containing memory data. 
 *
 * Output:          Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation    
 * Side Effects:    None
 *
 * Overview:        Sends the required commands to program and verify UID. 
 *
 * Note:            None
 *****************************************************************************/

int DeviceManager::WriteUID(AnsiString& MemoryBuffer)
{
  if (CurrentDevice == NULL) return 0;

  BYTE buf[64],buf2[64];
  BYTE buf1[0xFF+5];
  DWORD RecvLength,ExpectedReceiveLength,StartAddr;

  UINT BufferLength = MemoryBuffer.Length();
  UINT Index = 2,i;
  int RetVal;

  char *DataPtr = MemoryBuffer.c_str();

  // Erase the device first
  if ( (RetVal = EraseFLASH( 64,0x200000)) != 0) return RetVal;

  AbortOperation = false;
  while(Index <= BufferLength && !AbortOperation)
  {
      Application->ProcessMessages();
      // Read Start Address
      sscanf(&DataPtr[Index],"%06X",&StartAddr);
      Index = Index + 7;

      if (StartAddr < 0x800) {
            Index += (3 * 16);
            Index += 2;
            continue;
      }

      // Read the remaining Bytes of Data starting from this address
      int DummyLine=0;
      for (i = 0; i < 16 && Index <BufferLength; i++)
      {
            if (DataPtr[Index] != 'X')
                sscanf(&DataPtr[Index],"%02X",&buf[5+i]);
            else {buf[5+i] = 0xFF;DummyLine++;}
            Index+=3;
      }
      Index+=2;

      // Pad with 0xFF to make a 16 byte packet
      if (i < 16)
      {
        for(; i < 16; i++)
             buf[5+i] = 0xFF;
      }

      // The entire line is containing XX charecters.
      if (DummyLine == 16 ) continue;

      // Write the 16 bytes to memory

      buf[0] = WRITE_FLASH;
      buf[1] = i;
      RecvLength = 1;
      ExpectedReceiveLength = RecvLength;

      buf[2] = (StartAddr ) & 0xFF; // Addr Low
      buf[3] = ((StartAddr) & 0xFF00) >> 0x08; // Addr High
      buf[4] = ((StartAddr) & 0xFF0000) >> 0x10;// Addr Upper

      if (CurrentDevice->SendReceivePacket(buf,5+i,buf1,&RecvLength,500,1000) == 1)
      {
        if (buf1[0] != WRITE_FLASH) // Data received properly
        {
          return 2;
        }

      }
      else return 1;
      // Read the 16 bytes to memory

      buf2[0] = READ_FLASH;
      buf2[1] = i;
      RecvLength = 5+i;
      ExpectedReceiveLength = RecvLength;

      buf2[2] = (StartAddr ) & 0xFF; // Addr Low
      buf2[3] = ((StartAddr) & 0xFF00) >> 0x08; // Addr High
      buf2[4] = ((StartAddr) & 0xFF0000) >> 0x10;// Addr Upper

      //if (CurrentDevice->SendReceivePacket(buf2,5+16,buf1,&RecvLength,500,1000) == 1)
      if (CurrentDevice->SendReceivePacket(buf2,5,buf1,&RecvLength,500,1000) == 1)
      {
        if (buf1[0] != READ_FLASH) // Data received properly
        {
          return 2;
        }

      }
      else return 1;

      // Verify if the data written exists in the memory
      if(memcmp(&buf[5],&buf1[5],i) != 0)
        return -1;
  }
  if (AbortOperation) return 100;
  return 0;
}
/******************************************************************************
 * Function:        int DeviceManager::WriteEEDATA(AnsiString& MemoryBuffer)
 *
 * PreCondition:    None
 *
 * Input:           MemoryBuffer - Formatted buffer containing memory data. 
 *
 * Output:          Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation    
 * Side Effects:    None
 *
 * Overview:        Sends the required commands to program and verify EEDATA. 
 *
 * Note:            None
 *****************************************************************************/
int DeviceManager::WriteEEDATA(AnsiString& MemoryBuffer)
{
  if (CurrentDevice == NULL) return 0;

  BYTE buf[64];
  BYTE buf1[0xFF+5],buf2[0xFF+5];
  DWORD RecvLength,ExpectedReceiveLength,StartAddr;

  UINT BufferLength = MemoryBuffer.Length();
  UINT Index = 2,i;
  char *DataPtr = MemoryBuffer.c_str();

  AbortOperation = false;
  while(Index <= BufferLength && !AbortOperation)
  {
      Application->ProcessMessages();
      // Read Start Address
      sscanf(&DataPtr[Index],"%06X",&StartAddr);
      Index = Index + 7;

      // Read the remaining Bytes of Data starting from this address
      for (i = 0; i < 16 && Index <BufferLength; i++)
      {
            sscanf(&DataPtr[Index],"%03X",&buf[5+i]);
            Index+=3;
      }
      Index+=2;

      // Write the bytes to memory

      //buf[0] = WRITE_FLASH;
      buf[0] = WRITE_EEDATA;
      buf[1] = i;
      RecvLength = 1;
      ExpectedReceiveLength = RecvLength;

      buf[2] = (StartAddr ) & 0xFF; // Addr Low
      buf[3] = ((StartAddr) & 0xFF00) >> 0x08; // Addr High
      buf[4] = ((StartAddr) & 0xFF0000) >> 0x10;// Addr Upper

      if (CurrentDevice->SendReceivePacket(buf,5+i,buf1,&RecvLength,1000,1000) == 1)
      //if (CurrentDevice->SendReceivePacket(buf,5,buf1,&RecvLength,1000,1000) == 1)
      {
        //if (buf1[0] != WRITE_FLASH) // Data received properly
        if (buf1[0] != WRITE_EEDATA)
        {
          return 2;
        }

      }
      else return 1;

      buf2[0] = READ_EEDATA;
      buf2[1] = i;
      RecvLength = 5+i;
      ExpectedReceiveLength = RecvLength;

      buf2[2] = (StartAddr ) & 0xFF; // Addr Low
      buf2[3] = ((StartAddr) & 0xFF00) >> 0x08; // Addr High
      buf2[4] = ((StartAddr) & 0xFF0000) >> 0x10;// Addr Upper

      //if (CurrentDevice->SendReceivePacket(buf2,5+16,buf1,&RecvLength,500,1000) == 1)
      if (CurrentDevice->SendReceivePacket(buf2,5,buf1,&RecvLength,500,1000) == 1)
      {
        if (buf1[0] != READ_EEDATA) // Data received properly
        {
          return 2;
        }
      }
      else return 1;

      // Verify if the data written exists in the memory
      if(memcmp(&buf[5],&buf1[5],i) != 0)
        return -1;

  } // End of programming.
  if (AbortOperation) return 100;
  return 0;

}
/******************************************************************************
 * Function:        int DeviceManager::WriteCONFIG(AnsiString& MemoryBuffer)
 *
 * PreCondition:    None
 *
 * Input:           MemoryBuffer - Formatted buffer containing memory data. 
 *
 * Output:          Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation    
 * Side Effects:    None
 *
 * Overview:        Sends the required commands to program and verify CONFIG Memory. 
 *
 * Note:            None
 *****************************************************************************/
int DeviceManager::WriteCONFIG(AnsiString& MemoryBuffer)
{
  if (CurrentDevice == NULL) return 0;

  BYTE buf[64];
  BYTE buf1[0xFF+5],buf2[0xFF+5];
  DWORD RecvLength,ExpectedReceiveLength,StartAddr;
  UINT BufferLength = MemoryBuffer.Length();
  UINT Index = 2,i;
  char *DataPtr = MemoryBuffer.c_str();

  AbortOperation = false;
  while(Index <= BufferLength && !AbortOperation)
  {
      Application->ProcessMessages();
      // Read Start Address
      sscanf(&DataPtr[Index],"%06X",&StartAddr);
      Index = Index + 7;

      // Read the remaining Bytes of Data starting from this address
      int DummyLine=0;
      for (i = 0; i < 0xE && Index <BufferLength; i++)
      {
            if (DataPtr[Index] != 'X')
                sscanf(&DataPtr[Index],"%02X",&buf[5+i]);
            else {buf[5+i] = 0xFF;DummyLine++;}
            Index+=3;
      }
      Index+=2;
      // The entire line is containing XX charecters.
      if (DummyLine == 16 ) continue;

      // If the Address is UID Memory Region
      // don't program the locations.
      if (StartAddr == 0x3FFFFE || StartAddr == 0x3FFFFF) continue; 
      // Write the 16 bytes to memory

      buf[0] = WRITE_CONFIG;
      buf[1] = i;
      RecvLength = 1;
      ExpectedReceiveLength = RecvLength;

      buf[2] = (StartAddr ) & 0xFF; // Addr Low
      buf[3] = ((StartAddr) & 0xFF00) >> 0x08; // Addr High
      buf[4] = ((StartAddr) & 0xFF0000) >> 0x10;// Addr Upper

      if (CurrentDevice->SendReceivePacket(buf,5+i,buf1,&RecvLength,500,1000) == 1)
      {
        if (buf1[0] != WRITE_CONFIG) // Data received properly
        {
          return 2;
        }

      }
      else return 1;
      // Read the 16 bytes to memory

      buf2[0] = READ_CONFIG;
      buf2[1] = i;
      RecvLength = 5+i;
      ExpectedReceiveLength = RecvLength;

      buf2[2] = (StartAddr ) & 0xFF; // Addr Low
      buf2[3] = ((StartAddr) & 0xFF00) >> 0x08; // Addr High
      buf2[4] = ((StartAddr) & 0xFF0000) >> 0x10;// Addr Upper

      if (CurrentDevice->SendReceivePacket(buf2,5,buf1,&RecvLength,500,1000) == 1)
      {
        if (buf1[0] != READ_CONFIG) // Data received properly
        {
          return 2;
        }

      }
      else return 1;

      // Verify if the data written exists in the memory
      if(memcmp(&buf[5],&buf1[5],i) != 0)
        return -1;
  } // End of programming.
  if (AbortOperation) return 100;
  return 0;
}

/******************************************************************************
 * Function:        int DeviceManager::EraseFLASH(UINT Len, UINT Addr)
 *
 * PreCondition:    None
 *
 * Input:           Len - Length of memory region to be erased
 *                  Addr - Start Address
 *
 * Output:          Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation    
 * Side Effects:    None
 *
 * Overview:        Sends the required commands to erase FLASh 
 *
 * Note:            None
 *****************************************************************************/


int DeviceManager::EraseFLASH(UINT Len, UINT Addr)
{
  if (CurrentDevice == NULL) return 0;
  BYTE buf[64];
  BYTE buf1[0xFF+5],buf2[0xFF+5];
  DWORD RecvLength,ExpectedReceiveLength;
  UINT StartAddr, BytesErased = 0;
  UINT BlockSize = 64; // 18F can erase 64 bytes of data at a time.

  RecvLength = 1;

  AbortOperation = false;
  while(!AbortOperation)
  {
      Application->ProcessMessages();
      buf[0] = ERASE_FLASH;
      if (Len-BytesErased <= 0) break;
      buf[1] = 0x1;

      ExpectedReceiveLength = RecvLength;

      buf[2] = (Addr + BytesErased) & 0xFF; // Addr Low
      buf[3] = ((Addr + BytesErased) & 0xFF00) >> 0x08; // Addr High
      buf[4] = ((Addr + BytesErased) & 0xFF0000) >> 0x10;// Addr Upper

      if (CurrentDevice->SendReceivePacket(buf,5,buf1,&RecvLength,1000,5000) == 1)
      {
        if (buf1[0] != ERASE_FLASH) // Data received properly
            return 2;
      }
      else return 1;
      BytesErased = BytesErased + BlockSize;
  }
  if (AbortOperation) return 100;
  return 0;
}

/*** DEMO MODE FUNCTIONALITY ***/

/******************************************************************************
 * Function:        int DeviceManager::SetBoardID(UINT ID)
 *
 * PreCondition:    None
 *
 * Input:           ID - Used to set the identification value of demo mode board.
 *
 *
 * Output:          Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation    
 * Side Effects:    None
 *
 * Overview:        Sends the required commands to erase FLASh 
 *
 * Note:            Not Implemented
 *****************************************************************************/

int DeviceManager::SetBoardID(UINT ID)
{
  if (CurrentDevice == NULL) return 0;
}
/******************************************************************************
 * Function:        int DeviceManager::UpdateLED(UINT led, int State)
 *
 * PreCondition:    None
 *
 * Input:           led - LED number
 *                  State - On = 1/Off = 0
 *
 * Output:          Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation
 * Side Effects:    None
 *
 * Overview:        Sends the required commands update the LEDs
 *
 * Note:            None
 *****************************************************************************/
int DeviceManager::UpdateLED(UINT led, int State)
{
  if (CurrentDevice == NULL) return 0;
  BYTE buf[64],buf1[64];
  DWORD RecvLength=1;

  buf[0] = UPDATE_LED;
  buf[1] = led;
  buf[2] = (State?1:0);

  if ( CurrentDevice->SendReceivePacket(buf,3,buf1,&RecvLength) == 1)
  {
    if (RecvLength == 1 && buf1[0] == UPDATE_LED)
    {
        return 0;
    }
    else return 2;
  }
  return 1;
}
/******************************************************************************
 * Function:        int DeviceManager::SetTempRealMode()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation
 * Side Effects:    None
 *
 * Overview:        Sets the boards operation to real time temprature display.
 *
 * Note:            None
 *****************************************************************************/
int DeviceManager::SetTempRealMode()
{
  if (CurrentDevice == NULL) return 0;

  BYTE buf[64],buf1[64];
  DWORD RecvLength=1;

  buf[0] = SET_TEMP_REAL;

  if ( CurrentDevice->SendReceivePacket(buf,1,buf1,&RecvLength) == 1)
  {
    if (RecvLength == 1 && buf1[0] == SET_TEMP_REAL)
    {
        return 0;
    }
    else return 2;
  }

  return 1;

}
/******************************************************************************
 * Function:        int DeviceManager::ReadTemperature(UINT &Temperature)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          Temperature - Stores the received temprature values.
 *              Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation
 * Side Effects:    None
 *
 * Overview:        Retrieves the temprature data 
 *
 * Note:            None
 *****************************************************************************/
int DeviceManager::ReadTemperature(UINT &Temperature)
{
  if (CurrentDevice == NULL) return 0;

  BYTE buf[64],buf1[64];
  DWORD RecvLength=3;

  buf[0] = RD_TEMP;

  if ( CurrentDevice->SendReceivePacket(buf,1,buf1,&RecvLength) == 1)
  {
    if (RecvLength == 3 && buf1[0] == RD_TEMP)
    {
        Temperature = buf1[2];
        Temperature = (Temperature<< 0x08) | buf1[1];

        return 0;
    }
    else return 2;
  }
  return 1;


}
/******************************************************************************
 * Function:        int DeviceManager::SetTempLogging()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation
 * Side Effects:    None
 *
 * Overview:        Sets the boards operation to Data logging mode.
 *
 * Note:            None
 *****************************************************************************/

int DeviceManager::SetTempLogging()
{
  if (CurrentDevice == NULL) return 0;

  BYTE buf[64],buf1[64];
  DWORD RecvLength=1;

  buf[0] = SET_TEMP_LOGGING;

  if ( CurrentDevice->SendReceivePacket(buf,1,buf1,&RecvLength) == 1)
  {
    if (RecvLength == 1 && buf1[0] == SET_TEMP_LOGGING)
    {
        return 0;
    }
    else return 2;
  }

  return 1;
}
/******************************************************************************
 * Function:        int DeviceManager::ReadTempLogging(char *Data, int *Length)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          Data - Array of bytes used to store received temprature values.
 *                  Length - Contains the length of the received data.
 *              Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation
 * Side Effects:    None
 *
 * Overview:        Retrieves the temprature data from Temprature Log
 *
 * Note:            None
 *****************************************************************************/
int DeviceManager::ReadTempLogging(char *Data, int *Length)
{
  if (CurrentDevice == NULL) return 0;

  BYTE buf[64],buf1[64];
  DWORD RecvLength=62; // <RD_TEMP><LEN><LO><HIGH>.. upto 30 data points(60 bytes)

  buf[0] = RD_TEMP_LOGGING;
  int retVal = CurrentDevice->SendReceivePacket(buf,1,buf1,&RecvLength);

  if (  retVal== 2 || retVal == 1) // Expecting received data length less than 62
  {
    if (RecvLength > 2 && buf1[0] == RD_TEMP_LOGGING)
    {
        *Length = buf1[1];
        memcpy(Data,&buf1[2],buf1[1]);
        return 0;
    }
    else return 2;
  }
  return 1;

}
/******************************************************************************
 * Function:        int DeviceManager::ReadPot(UINT &PotValue)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          PotValue - Stores the received POT values.
 *              Return Value:
 *                  0 - Success
 *                  1 - Send/Receive Failed
 *                  2 - Data is not as expected
 *                  100 - Aborted Opeation
 * Side Effects:    None
 *
 * Overview:        Retrieves the POT data
 *
 * Note:            None
 *****************************************************************************/
int DeviceManager::ReadPot(UINT &PotValue)
{
  if (CurrentDevice == NULL) return 0;

  BYTE buf[64],buf1[64];
  DWORD RecvLength=3;

  buf[0] = RD_POT;

  if ( CurrentDevice->SendReceivePacket(buf,1,buf1,&RecvLength) == 1)
  {
    if (RecvLength == 3 && buf1[0] == RD_POT)
    {
        PotValue = buf1[2];
        PotValue = (PotValue<< 0x08) | buf1[1];
        return 0;
    }
    else return 2;
  }
  return 1;
}
/******************************************************************************
 * Function:        AnsiString DeviceManager::GetCurrentDeviceUSBDetails()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        None
 *
 * Note:            Not implemented
 *****************************************************************************/

// Get Board Details
AnsiString DeviceManager::GetCurrentDeviceUSBDetails()
{
  if (CurrentDevice == NULL) return 0;
}
/******************************************************************************
 * Function:        void DeviceManager::ToggleLEDD4()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Toggles LED4 on the board 
 *
 * Note:            None
 *****************************************************************************/

// Get Board Details
void DeviceManager::ToggleLEDD4()
{
  if (CurrentDevice == NULL) return ;
  static bool State=true;
  State = !State;
  CurrentDevice->OpenDevice();
  UpdateLED(0x04,State);
  CurrentDevice->CloseDevice();
}
/******************************************************************************
 * Function:        void DeviceManager::IdentifyBoards()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Toggles LED3 and LED4 so that the all boards can be
 *                  identified.
 *
 * Note:            None
 *****************************************************************************/

void DeviceManager::IdentifyBoards()
{
    // Return List of all Devices connected
    if (CurrentDevice != NULL && IsOpen == true )
        CurrentDevice->CloseDevice();

    for (int i = 0; i < DeviceList->Count; i++)
    {
        Device *Obj = (Device *)DeviceList->Items[i];

        if (Obj == CurrentDevice && IsOpen == true ) continue;
        if (Obj->OpenDevice() == 0)
        {
            UpdateLED(3,(Obj->DemoDevID*0x02)>>1);
            UpdateLED(4,(Obj->DemoDevID*0x01));
            Obj->CloseDevice();
        }
    }
    if (CurrentDevice != NULL && IsOpen == true )
    {
        CurrentDevice->OpenDevice();
        IdentifyCurrentDevice();
    }

}

/******************************************************************************
 * Function:        void DeviceManager::IdentifyCurrentDevice()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Toggles LED3 and LED4 so that the active board can be
 *                  identified.
 *
 * Note:            None
 *****************************************************************************/
void DeviceManager::IdentifyCurrentDevice()
{
   BYTE b1 = (CurrentDevice->DemoDevID*0x02)>>1;
   BYTE b2 = (CurrentDevice->DemoDevID*0x01);

   UpdateLED(3,b1);
   UpdateLED(4,b2);
   Sleep(500);
   UpdateLED(3,1);
   UpdateLED(4,1);
   Sleep(500);
   UpdateLED(3,b1);
   UpdateLED(4,b2);
}

//---------------------------------------------------------------------------
#pragma package(smart_init)
