/*********************************************************************
 *
 *                Microchip Full Speed USB Demo Tool Version 1.0
 *
 *********************************************************************
 * FileName:        codefrmPICDEMUSBFSDemoTool.cpp
 * Dependencies:    See INCLUDES section below
 * Processor:       PIC18
 * Compiler:        Borland C++ Builder 6.0
 * Company:         Microchip Technology, Inc.
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the Company) for its PICmicro Microcontroller is intended and
 * supplied to you, the Companys customer, for use solely and
 * exclusively on Microchip PICmicro Microcontroller products. The
 * software is owned by the Company and/or its supplier, and is
 * protected under applicable copyright laws. All rights are reserved.
 * Any use in violation of the foregoing restrictions may subject the
 * user to criminal sanctions under applicable laws, as well as to
 * civil liability for the breach of the terms and conditions of this
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN AS IS CONDITION. NO WARRANTIES,
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Ravi Kumar M        11/04/04     Original.
 * Ravi Kumer M        11/05/04     The code that works on the addresses
 *                                  0x3FFFFE and 0x3FFFFF is commented.
 ********************************************************************/

/* I N C L U D E S **********************************************************/
//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include "codefrmPICDEMUSBFSDemoTool.h"
#include <dbt.h>
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma link "PERFGRAP"
#pragma link "VrBitmapDial"
#pragma link "VrControls"
#pragma link "VrAngularMeter"
#pragma link "VrLabel"
#pragma link "VrLeds"
#pragma link "VrSwitch"
#pragma link "VrCheckLed"
#pragma link "VrScope"
#pragma resource "*.dfm"

/** V A R I A B L E S ********************************************************/
//---------------------------------------------------------------------------
TfrmFullSpeedUSBDemoTool *frmFullSpeedUSBDemoTool;

/** D E C L A R A T I O N S **************************************************/
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::TfrmFullSpeedUSBDemoTool()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Constructor for the application window.
 *
 *
 * Note:            None
 *****************************************************************************/
__fastcall TfrmFullSpeedUSBDemoTool::TfrmFullSpeedUSBDemoTool(TComponent* Owner)
        : TForm(Owner),Busy(false)
{
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::Exit1Click(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Even handler for Exit button of the form
 *
 *
 * Note:            None
 *****************************************************************************/
void __fastcall TfrmFullSpeedUSBDemoTool::Exit1Click(TObject *Sender)
{
        this->Close();
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::rbDataLoggingClick(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Even handler for Radio Button "Data Logging"
 *
 *
 * Note:            None
 *****************************************************************************/
void __fastcall TfrmFullSpeedUSBDemoTool::rbDataLoggingClick(
      TObject *Sender)
{
        cmdAcquireData->Enabled = true;
        edtTemperature->Enabled = false;
        deviceManager.SetTempLogging();
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::rbRealTimeClick(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Even handler for Radio Button "Real Time"
 *
 *
 * Note:            None
 *****************************************************************************/
void __fastcall TfrmFullSpeedUSBDemoTool::rbRealTimeClick(
      TObject *Sender)
{
        cmdAcquireData->Enabled = false;
        edtTemperature->Enabled = true;
        deviceManager.SetTempRealMode();
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmdConnectClick(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Even handler for Radio Button "Connect" in "Demo Mode View"
 *                  Calls required functions to connect to the selected board
 *                  which is in demo mode.
 *                  Enables/ Disables the controls as required by the application.
 *
 * Note:            None
 *****************************************************************************/
void __fastcall TfrmFullSpeedUSBDemoTool::cmdConnectClick(TObject *Sender)
{
        if (cmdConnect->Caption == "&Connect") {
                if(cmbDEMOCommunicationChannel->ItemIndex == -1)
                {
                    cmdConnect->Enabled = false;
                    return;
                }
                else
                    cmdConnect->Enabled = true;

                //ShowStatusMessage("Connecting to board...");
                if (deviceManager.OpenDevice()==false) return;
                //ShowStatusMessage("Identifying board...");
                //deviceManager.IdentifyCurrentDevice();
                ShowStatusMessage("USB Demo Firmware Version " + deviceManager.ReadVersion() );
                cmdConnect->Caption = "&Disconnect";
                cmbDEMOCommunicationChannel->Enabled = false;
                cmbBaudRate->Enabled = false;
                //demoModeTimer->Enabled = true;
                EnableDemoInterface();
                // Enable Real Time Mode
                if (rbRealTime->Checked)  rbRealTimeClick(Sender);
                else rbDataLoggingClick(Sender);
        }
        else
        {
                DisableDemoInterface();
                deviceManager.CloseDevice();
                cmdConnect->Caption = "&Connect";
                if(cmbDEMOCommunicationChannel->ItemIndex == -1)
                    cmdConnect->Enabled = false;
                else
                    cmdConnect->Enabled = true;
                cmbDEMOCommunicationChannel->Enabled = true;
                //if (cmbDEMOCommunicationChannel->Text.SubString(0,3)  == "COM")
                //        cmbBaudRate->Enabled = true;
                // Update Status Bar
                StatusBar->Panels->Items[0]->Text = "Ready";
        }

}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmdLoadHEXClick(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Displays the windows "File Open" dialog box and loads the
 *                  contents of the HEX file selected into the buffers maintained
 *                  by HEXFileHandler class.
 *
 * Note:            None
 *****************************************************************************/

void __fastcall TfrmFullSpeedUSBDemoTool::cmdLoadHEXClick(TObject *Sender)
{
    Busy = true;
    if (OD1->Execute())
    {
        hex.ImportP18HEXFile(OD1->FileName );

        CheckConfiguration(hex.GetConfig());

        if(IsHexEmpty())
            ShowWarning("No HEX file data.");
        else
        {
            ShowHexContent();
        }
    }
    Busy = false;
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmdSaveToHEXClick(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Displays the windows "File Save" dialog box and saves the
 *                  contents of the buffers maintained by HEXFileHandler class
 *                  in Intel HEX file format.
 *
 * Note:            None
 *****************************************************************************/

void __fastcall TfrmFullSpeedUSBDemoTool::cmdSaveToHEXClick(TObject *Sender)
{
        Busy = true;
        if (SD1->Execute())
            hex.ExportP18HEXFile(SD1->FileName);
        Busy = false;
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmbDEMOCommunicationChannelSelect(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Even handler that executes after an item from the drop down list
 *                  is selected. Enables/ Disables appropriate windows controls.
 *
 *
 * Note:            This combo box is populated with the connected demo board details
 *****************************************************************************/

void __fastcall TfrmFullSpeedUSBDemoTool::cmbDEMOCommunicationChannelSelect(
      TObject *Sender)
{
        if (cmbDEMOCommunicationChannel->ItemIndex == -1)
        {
            cmdConnect->Enabled = false;
            return;
        }
        else
            cmdConnect->Enabled = true;
        if (cmdConnect->Caption != "&Connect") {cmdConnectClick(this);}

        // CommunicationChannel == cmbCommunicationChannel->Text
        if (  cmbDEMOCommunicationChannel->Text.SubString(0,3) != "COM")
                {
                        cmbBaudRate->Enabled = false;
//                        cmdGetUSBDetails->Enabled = true;
                        cmbBaudRate->ItemIndex = -1;

                        Busy = true;
                        deviceManager.SelectDevice(cmbDEMOCommunicationChannel->Text);

                        //deviceManager.OpenDevice();
                        //ShowStatusMessage("USB Demo Firmware Version " + deviceManager.ReadVersion() );
                        //deviceManager.CloseDevice();

                        Busy = false;
                }
        else
                {
                        cmbBaudRate->Enabled = true;
                        cmbBaudRate->ItemIndex = 3;
//                        cmdGetUSBDetails->Enabled = false;
                }
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::ResetDemoMode()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function is called when the Demo Mode tab in the
 *                  tab control is selected. It populates the CommunicationChannel
 *                  drop list box and intializes the varibles to indicate that
 *                  the software is in demo mode.
 * Note:            None
 *****************************************************************************/
void TfrmFullSpeedUSBDemoTool::ResetDemoMode()
{
        DisableDemoInterface();
        deviceManager.CloseDevice();

        deviceManager.StopExecution();
        deviceManager.ActiveMode = 1; // Demo Mode
        deviceManager.StartExecution();
        cmbDEMOCommunicationChannelSelect(this);
        StatusBar->Panels->Items[0]->Text = "Ready";
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::ResetBootMode()
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function is called when the Boot Mode tab in the
 *                  tab control is selected. It populates the CommunicationChannel
 *                  drop list box and intializes the varibles to indicate that
 *                  the software is in boot mode.
 * Note:            None
 *****************************************************************************/
void TfrmFullSpeedUSBDemoTool::ResetBootMode()
{
        DisableBootInterface();
        deviceManager.CloseDevice();

        cmbDEMOCommunicationChannel->Clear();
        if (cmdConnect->Caption != "&Connect")
        {
            cmdConnectClick(this);
            cmdConnect->Enabled = false;
        }

        deviceManager.StopExecution();
        deviceManager.ActiveMode = 0;   // Boot Mode
        deviceManager.StartExecution();
        cmbBOOTListDropDown(this);
        StatusBar->Panels->Items[0]->Text = "Ready";
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::TabbedNotebookChange(
 *                        TObject *Sender, int NewTab, bool &AllowChange)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Even handler for Tab control. Calls ResetBootMode() or
 *                  ResetDemoMode() depending on the tab selected.
 *
 *
 * Note:            None
 *****************************************************************************/
void __fastcall TfrmFullSpeedUSBDemoTool::TabbedNotebookChange(
      TObject *Sender, int NewTab, bool &AllowChange)
{
        switch(NewTab)
        {
          case 0: ResetBootMode(); break;
          case 1: ResetDemoMode(); break;
        }
        //ShowStatusMessage("Mode of operation changed.");
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmdClearScreenClick(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Clears the Message Window
 *
 *
 *
 * Note:            None
 *****************************************************************************/


void __fastcall TfrmFullSpeedUSBDemoTool::cmdClearScreenClick(TObject *Sender)
{
    txtMessage->Text = "";    
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::CheckConfiguration(AnsiString& Config)
 *
 * PreCondition:    None
 *
 * Input:           Config - Contains the Config Memory buffer content loaded
 *                  from HEX file.
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        The content of the variable Config is verified to check if
 *                  all the configuration data required to use USB is available
 *                  or not. Based on user input for a Message Box, it can
 *                  set the variable Config to a different set of values
 *                  than those loaded from HEX file.
 *
 * Note:            None
 *****************************************************************************/


void TfrmFullSpeedUSBDemoTool::CheckConfiguration(AnsiString& Config)
{
    // NOTES:
    // 1. There are configuration registers in the addresses 300000 to 30000D
    // So, the config line generated will be a single line with
    // address and 14 or less number of bytes of data.

    // If the HEX file loaded may be having only a partial list of
    // initializations, then if any initialization is different from
    // default and missing in the HEX file, show ERROR.

    // If the HEX file has initialization different from waht is expected
    // again show error

    // Configuration Addresses
    int ConfigAddresses[0x10]={
                              0x300000,0x300001,0x300002,0x300003,
                              0x300004,0x300005,0x300006,0x300007,
                              0x300008,0x300009,0x30000A,0x30000B,
                              0x30000C,0x30000D,0x3FFFFE,0x3FFFFF};
    //Silicon Default Values
/*    int SiliconDefaultValues[0x10] = {0x0,0x0E,0x1F,0x1F,
                                      0x00,0x83,0x85,0x00,
                                      0x0F,0xC0,0x0F,0xE0,
                                      0x0F,0x40,0x00,0x0B};
*/
    int SiliconDefaultValues[0x10] = {0x24,0x0E,0x3F,0x1E,
                                      0x00,0x81,0x81,0x00,
                                      0x0F,0xC0,0x0F,0xA0 /* Boot Block Protect */,
                                      0x0F,0x40,0x00,0x0B};

    int ConfigFileData[0x10] = {0x24,0x0E,0x3F,0x1E,
                                      0x00,0x81,0x81,0x00,
                                      0x0F,0xC0,0x0F,0xA0 /* Boot Block Protect*/,
                                      0x0F,0x40,0x00,0x0B};

    int BaseAddress;
    int Address;
    int Data;
    char *DataPtr = Config.c_str();
    int i=0,Offset,Index;

    // Read the values set in the config buffer if Config is not empty
    // and replace the values in default values buffer with values
    // retrieved from HEX file.
    if (!Config.IsEmpty())
      for (i  = 2; DataPtr[i]!= 0;)
      {
        // Read Base Address
        sscanf(&DataPtr[i],"%06X",&BaseAddress);
        i+=7;
        Offset = 0;

        // Read the Bytes till \0 or \n is found
        while(DataPtr[i]!= 0 && DataPtr[i]!= '\r')
        {
            sscanf(&DataPtr[i],"%X",&Data);
            i=i+3;

            if (BaseAddress + Offset == 0x3FFFFE) Index = 0xE;
            else if (BaseAddress + Offset == 0x3FFFFF) Index = 0xF;
            else {
                // Find the index of the address
                for(Index  = 0; Index  < 0x10; Index++)
                   if (ConfigAddresses[Index] == (BaseAddress+Offset)) break;
            }

            if (Index < 0 || Index > 0xF) {
                ShowWarning("Buffer Corrupted.");
                return;
            }
            // If a valid address is obtained, replace the Silicon Default Value
            // of the address with the value read from Hex file
            ConfigFileData[Index] = Data;
            Offset++;
        }
        if (DataPtr[i]== '\r') i+=2;
      }

    // Used to generate the final Config Values
    // As per user's selection these final set of values will be
    // Loaded on the board.
    bool ConfigError = false;

    for (Index = 0; Index < 0x10; Index++)
      {
            Data = ConfigFileData[Index];

            switch(ConfigAddresses[Index])
            {
                case 0x300000:
                              // 1. USB Clock Selection = clk_src__from_96MHz_PLL/2
                              if ((Data & 0x20) != 0x20)
                              {
                                    ConfigError = true;
                                    ShowWarning("USB Clock Selection is not [ clk_src__from_96MHz_PLL/2] ");
                              }
                              // 2. CPU Sys Clock Select = no divide (=/2 in PLL mode)
                              if ((Data & 0x18) != 0x00)
                              {
                                    ConfigError = true;
                                    ShowWarning("CPU Sys Clock Select is not [ no divide (=/2 in PLL mode) ] ");
                              }
                              // 3. OSC Select  = div by 5 {20 MHz input}
                              if ((Data & 0x07) != 0x04)
                              {
                                    ConfigError = true;
                                    ShowWarning("OSC Select  is not [ div by 5 {20 MHz input} ]");
                              }
//                              SiliconDefaultValues[Index] &= (~0x18);
//                              SiliconDefaultValues[Index] = 0x24;
                              break;
                case 0x300001:
                              // 1. Oscillator = HS: HS+PLL, USB-HS
                              if ((Data & 0x0F) != 0x0E)
                              {
                                    ConfigError = true;
                                    ShowWarning("Oscillator is not [ HS: HS+PLL, USB-HS ]");

                              }
//                              SiliconDefaultValues[Index] = 0x0E;
                              break;
                case 0x300002:
                              // 1. USB Voltage Regulator  =  Enabled
                              if ((Data & 0x20) != 0x20)
                              {
                                    ConfigError = true;
                                    ShowWarning("USB Voltage Regulator  is not [ Enabled ]");
                              }
//                              SiliconDefaultValues[Index] = 0x3F;
                              break;
                case 0x300003:
                              // 1. Watchdog Timer  = Disabled
                              if ((Data & 0x01) == 0x01)
                              {
                                    ConfigError = true;
                                    ShowWarning("Watchdog Timer is not [ Disabled ]");
                              }
//                              SiliconDefaultValues[Index] &= 0xFE;
                              break;

                case 0x300005:
                              // 1. Port B A/D = configured as Digital I/O
                              if ((Data & 0x02) == 0x02)
                              {
                                    ConfigError = true;
                                    ShowWarning(" Port B A/D is not [ configured as Digital I/O ]");
                              }
//                              SiliconDefaultValues[Index] &= 0xFD;
                              break;
                case 0x300006:
                              // 1. Low Voltage Programming = Disabled
                              if ((Data & 0x04) == 0x04)
                              {
                                    ConfigError = true;
                                    ShowWarning(" Low Voltage Programming is not [ Disabled ]");
                              }
//                              SiliconDefaultValues[Index] &= 0xFB;
                              break;
                case 0x30000B:
                            // Table Write Protect for Boot Block Enable
                               if (Data != 0xA0)
                               {
                                    ConfigError = true;
                                    ShowWarning(" Table Write Protect Boot is not [ Enabled ]");
                               }
                               SiliconDefaultValues[Index] &= 0xA0;
                               break;
            } // End of switch
      } // End of for

      if (ConfigError)
      {
            int RetCode;

            RetCode = Application->MessageBoxA(
                "Configuration data contained in this hex file is\r\n"
                "different from the board's default setting.\r\n\n"
                "Using a different setting could cause the bootload\r\n"
                "interface to stop functioning.\r\n\n"
                "Click:\r\n"
                "Yes - To use the setting contained in the hex file.\r\n"
                "No - To use the board's default setting.\r\n"
                "Cancel - To use the board's current setting.\r\n",
                "Configuration Data",
                MB_YESNOCANCEL|MB_TASKMODAL|MB_ICONWARNING|MB_DEFBUTTON3);

          if (RetCode == IDNO)
          {
            // Reset the Config's content to default values.
            Config = "\r\n300000 ";
            for (Index = 0; Index < 0x10-2; Index++)
                      Config = Config + IntToHex(SiliconDefaultValues[Index],2) + " ";
//            Config = Config + "\r\n3FFFFE " +IntToHex(SiliconDefaultValues[Index++],2) + " ";
//            Config = Config + "\r\n3FFFFF "+IntToHex(SiliconDefaultValues[Index],2) + " ";
            ShowWarning("CONFIGURATION MEMORY Changed to : \r\n"+Config+"\r\n");
          }
          else if(RetCode == IDCANCEL)
          {
            Config = "";
          }
      }

}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::ShowWarning(AnsiString WarningText)
 *
 * PreCondition:    None
 *
 * Input:           WarningText - Message String that need to be displayed
 *                  as warning in the message window.
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Prepends the WarningText with the word "WARNING" and displays
 *                  it in the message window
 *
 * Note:            None
 *****************************************************************************/
void TfrmFullSpeedUSBDemoTool::ShowWarning(AnsiString WarningText)
{
      txtMessage->Text = "WARNING - " + WarningText +"\r\n" + txtMessage->Text;
      txtMessage->SetFocus();
      Update();
}

/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::ShowMessage(AnsiString WarningText)
 *
 * PreCondition:    None
 *
 * Input:           WarningText - Message String that need to be displayed
 *                  as message in the message window.
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Prepends the WarningText with the word "MESSAGE" and displays
 *                  it in the message window
 *
 * Note:            None
 *****************************************************************************/
void TfrmFullSpeedUSBDemoTool::ShowMessage(AnsiString WarningText)
{
      txtMessage->Text = "MESSAGE - " + WarningText +"\r\n" + txtMessage->Text;
      txtMessage->SetFocus();
      Update();
}

/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::FormCreate(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Event handler that executes when a form is created.
 *                  Depending on the Tab, appropriate mode data is loaded
 *                  into the application variables.
 *
 * Note:            None
 *****************************************************************************/
void __fastcall TfrmFullSpeedUSBDemoTool::FormCreate(TObject *Sender)
{
    if(TabbedNotebook->ActivePage == "Bootload Mode")
        ResetBootMode();
    else if(TabbedNotebook->ActivePage == "Demo Mode")
        ResetDemoMode();
    else
        ShowMessage("Program flow logic error");
}

/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmbBOOTListSelect(
 *                  TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Event handler for cmbBOOTList. cmbBOOTList displays the
 *                  available boot mode boards and user can select one among
 *                  them at a time. This event occurs just after the selection
 *                  is completed.
 *
 * Note:            None
 *****************************************************************************/

void __fastcall TfrmFullSpeedUSBDemoTool::cmbBOOTListSelect(
      TObject *Sender)
{
    Busy = true;

    EnableBootInterface();

    deviceManager.SelectDevice(cmbBOOTList->Text);

    if (deviceManager.OpenDevice())
    {
        StatusBar->Panels->Items[0]->Text = "USB Bootloader Firmware Version " + \
                                            deviceManager.ReadVersion();
        deviceManager.CloseDevice();
    }
    Busy = false;
}

/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::OnDeviceChange(TWMDeviceChange& Message)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Event handler for Device Change(WM_DEVICECHANGE) activity.
 *                  Windows identifies whenever a plug and play device is
 *                  connected or removed and generates the WM_DEVICECHANGE event.
 *
 *
 * Note:            None
 *****************************************************************************/

void __fastcall TfrmFullSpeedUSBDemoTool::OnDeviceChange(TWMDeviceChange& Message)
{
    Busy = true;
    if(deviceManager.ActiveMode == 0)
    {
        if(cmbBOOTList->ItemIndex != -1)
        {
            if(deviceManager.OpenDevice() == false)
            {
                DisableBootInterface();
                ChangeAbortStatus(SAVE_CONTEXT);
                ChangeAbortStatus(RESTORE_CONTEXT);
                cmbBOOTList->Clear();
                StatusBar->Panels->Items[0]->Text = "Ready";
                //cannot call ResetBootMode here.. causes Protection Fault..
            }
            else
                deviceManager.CloseDevice();
        }//end if
    }//end if

    if(deviceManager.ActiveMode == 1)
    {
        if(cmdConnect->Caption == "&Disconnect")
        {
            cmbDEMOCommunicationChannel->Clear();
            cmdConnectClick(this);
        }
    }//end if

    Busy = false;

#if 0
    char msg[100];
    Busy = true;
    if(Message.wParam == DBT_DEVICEARRIVAL ||
       Message.wParam == DBT_DEVICEREMOVECOMPLETE ||
       Message.wParam == DBT_DEVNODES_CHANGED)
         sprintf(msg,"NOTE: PICDEM Boards Status Changed.\r\n");

    //txtMessage->Text = AnsiString(msg) + txtMessage->Text;
    Message.Result = true;
    deviceManager.Refresh();

    //big question..
    // if i have a device open, was it detached?
    //try getting version
    if(Message.wParam == DBT_DEVICEREMOVECOMPLETE)
        ShowMessage("remove...");
    else if(Message.wParam == DBT_DEVICEARRIVAL)
        ShowMessage("arr..");
    else if(Message.wParam == DBT_DEVNODES_CHANGED)
        ShowMessage("dev..");
    else
        ShowMessage("what");

    deviceManager.OpenDevice();
    if(deviceManager.ReadVersion() == "Failed to obtain Version")
    {
        if(deviceManager.ActiveMode == 0)
        {
            cmbBOOTList->Items->Clear();
        }//end if
        if(deviceManager.ActiveMode == 1)
        {
            cmbDEMOCommunicationChannel->Items->Clear();
        }//end if
        
    }//end if else

    deviceManager.CloseDevice();
    Busy = false;
#endif
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmdReadDeviceClick(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Event handler for cmdReadDevice button. Invokes the required
 *                  code to read memory from the connected device and move the
 *                  memory content to various data buffers.
 *
 *
 * Note:            None
 *****************************************************************************/


void __fastcall TfrmFullSpeedUSBDemoTool::cmdReadDeviceClick(TObject *Sender)
{
    static BYTE buff[0x8000];
    int i;
    Busy = true;

    cmdSaveToHEX->Enabled = false;
    cmdProgramDevice->Enabled = false;

    ChangeAbortStatus(SAVE_CONTEXT);
    this->ShowMessage("Reading Device...");
    this->Update();
    deviceManager.OpenDevice();

    if (deviceManager.ReadFLASH(0x8000,0,buff) != 0) goto end_program;
    FormatMemoryDump(hex.GetProgramMemory(),0,buff,0x8000);

    if (deviceManager.ReadEEDATA(256,0,buff) != 0) goto end_program;
    FormatMemoryDump(hex.GetEEMemory(),0,buff,256);

    //Read User ID
    for (i = 0; i < 0x08; i ++)
    {
        if(deviceManager.ReadCONFIG(0x200000+i,&buff[i]) != 0) goto end_program;
    }
    FormatConfigMemoryDump(hex.GetUID(),0x200000,buff);

    //Read Configuration Data
    for (i = 0; i < 0x10-2; i ++)
    {
       if( deviceManager.ReadCONFIG(0x300000+i,&buff[i]) != 0) goto end_program;

    }
    deviceManager.ReadCONFIG(0x3FFFFE,&buff[i++]);
    deviceManager.ReadCONFIG(0x3FFFFF,&buff[i]);
    FormatConfigMemoryDump(hex.GetConfig(),0x300000,buff);

    if(!IsHexEmpty())
    {
        ShowHexContent();
        ShowMessage("Read Completed");
        ChangeAbortStatus(RESTORE_CONTEXT);
        cmdSaveToHEX->Enabled = true;
        cmdProgramDevice->Enabled = true;

        goto end_program2;
    }//end if
end_program:
    ShowWarning("Failed to read");
    ChangeAbortStatus(RESTORE_CONTEXT);
    hex.ClearAllMemoryBuff();
    IsHexEmpty();               //Update cmdProgramDevice & cmdSaveToHex
end_program2:
    deviceManager.CloseDevice();
    Busy = false;
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::ShowStatusMessage(AnsiString Message)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          Message
 *
 * Side Effects:    None
 *
 * Overview:        Displays Message in the Status Bar Panel 0
 *
 *
 * Note:            None
 *****************************************************************************/
void TfrmFullSpeedUSBDemoTool::ShowStatusMessage(AnsiString Message)
{
     StatusBar->Panels->Items[0]->Text = Message;
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::FormatMemoryDump(AnsiString &FormatBuffer,
 *                    DWORD StartAddress, BYTE *ByteBuffer, DWORD Length)
 *
 * PreCondition:    None
 *
 * Input:           ByteBuffer - contains a series of bytes
 *                  Length - Length of ByteBuffer
 *                  StartAddress - StartAddres to be used for formatting
 *                  FormatBuffer - Referense to AnsiString object that holds
 *                  all the formatted data.
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Formats the ByteBuffer into application specific format.
 *
 *
 * Note:
 *                  The format is as following.
 *                  <\r\n>
 *                  <<6-digit HEX Address><Space><2-digit HEX Data1>...<Space><2-digit HEX Data16><\r\n>
 *****************************************************************************/
void TfrmFullSpeedUSBDemoTool::FormatMemoryDump(AnsiString &FormatBuffer, DWORD StartAddress, BYTE *ByteBuffer, DWORD Length)
{
     DWORD Offset = StartAddress%16;
     DWORD Address = StartAddress-Offset;

     FormatBuffer = "";
     for (DWORD i = 0; i < (Length+Offset); i++)
     {
     if ( (Address+i) % 0x10 == 0)
            FormatBuffer.cat_printf("\r\n%06X ",Address+i);
     if (Address+i < StartAddress)
        FormatBuffer.cat_printf("%02X ",0xFF);
     else
        FormatBuffer.cat_printf("%02X ",ByteBuffer[i-Address]);
     }
}

/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::FormatMemoryDump(AnsiString &FormatBuffer,
 *                    DWORD StartAddress, BYTE *ByteBuffer, DWORD Length)
 *
 * PreCondition:    None
 *
 * Input:           ByteBuffer - contains a series of bytes
 *                  Length - Length of ByteBuffer
 *                  StartAddress - StartAddres to be used for formatting
 *                  FormatBuffer - Referense to AnsiString object that holds
 *                  all the formatted data.
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Formats the ByteBuffer to application specific data.
 *                  In case of the configuration data the addresses 0x3FFFFE and
 *                  0x3FFFFF needs special format.
 * Note:
 *                  The format is as following.
 *                  <\r\n>
 *                  <<6-digit HEX Address><Space><2-digit HEX Data[1]>...<Space><2-digit HEX Data[n]><\r\n>
 *****************************************************************************/
void TfrmFullSpeedUSBDemoTool::FormatConfigMemoryDump(AnsiString &FormatBuffer, DWORD StartAddress, BYTE *ByteBuffer)
{
    int Index;

    if(StartAddress == 0x300000)
    {

        FormatBuffer = "\r\n300000 ";
        for (Index = 0; Index < 0x10-2; Index++)
            FormatBuffer.cat_printf("%02X ",ByteBuffer[Index]);
        FormatBuffer.cat_printf("\r\n3FFFFE %02X ",ByteBuffer[Index++]);
        FormatBuffer.cat_printf("\r\n3FFFFF %02X ",ByteBuffer[Index]);
    }

    if(StartAddress == 0x200000)
    {
        FormatBuffer = "\r\n200000 ";
        for(Index = 0; Index < 0x08; Index++)
            FormatBuffer.cat_printf("%02X ",ByteBuffer[Index]);
    }
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmdEraseDeviceClick(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Event handler for cmdEraseDevice. When this button is selected
 *                  the  board's FLASH memory will be erased.
 *
 * Note:            None
 *****************************************************************************/

void __fastcall TfrmFullSpeedUSBDemoTool::cmdEraseDeviceClick(TObject *Sender)
{
   Busy = true;
   BYTE buff[0x8000];

   ChangeAbortStatus(SAVE_CONTEXT);
   deviceManager.OpenDevice();
   this->ShowMessage("Erasing Flash starting from 0x800 to 0x7FFF...");
   this->Update();
   if (deviceManager.EraseFLASH( 0x8000-0x800,0x800) == 0)
   {
       ShowMessage("Verifying Memory starting from 0x800 to 0x7FFF..."); 
       if (deviceManager.ReadFLASH(0x8000-0x800,0x800,buff) == 0)
       {
            int i;
            for (i = 0; i < (0x8000-0x800); i++)
            {
                if (buff[i] != 0xFF)
                {
                    ShowWarning("Expected Value at Address 0x"+IntToHex(i,6)+ " : 0xFF");
                    ShowWarning("Actual Value :" + IntToHex(buff[i],2));
                    ShowWarning("Failed to Verify.");
                    break;
                }
            }
            if (i == (0x8000-0x800))
            {
                this->ShowMessage("Verify Completed.");
                this->ShowMessage("Erase Completed.");
            }
       }
       else ShowWarning("Failed to read for verification.");

   }
   else
       this->ShowWarning("Failed to erase.");

   deviceManager.CloseDevice();
   ChangeAbortStatus(RESTORE_CONTEXT);
   Busy = false;
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmdProgramDeviceClick(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Event handler for cmdProgramDevice. When this button is selected
 *                  the  board's FLASH memory will be erased, and porogrammed
 *                  with the buffer contents. 
 *
 * Note:            None
 *****************************************************************************/

void __fastcall TfrmFullSpeedUSBDemoTool::cmdProgramDeviceClick(TObject *Sender)
{

    int RetVal;
    // Program FLASH
    Busy = true;

    ChangeAbortStatus(SAVE_CONTEXT);
    deviceManager.OpenDevice();

    if ( !hex.GetProgramMemory().IsEmpty() )
    {
        ShowMessage("Erasing and Programming FLASH...");
        //ShowMessage("Erasing FLASH...");
        RetVal = deviceManager.WriteFLASH(hex.GetProgramMemory());
        if ( RetVal== 100) goto end_program;
        if (RetVal == 0)
             ShowMessage("Programming FLASH Completed");
        else ShowWarning("Failed to program FLASH");
    }

    // Program EEPROM
    if ( !hex.GetEEMemory().IsEmpty() )
    {
        this->ShowMessage("Programming EEPROM...");
        RetVal = deviceManager.WriteEEDATA(hex.GetEEMemory());
        if ( RetVal== 100) goto end_program;
        if (RetVal == 0)
             ShowMessage("Programming EEPROM Completed");
        else ShowWarning("Failed to program EEPROM");
    }

    // Program USER ID Data
    if ( !hex.GetUID().IsEmpty())
    {
        this->ShowMessage("Erasing and Programming USER ID...");
        RetVal = deviceManager.WriteUID(hex.GetUID());
        if ( RetVal== 100) goto end_program;
        if (RetVal == 0)
             ShowMessage("Programming USER ID Completed");
        else ShowWarning("Failed to program USER ID");
    }

    // Program CONFIG Data
    if ( !hex.GetConfig().IsEmpty()  )
    {
        this->ShowMessage("Programming CONFIG DATA...");
        RetVal = deviceManager.WriteCONFIG(hex.GetConfig());
        if ( RetVal== 100) goto end_program;
        if (RetVal == 0)
             ShowMessage("Programming CONFIG DATA Completed");
        else ShowWarning("Failed to program CONFIG DATA");

    }
end_program:
    deviceManager.CloseDevice();
    ChangeAbortStatus(RESTORE_CONTEXT);
    Busy = false;

}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmdExecuteClick(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Event handler for cmdExecute. When this button is selected
 *                  the RESET command is sent to the board. So, it starts
 *                  execution in demo mode. If S2 on board is pressed the board
 *                  will be initialized to boot mode.
 *
 * Note:            None
 *****************************************************************************/

void __fastcall TfrmFullSpeedUSBDemoTool::cmdExecuteClick(TObject *Sender)
{
    if (deviceManager.OpenDevice())
    {
        if(deviceManager.ResetBoard() != 0)
            ShowWarning("Failed to RESET board");
        deviceManager.CloseDevice();
    }
    ResetBootMode();
}

/******************************************************************************
 * Function:        AnsiString TfrmFullSpeedUSBDemoTool::ProcessTemperature(UINT Temp)
 *
 * PreCondition:    None
 *
 * Input:           Temp - 2 byte A2D format
 *
 * Output:          AnsiString containing the celsius format.
 *
 * Side Effects:    None
 *
 * Overview:        The temprature value from A2D converter is received and
 *                  converted to celsius. Format : +/-XXX.XX
 *
 *
 * Note:            None
 *****************************************************************************/
AnsiString TfrmFullSpeedUSBDemoTool::ProcessTemperature(UINT Temp)
{
    BYTE HBYTE;
    BYTE LBYTE;

    WORD temp;
    char tempString[10];

    temp = (Temp >> 3);

    if ( (temp & 0x1000) == 0x0)
    {
        temp &= 0x1FFF; //0b00011111
        tempString[0] = '+';
    }
    else
    {
        temp |= 0xE000; //0b11100000
        tempString[0] = '-';
        temp = (temp ^ 0xFFFF); // Negate
        temp++;
    }
    temp = (temp*10U) >> 4; 
    // Show in the Graph
    // The VCL Component needs a correction
    // of a offset value of 2

    temp+=2;

    TempScope->SetData(0, temp);

    // Show in the Text Box
    for (int i = 4; i > 0; i--)
    {
        tempString[i] = (((char)(temp % 10)) & 0x0F) | 0x30;
        temp = temp/10;
    }
    tempString[5] = tempString[4];
    tempString[4]='.';
    tempString[6]=176; // Degree symbol
    tempString[7]='C';
    tempString[8]=0x00;     // Null-Terminated

    edtTemperature->Text =tempString;

    return (edtTemperature->Text);


}
/******************************************************************************
 * Function:        void TfrmFullSpeedUSBDemoTool::ProcessPOT(UINT POT)
 *
 * PreCondition:    None
 *
 * Input:           POT - 2 byte A2D format
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        The Potentiometer value is received and
 *                  converted to actual resistance value. Format : XXXXX.
 *                  And it will be displayed on edtPOT and updated on the
 *                  graphical display POTAngMeter.
 *
 * Note:            None
 *****************************************************************************/
void TfrmFullSpeedUSBDemoTool::ProcessPOT(UINT POT)
{
    double Value = (POT/1023.0)*10000;
    edtPOT->Text = (int)Value;
    POTAngMeter->Position = Value/1000;

}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::demoModeTimerTimer(
 *                  TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Event Handler for demoModeTimer, this event occurs every
 *                  250 ms. The code in the event handler updates the demo mode
 *                  display. If Temprature data logging mode is active, then
 *                  there won't be any update in the temprature display.
 *
 * Note:            None
 *****************************************************************************/

void __fastcall TfrmFullSpeedUSBDemoTool::demoModeTimerTimer(
      TObject *Sender)
{
    UINT Temperature;
    UINT POT;

    demoModeTimer->Enabled = false;

    // Read the temperature if Real Time Mode is enabled.
    if (rbRealTime->Checked)
    {
         if(deviceManager.ReadTemperature(Temperature) == 0)
            ProcessTemperature(Temperature);
         else
         {
            cmdConnectClick(Sender);
            return;
         }
    }

    // Read the POT Value
    if(deviceManager.ReadPot(POT) == 0)
        ProcessPOT(POT);
    else
    {
        cmdConnectClick(Sender);
        return;
    }

    // Set the LED Status
    if(deviceManager.UpdateLED(3,VrLED3->Checked) != 0)
    {
        cmdConnectClick(Sender);
        return;
    }
    if(deviceManager.UpdateLED(4,VrLED4->Checked) != 0)
    {
        cmdConnectClick(Sender);
        return;
    }

    demoModeTimer->Enabled = true;
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmdAcquireDataClick(
 *                  TObject *Sender
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Event Handler for cmdAcquireData. This button is enabled
 *                  only when Data Logging radio button is selected.
 *                  It retrievs all the logged data at that point of time
 *                  and displays it in the Message Window. 
 *
 * Note:            None
 *****************************************************************************/

void __fastcall TfrmFullSpeedUSBDemoTool::cmdAcquireDataClick(
      TObject *Sender)
{
    BYTE Data[64];
    int Length;

    cmdAcquireData->Enabled = false;
    demoModeTimer->Enabled = false;

    //Sleep(350);
    // Read the temperature logging
    int RetVal = deviceManager.ReadTempLogging(Data,&Length);
    if (RetVal == 2)
    {
       // ShowWarning("Length of data received is not sufficient for calculations.");
        ShowWarning("No data acquired.");
    }
    else if (RetVal == 1)
    {
        ShowWarning("Failed to acquire data.");
    }
    else if (RetVal == 0)
    {
      for (int i = 0; i < Length; i+=2)
      {
        UINT Temp = (Data[i+1] << 0x8) + Data[i];
        ShowMessage("Temperature Point "+IntToStr(i/2)+" : " +ProcessTemperature(Temp));
      }
    }
    // Display in the output Window
    // Update the graph if if Needed.

    demoModeTimer->Enabled = true;
    cmdAcquireData->Enabled = true;
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmdIdentifyBoardsClick(
 *                  TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        None
 *
 *
 *
 *
 * Note:            Not Implemented
 *****************************************************************************/




void __fastcall TfrmFullSpeedUSBDemoTool::cmdIdentifyBoardsClick(
      TObject *Sender)
{
        //deviceManager.IdentifyBoards();
}

/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmdAbortOperationClick(
 *                  TObject *Sender)
 *
 * PreCondition:    The cmdAbortOperation button will be active only
 *                  when the Program Device, Read Device, Erase Device
 *                  are selected.
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Even handler for cmdAbortOperation. Sets the AbortOperation
 *                  variable of deviceManager Object to true. The Program Device,
 *                  Read Device, and Erase Device operations continuously check
 *                  to see if this variable is set to true. If this variable is
 *                  set to true, then the operation will be stopped.
 *
 * Note:            None
 *****************************************************************************/

void __fastcall TfrmFullSpeedUSBDemoTool::cmdAbortOperationClick(
      TObject *Sender)
{
    //ShowMessage("Aborting Operation...");
    deviceManager.AbortOperation = true;
    ShowMessage("*** Operation Aborted ***");

}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::Image1Click(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This event handler of Image1 will take to open the web site
 *                  www.microchip.com in a internet browser.
 *
 *
 * Note:            None
 *****************************************************************************/

void __fastcall TfrmFullSpeedUSBDemoTool::Image1Click(TObject *Sender)
{
    ShellExecute(0, NULL, static_cast<const char*>("http://www.microchip.com"),\
                NULL, NULL, SW_NORMAL);
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmbBOOTListDropDown(TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Even handler for DropDown event of cmbBOOTList.
 *                  Used to refresh the drop down list box.
 *
 *
 * Note:            None
 *****************************************************************************/

void __fastcall TfrmFullSpeedUSBDemoTool::cmbBOOTListDropDown(
      TObject *Sender)
{
    deviceManager.Refresh();
    cmbBOOTList->Items->Clear();
    cmbBOOTList->Items->SetText(deviceManager.GetDeviceList().c_str());
    if(cmbBOOTList->Items->Text == "")
        DisableBootInterface();
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::cmbDEMOCommunicationChannelDropDown
 *                  (TObject *Sender)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Even handler for DropDown event of cmbDEMOCommunicationChannel.
 *                  Used to refresh the drop down list box.
 *
 *
 * Note:            None
 *****************************************************************************/

void __fastcall TfrmFullSpeedUSBDemoTool::cmbDEMOCommunicationChannelDropDown(
      TObject *Sender)
{
    deviceManager.Refresh();
    cmbDEMOCommunicationChannel->Items->Clear();
    cmbDEMOCommunicationChannel->Items->SetText(deviceManager.GetDeviceList().c_str());
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::DisableBootInterface()
 *
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        All buttons in boot mode screen, except "Load HEX File"
 *                  will be disabled by this function.
 *
 * Note:            None
 *****************************************************************************/

void TfrmFullSpeedUSBDemoTool::DisableBootInterface()
{
    cmdProgramDevice->Enabled = false;
    cmdAbortOperation->Enabled = false;
    cmdReadDevice->Enabled = false;
    cmdExecute->Enabled = false;
    cmdEraseDevice->Enabled = false;
    cmdLoadHEX->Enabled = true;
}

/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::EnableBootInterface()
 *
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Enables the buttons boot mode interface depending
 *                  on the validity of HEX buffers.
 *
 * Note:            None
 *****************************************************************************/

void TfrmFullSpeedUSBDemoTool::EnableBootInterface()
{
    IsHexEmpty();                   // Update cmdProgramDevice & cmdSaveToHEX
    cmdReadDevice->Enabled = true;
    cmdExecute->Enabled = true;
    cmdEraseDevice->Enabled = true;
    cmdAbortOperation->Enabled = false;

    //cmdIdentifyBoards->Enabled = true;
}

/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::DisableDemoInterface()
 *
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Disables the controls in demo mode interface.
 *
 *
 * Note:            None
 *****************************************************************************/

void TfrmFullSpeedUSBDemoTool::DisableDemoInterface()
{
    rbRealTime->Enabled = false;
    rbDataLogging->Enabled = false;
    VrLED3->Enabled = false;
    VrLED4->Enabled = false;
    cmdAcquireData->Enabled = false;
    edtTemperature->Enabled = false;
    edtPOT->Enabled = false;
    demoModeTimer->Enabled = false;

    edtTemperature->Text = "";
    edtPOT->Text = "";
}

/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::EnableDemoInterface()
 *
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Enables the controls in demo mode interface.
 *
 *
 * Note:            None
 *****************************************************************************/

void TfrmFullSpeedUSBDemoTool::EnableDemoInterface()
{
    rbRealTime->Enabled = true;
    rbDataLogging->Enabled = true;
    VrLED3->Enabled = true;
    VrLED4->Enabled = true;
    if(rbRealTime->Checked == true)
    {
        cmdAcquireData->Enabled = false;
        edtTemperature->Enabled = true;
        demoModeTimer->Enabled = true;
    }//end if

    if(rbDataLogging->Checked == true)
    {
        cmdAcquireData->Enabled = true;
        edtTemperature->Enabled = false;
        demoModeTimer->Enabled = false;
    }//end if
    edtPOT->Enabled = true;
}

/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::IsHexEmpty()
 *
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          TRUE- if buffers are empty
 *
 * Side Effects:    None
 *
 * Overview:        Use this routine to check if the memory buffers are empty.
 *
 *
 * Note:            None
 *****************************************************************************/

BOOL TfrmFullSpeedUSBDemoTool::IsHexEmpty()
{
    BOOL toReturn = true;
    if((!hex.GetProgramMemory().IsEmpty())||
        (!hex.GetEEMemory().IsEmpty())||
        (!hex.GetConfig().IsEmpty())||
        (!hex.GetUID().IsEmpty()))
    {
        cmdSaveToHEX->Enabled = true;
        if(cmbBOOTList->ItemIndex != -1)
        {
            cmdProgramDevice->Enabled = true;
        }
        else
        {
            cmdProgramDevice->Enabled = false;
        }
        toReturn = false;
    }
    else
    {
        cmdProgramDevice->Enabled = false;
        cmdSaveToHEX->Enabled = false;
    }

    return toReturn;
}
/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::ShowHexContent()
 *
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Displays the memory buffers in Message Window.
 *
 *
 * Note:            None
 *****************************************************************************/
void TfrmFullSpeedUSBDemoTool::ShowHexContent()
{

        txtMessage->Clear();

        txtMessage->Text = "\r\nAddr.  00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F\r\n";

        UINT Len = txtMessage->Text.Length();
        Len = Len + (28*4) +
                    hex.GetProgramMemory().Length() +
                    hex.GetEEMemory().Length() +
                    hex.GetUID().Length() +
                    hex.GetConfig().Length();
        char *buf = new char[Len];

        if (buf == NULL) {
                ShowWarning("Failed to allocate memory for display.");
                return;
        }
        strcpy(buf,"\r\n\r\nPROGRAM MEMORY:\r\n");
        strcat(buf,hex.GetProgramMemory().c_str());
        strcat(buf,"\r\n\r\nEEPROM MEMORY:\r\n");
        strcat(buf,hex.GetEEMemory().c_str());
        strcat(buf,"\r\n\r\nUSER ID MEMORY:\r\n");
        strcat(buf,hex.GetUID().c_str());
        strcat(buf,"\r\n\r\nCONFIGURATION MEMORY:\r\n");
        strcat(buf,hex.GetConfig().c_str() );
        txtMessage->Text = txtMessage->Text + buf;

        delete []buf;
/*
        txtMessage->Text = (txtMessage->Text) + "\r\n\r\nPROGRAM MEMORY:\r\n";
        txtMessage->Text = (txtMessage->Text) + hex.GetProgramMemory();
        txtMessage->Text = (txtMessage->Text) + "\r\n\r\nEEPROM MEMORY:\r\n";
        txtMessage->Text = (txtMessage->Text) + hex.GetEEMemory();
        txtMessage->Text = (txtMessage->Text) + "\r\n\r\nUSER ID MEMORY:\r\n";
        txtMessage->Text = (txtMessage->Text) + hex.GetUID();
        txtMessage->Text = (txtMessage->Text) + "\r\n\r\nCONFIGURATION MEMORY:\r\n";
        txtMessage->Text = (txtMessage->Text) + hex.GetConfig();
*/

}

/******************************************************************************
 * Function:        TfrmFullSpeedUSBDemoTool::ChangeAbortStatus(int update)
 *
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Updates the control's state from enabled to disabled,
 *                  or from disabled to enabled based on the value of
 *                  the argument update.
 *
 * Note:            None
 *****************************************************************************/

void TfrmFullSpeedUSBDemoTool::ChangeAbortStatus(int update)
{
    static BOOL LoadHEXStat = true;
    static BOOL SaveToHEXStat = false;
    static BOOL ProgramDeviceStat = false;
    static BOOL ExecuteStat = false;
    static BOOL ReadDeviceStat = false;
    static BOOL EraseDeviceStat = false;

    // if update == 1, enable abort and save cmdContext
    // if update == 0, disable abort and restore cmd Context
    if(update == SAVE_CONTEXT)
    {
        LoadHEXStat = cmdLoadHEX->Enabled;
        SaveToHEXStat = cmdSaveToHEX->Enabled;
        ProgramDeviceStat = cmdProgramDevice->Enabled;
        ExecuteStat = cmdExecute->Enabled;
        ReadDeviceStat = cmdReadDevice->Enabled;
        EraseDeviceStat = cmdEraseDevice->Enabled;

        cmdLoadHEX->Enabled = false;
        cmdSaveToHEX->Enabled = false;
        cmdProgramDevice->Enabled = false;
        cmdExecute->Enabled = false;
        cmdReadDevice->Enabled = false;
        cmdEraseDevice->Enabled = false;

        cmdAbortOperation->Enabled = true;
    }
    else
    {
        cmdLoadHEX->Enabled = LoadHEXStat;
        cmdSaveToHEX->Enabled = SaveToHEXStat;
        cmdProgramDevice->Enabled = ProgramDeviceStat;
        cmdExecute->Enabled = ExecuteStat;
        cmdReadDevice->Enabled = ReadDeviceStat;
        cmdEraseDevice->Enabled = EraseDeviceStat;

        cmdAbortOperation->Enabled = false;
    }//end if
}





//---------------------------------------------------------------------------

void __fastcall TfrmFullSpeedUSBDemoTool::FormResize(TObject *Sender)
{
//    if (this->Width > 567)
//        this->Width = 567;

}
//---------------------------------------------------------------------------

